/* linear regression model with informative priors */

data {
  int<lower=1> K;   # number of covariates
  int<lower=1> N;   # number of observations
  matrix[N,K] X;    # predictor matrix
  real y[N];        # outcome variable  
  
  // stuff for informative priors in regression
  vector[K] beta_0;
  cov_matrix[K] A_0;
  real<lower=0> v_0; /* or could be an int */
  real<lower=0> s2_0;
}
transformed data {
  matrix[K,K] L;
  L <- inverse_spd(crossprod(X) + A_0);
  L <- 0.5 * L + 0.5 * transpose(L); // make numerically symmetric again
  L <- cholesky_decompose(L);
  print("K = ", K);
  print("N = ", N);
}
parameters {
  vector[K] z;          # unrestricted
  real<lower=0> sigma2; # restricted to be >= 0
}
transformed parameters {
  // Multivariate Matt trick
  vector[K] beta;
  beta <- beta_0 + L * z; 
}
model {
  y ~ normal(X * beta, sqrt(sigma2)); # log-likelihood
  # priors
  z ~ normal(0.0, 1.0);   // implies beta ~ multi_normal(beta_0, L * L')
  sigma2 ~ inv_gamma(v_0, s2_0);
}
generated quantities {
  vector[N] y_tilde;
  for (n in 1:N) {
    y_tilde[n] <- normal_rng(row(X,n) * beta, sqrt(sigma2));
  }
}
