# This file is part of RStan
# Copyright (C) 2015 Jiqiang Guo and Benjamin Goodrich
#
# RStan is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# RStan is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

#' A ReferenceClass to represent a Stan program \emph{without} data
#' 
#' To create an object of this class, call \code{StanProgram} specifying either
#' 
#' \itemize{
#' \item file. an absolute or relative path to a .stan file
#' \item code. a character vector defining a program in the Stan language 
#' }
#' 
#' @field stan_code A derived character vector of any length with Stan code
#' @field cpp_code A derived character vector of any length containing C++ code
#' @field dso A derived object of \code{\link{cxxdso-class}}
#' 
#' @seealso \code{\link{StanProgramWithData-class}}
#' @examples
#' \donttest{
#' example("rstan-package")          # creates the mc object 
#' program <- StanProgram(code = mc) # or better file = path
#' program                           # implicit program$show()
#' length(program$cpp_code)          # number of lines in the C++ code
#' program$save()                    # saved in tempdir() in this case
#' program$help("expose")            # can omit the quotes
#' program$expose()                  # warning about no user-defined functions
#' # see help("StanProgramWithData-class") for what to do next
#' }
#' @export StanProgram
StanProgram <- 
  setRefClass("StanProgram", 
              fields = list(stan_code = "character", cpp_code = "character", 
                            dso = "cxxdso"), 
              methods = list(initialize = function(file, code, auto_write, ...) {},
                             show = function() {},
                             cpp_code = function() {},
                             expose = function() {},
                             instantiate = function(data = list()) {},
                             save = function(file) {},
                             identical = function(program) {}
                             help = function(topic) {}))
StanProgram$lock(names(StanProgram$fields()))

#' A ReferenceClass representing a Stan program \emph{with} data
#' 
#' To create an object of this class, call the \code{$initialize} method
#' of the \code{\link{StanProgram-class}}, which can be passed a named list
#' or an environment containing objects declared in the data block of a
#' Stan program. The calling environment is also searched for such objects.
#' 
#' @seealso \code{\link{StanProgram-class}}
#' @examples
#' \donttest{
#' example(StanProgram-class)           # create J, y, sigma, and program
#' dprogram <- program$initialize()     # J, y, and sigma are found
#' # add illustrations of $method useage
#' }
#' @export StanProgramWithData
StanProgramWithData <-
  # Do we want a field for a named list of parameter dimensions?
  setRefClass("StanProgramWithData", methods = list(
    help = function(topic) {},
    # everything below will basically be provided by the C++ library
    log_prob = function(uparams) {},
    grad = function(uparams) {},
    log_prob_grad = function(uparams) {},
    hessian = function(uparams) {},
    log_prob_grad_hessian = function(uparams) {},
    laplace_approx = function(uparams) {},
    constrain_params = function(uparams) {},
    unconstrain_params = function(params) {},
    lbfgs = function() {},
    bfgs = function() {},
    newton = function() {},
    optimize = function() {}, # no arguments allowed
    sample = function() {}    # no arguments allowed
  )
)

#' An unexported S4 class tree defining containers for MCMC output from Stan
#' 
#' These classes are not intended to be generated by users. Objects from these
#' classes are contained in an object of \code{\link{StanFitMCMC-class}} and
#' can be created by performing operations on existing \code{StanParameter}s.
#' There are associated S4 methods that can be called by users. 
#' 
#' The \code{StanParameter-class} is virtual and the inheritance logic is as 
#' follows:
#' \itemize{
#'   \item StanReal: inherits from \code{StanParameter-class} and represents 
#'     real scalars
#'   \itemize{
#'     \item StanInteger: inherits from \code{StanReal-class} and represents
#'       integer scalars
#'     \itemize{
#'       \item StanFactor: inherits from \code{StanInteger-class} and
#'         represents categorical scalars; hence it has additional slots for
#'         'levels' and 'ordered' that are passed to \code{\link{factor}}
#'      }
#'   }
#'   \item StanMatrix: inherits from  \code{StanParameter-class} and represents
#'     real matrices
#'   \itemize{
#'     \item StanCovMatrix: inherits from \code{StanMatrix-class} and
#'       represents covariance matrices
#'      \itemize{
#'        \item StanCorrMatrix: inherits from \code{StanCovMatrix-class} and
#'          represents correlation matrices
#'      }
#'      \item StanCholeskyFactorCov: inherits from \code{StanMatrix-class} and
#'         represents a Cholesky factor of a covariance matrix
#'      \itemize{
#'         \item StanCholeskyFactorCorr: inherits from 
#'           \code{StanCholeskyFactorCov-class} and represents a Cholesky
#'           factor of a correlation matrix
#'      }
#'      \item StanVector: inherits from \code{StanMatrix-class} and represents
#'         a vector
#'      \itemize{
#'        \item StanRowVector: inherits from \code{StanVector-class} and
#'          represents a row vector
#'        \item StanSimplex: inherits from \code{StanVector-class} and
#'          represents a simplex
#'        \item StanUnit: inherits from \code{StanVector-class} and represents
#'          a unit vector on a hypersphere
#'      }
#'    }
#' }
#' 
#' @slot name A length-one character vector giving the name of the parameter
#'   in the Stan program
#' @slot theta A numeric array of at least three dimensions providing the MCMC
#'   output from the Stan program, of which the last two dimensions are named
#'   'chains' and 'iterations'
#' @slot type A length-one character vector among
#'   \itemize{
#'     \item unconstrained parameter
#'     \item constrained parameter
#'     \item transformed parameter
#'     \item generated quantity
#'     \item diagnostic
#'   }

setClass("StanParameter", contains = "VIRTUAL",
         slots = list(name = "character", theta = "array", type = "character"),
         prototype = list(name = NA_character_,
                          theta = array(NA_integer_, dim = rep(0L,3L), dimnames = 
                                        list(NULL_INT <- as.integer(NULL), 
                                             chains = NULL_INT, 
                                             iterations = NULL_INT)),
                          type = "generated quantity"),
         validity = function(object) {
           if (length(object@name) != 1)
             return("'name' must be of length one")
           if (!is.numeric(object@theta))
             return("'theta' must be numeric")
           if (length(dim(object@theta)) < 3L)
             return("'theta' must be at least three-dimensional")
           if (length(object@type) != 1)
             return("'type' must be of length one")
           types <- c("unconstrained parameter", "constrained parameter", 
                      "transformed parameter", "generated quantity", "diagnostic")
           if (!(object@type %in% types))
             return(paste("'type' must be among:", 
                          paste(types, collapse = "\n"), sep = "\n"))
           dn <- names(dimnames(object@theta))
           len <- length(dn)
           if ("iterations" != dn[len])
             return("'iterations' must be the 'name' of the last dimension of 'theta'")
           if ("chains" != dn[len-1L])
             return("'chains' must be the 'name' of the next-to-last dimension of 'theta'")
           return(TRUE)
         }
)
#' @rdname StanParameter-class
setClass("StanReal", contains = "StanParameter", validity = function(object) {
  if (nrow(object@theta) != 1)
    return("a 'StanReal' object can only have one row")
  return(TRUE)
})
#' @rdname StanParameter-class
setClass("StanInteger", contains = "StanReal", validity = function(object) {
  if (!is.integer(object@theta))
    return("'theta' must contain integers")
  if (!(object@type %in% c("generated quantity", "diagnostic")))
    return("integer parameters can only be of type 'generated quantity' or 'diagnostic'")
  return(TRUE)
})
#' @rdname StanParameter-class
setClass("StanFactor", contains = "StanInteger", 
         slots = list(labels = "character", ordered = "logical"),
         prototype = list(labels = as.character(NULL), ordered = FALSE), 
         validity = function(object) {
           if (length(labels) < length(unique(object@theta)))
             return("cannot have fewer 'labels' than the number of unique values")
           return(TRUE)
})
#' @rdname StanParameter-class
setClass("StanMatrix", contains = "StanParameter",
         prototype = list(theta = array(NA_real_, rep(0L,4L), dimnames = 
                                        list(NULL_INT <- NULL, NULL_INT, 
                                             chains = NULL_INT, 
                                             iterations = NULL_INT))),
         validity = function(object) {
           if (!is.numeric(object@theta) || is.integer(object@theta))
             return("'theta' must contain real numbers")
           if (length(dim(object@theta)) < 4L)
             return("'theta' must be at least four-dimensional")
           return(TRUE)
})
#' @rdname StanParameter-class
setClass("StanCovMatrix", contains = "StanMatrix", 
         validity = function(object) {
  dims <- dim(object@theta)
  len <- length(dims)
  rows <- dims[len - 3L]
  cols <- dims[len - 2L]
  if (rows != cols)
    return("number of rows must equal the number of columns")
  return(TRUE)
})
#' @rdname StanParameter-class
setClass("StanCorrMatrix", contains = "StanCovMatrix")
#' @rdname StanParameter-class
setClass("StanCholeskyFactorCov", contains = "StanMatrix")
#' @rdname StanParameter-class
setClass("StanCholeskyFactorCorr", contains = "StanCholeskyFactorCov")
#' @rdname StanParameter-class
setClass("StanVector", contains = "StanMatrix")
#' @rdname StanParameter-class
setClass("StanRowVector", contains = "StanVector")
#' @rdname StanParameter-class
setClass("StanSimplex", contains = "StanVector")
#' @rdname StanParameter-class
setClass("StanUnit", contains = "StanVector")

#' A ReferenceClass representing MCMC output
#' 
#' @field warmup_draws A list of \code{\link{StanParameter}}s during warmup
#' @field sample_draws A list of \code{\link{StanParameter}}s after warmup
#' @field added_draws  A list of user-defined \code{\link{StanParameter}}s
#' @field timestamps An integer vector of length two for the start / end time
#' @field seed An integer vector of lenght one giving the PRNG seed
#' @field inits A chains-length list of lists of initial values
#' @field mass_matrix A numeric matrix representing the adapted mass matrix
#' @field permutation A positive integer vector of length iterations x chains
#' 
#' @seealso \code{\link{StanFitOptimization-methods}}
StanFitMCMC <- 
  setRefClass("StanFitMCMC", fields = list(warmup_draws = "list",
                                           sample_draws = "list",
                                           added_draws  = "list",
                                           timestamps = "integer",
                                           seed = "integer",
                                           inits = "list",
                                           mass_matrix = "array", 
                                           permutation = "integer"),
            # FIXME: add other extraction methods but without the get_ prefixes
            methods = list(show = function() {},
                           summary = function() {},
                           as.mcmc.list = function() {},
                           extract = function() {},
                           add_params = function(...) {},
                           help = function(topic) {})
)
StanFitMCMC$lock(setdiff(names(StanFitMCMC$fields()), "added_draws"))

