/*
 * Decompiled with CFR 0.152.
 */
package de.lmu.ifi.dbs.elki.algorithm.clustering.correlation;

import de.lmu.ifi.dbs.elki.algorithm.AbstractAlgorithm;
import de.lmu.ifi.dbs.elki.algorithm.DependencyDerivator;
import de.lmu.ifi.dbs.elki.algorithm.clustering.ClusteringAlgorithm;
import de.lmu.ifi.dbs.elki.algorithm.clustering.correlation.cash.CASHInterval;
import de.lmu.ifi.dbs.elki.algorithm.clustering.correlation.cash.CASHIntervalSplit;
import de.lmu.ifi.dbs.elki.algorithm.clustering.correlation.cash.ParameterizationFunction;
import de.lmu.ifi.dbs.elki.data.Cluster;
import de.lmu.ifi.dbs.elki.data.Clustering;
import de.lmu.ifi.dbs.elki.data.DoubleVector;
import de.lmu.ifi.dbs.elki.data.HyperBoundingBox;
import de.lmu.ifi.dbs.elki.data.NumberVector;
import de.lmu.ifi.dbs.elki.data.model.ClusterModel;
import de.lmu.ifi.dbs.elki.data.model.CorrelationAnalysisSolution;
import de.lmu.ifi.dbs.elki.data.model.LinearEquationModel;
import de.lmu.ifi.dbs.elki.data.model.Model;
import de.lmu.ifi.dbs.elki.data.spatial.SpatialUtil;
import de.lmu.ifi.dbs.elki.data.type.SimpleTypeInformation;
import de.lmu.ifi.dbs.elki.data.type.TypeInformation;
import de.lmu.ifi.dbs.elki.data.type.TypeUtil;
import de.lmu.ifi.dbs.elki.data.type.VectorFieldTypeInformation;
import de.lmu.ifi.dbs.elki.database.Database;
import de.lmu.ifi.dbs.elki.database.ProxyDatabase;
import de.lmu.ifi.dbs.elki.database.datastore.DataStoreUtil;
import de.lmu.ifi.dbs.elki.database.datastore.WritableDataStore;
import de.lmu.ifi.dbs.elki.database.ids.DBIDIter;
import de.lmu.ifi.dbs.elki.database.ids.DBIDUtil;
import de.lmu.ifi.dbs.elki.database.ids.DBIDs;
import de.lmu.ifi.dbs.elki.database.ids.HashSetModifiableDBIDs;
import de.lmu.ifi.dbs.elki.database.ids.ModifiableDBIDs;
import de.lmu.ifi.dbs.elki.database.relation.MaterializedRelation;
import de.lmu.ifi.dbs.elki.database.relation.Relation;
import de.lmu.ifi.dbs.elki.datasource.filter.normalization.NonNumericFeaturesException;
import de.lmu.ifi.dbs.elki.logging.Logging;
import de.lmu.ifi.dbs.elki.logging.progress.FiniteProgress;
import de.lmu.ifi.dbs.elki.math.linearalgebra.LinearEquationSystem;
import de.lmu.ifi.dbs.elki.math.linearalgebra.VMath;
import de.lmu.ifi.dbs.elki.math.linearalgebra.pca.PCARunner;
import de.lmu.ifi.dbs.elki.math.linearalgebra.pca.StandardCovarianceMatrixBuilder;
import de.lmu.ifi.dbs.elki.math.linearalgebra.pca.filter.FirstNEigenPairFilter;
import de.lmu.ifi.dbs.elki.utilities.datastructures.heap.ComparatorMaxHeap;
import de.lmu.ifi.dbs.elki.utilities.datastructures.heap.ObjectHeap;
import de.lmu.ifi.dbs.elki.utilities.documentation.Description;
import de.lmu.ifi.dbs.elki.utilities.documentation.Reference;
import de.lmu.ifi.dbs.elki.utilities.documentation.Title;
import de.lmu.ifi.dbs.elki.utilities.io.FormatUtil;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.AbstractParameterizer;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.OptionID;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.constraints.CommonConstraints;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.constraints.ParameterConstraint;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameterization.Parameterization;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.DoubleParameter;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.Flag;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.IntParameter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import net.jafama.FastMath;

@Title(value="CASH: Robust clustering in arbitrarily oriented subspaces")
@Description(value="Subspace clustering algorithm based on the Hough transform.")
@Reference(authors="Elke Achtert, Christian B\u00f6hm, J\u00f6rn David, Peer Kr\u00f6ger, Arthur Zimek", title="Robust clustering in arbitraily oriented subspaces", booktitle="Proc. 8th SIAM Int. Conf. on Data Mining (SDM'08)", url="https://doi.org/10.1137/1.9781611972788.69", bibkey="DBLP:conf/sdm/AchtertBDKZ08")
public class CASH<V extends NumberVector>
extends AbstractAlgorithm<Clustering<Model>>
implements ClusteringAlgorithm<Clustering<Model>> {
    private static final Logging LOG = Logging.getLogger(CASH.class);
    protected int minPts;
    protected int maxLevel;
    protected int minDim;
    protected double jitter;
    protected boolean adjust;
    private int noiseDim;
    private ModifiableDBIDs processedIDs;
    private Relation<ParameterizationFunction> fulldatabase;

    public CASH(int minPts, int maxLevel, int minDim, double jitter, boolean adjust) {
        this.minPts = minPts;
        this.maxLevel = maxLevel;
        this.minDim = minDim;
        this.jitter = jitter;
        this.adjust = adjust;
    }

    public Clustering<Model> run(Relation<V> rel) {
        this.fulldatabase = this.preprocess(rel);
        this.processedIDs = DBIDUtil.newHashSet(this.fulldatabase.size());
        this.noiseDim = CASH.dimensionality(this.fulldatabase);
        FiniteProgress progress = LOG.isVerbose() ? new FiniteProgress("CASH Clustering", this.fulldatabase.size(), LOG) : null;
        Clustering<Model> result = this.doRun(this.fulldatabase, progress);
        LOG.ensureCompleted(progress);
        if (LOG.isVerbose()) {
            StringBuilder msg = new StringBuilder(1000);
            for (Cluster<Model> c : result.getAllClusters()) {
                if (c.getModel() instanceof LinearEquationModel) {
                    LinearEquationModel s = (LinearEquationModel)c.getModel();
                    msg.append("\n Cluster: Dim: " + s.getLes().subspacedim() + " size: " + c.size());
                    continue;
                }
                msg.append("\n Cluster: " + c.getModel().getClass().getName() + " size: " + c.size());
            }
            LOG.verbose(msg.toString());
        }
        return result;
    }

    private Relation<ParameterizationFunction> preprocess(Relation<V> vrel) {
        DBIDs ids = vrel.getDBIDs();
        SimpleTypeInformation<ParameterizationFunction> type = new SimpleTypeInformation<ParameterizationFunction>(ParameterizationFunction.class);
        WritableDataStore<ParameterizationFunction> prep = DataStoreUtil.makeStorage(ids, 2, ParameterizationFunction.class);
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            prep.put(iter, new ParameterizationFunction((NumberVector)vrel.get(iter)));
            iter.advance();
        }
        return new MaterializedRelation<ParameterizationFunction>(type, ids, null, prep);
    }

    private Clustering<Model> doRun(Relation<ParameterizationFunction> relation, FiniteProgress progress) {
        Clustering<Model> res = new Clustering<Model>("CASH clustering", "cash-clustering");
        int dim = CASH.dimensionality(relation);
        ComparatorMaxHeap<CASHInterval> heap = new ComparatorMaxHeap<CASHInterval>(new Comparator<CASHInterval>(){

            @Override
            public int compare(CASHInterval o1, CASHInterval o2) {
                return Integer.compare(o1.priority(), o2.priority());
            }
        });
        HashSetModifiableDBIDs noiseIDs = DBIDUtil.newHashSet(relation.getDBIDs());
        this.initHeap(heap, relation, dim, noiseIDs);
        if (LOG.isVerbose()) {
            LOG.verbose("dim " + dim + " database.size " + relation.size());
        }
        while (!heap.isEmpty()) {
            CASHInterval interval = this.determineNextIntervalAtMaxLevel(heap);
            if (LOG.isVerbose()) {
                LOG.verbose("next interval in dim " + dim + ": " + interval);
            }
            if (interval == null) break;
            HashSetModifiableDBIDs clusterIDs = DBIDUtil.newHashSet();
            if (dim > this.minDim + 1) {
                double[][] basis_dim_minus_1;
                ModifiableDBIDs ids;
                if (this.adjust) {
                    ids = DBIDUtil.newHashSet();
                    basis_dim_minus_1 = this.runDerivator(relation, dim, interval, ids);
                } else {
                    ids = interval.getIDs();
                    basis_dim_minus_1 = this.determineBasis(SpatialUtil.centroid(interval));
                }
                if (ids.size() != 0) {
                    MaterializedRelation<ParameterizationFunction> db = this.buildDB(dim, basis_dim_minus_1, ids, relation);
                    Clustering<Model> res_dim_minus_1 = this.doRun(db, progress);
                    for (Cluster<Model> cluster : res_dim_minus_1.getAllClusters()) {
                        res.addToplevelCluster(cluster);
                        noiseIDs.removeDBIDs(cluster.getIDs());
                        clusterIDs.addDBIDs(cluster.getIDs());
                        this.processedIDs.addDBIDs(cluster.getIDs());
                    }
                }
            } else {
                LinearEquationSystem les = this.runDerivator(relation, dim - 1, interval.getIDs());
                Cluster<LinearEquationModel> c = new Cluster<LinearEquationModel>((DBIDs)interval.getIDs(), new LinearEquationModel(les));
                res.addToplevelCluster(c);
                noiseIDs.removeDBIDs(interval.getIDs());
                clusterIDs.addDBIDs(interval.getIDs());
                this.processedIDs.addDBIDs(interval.getIDs());
            }
            ArrayList heapVector = new ArrayList(heap.size());
            ObjectHeap.UnsortedIter iter = heap.unsortedIter();
            while (iter.valid()) {
                heapVector.add(iter.get());
                iter.advance();
            }
            heap.clear();
            for (CASHInterval currentInterval : heapVector) {
                currentInterval.removeIDs(clusterIDs);
                if (currentInterval.getIDs().size() < this.minPts) continue;
                heap.add(currentInterval);
            }
            if (progress == null) continue;
            progress.setProcessed(this.processedIDs.size(), LOG);
        }
        if (!noiseIDs.isEmpty()) {
            if (dim == this.noiseDim) {
                res.addToplevelCluster(new Cluster<ClusterModel>(noiseIDs, true, ClusterModel.CLUSTER));
                this.processedIDs.addDBIDs(noiseIDs);
            } else if (noiseIDs.size() >= this.minPts) {
                LinearEquationSystem les = this.runDerivator(this.fulldatabase, dim - 1, noiseIDs);
                res.addToplevelCluster(new Cluster<LinearEquationModel>(noiseIDs, true, new LinearEquationModel(les)));
                this.processedIDs.addDBIDs(noiseIDs);
            }
        }
        if (LOG.isDebugging()) {
            StringBuilder msg = new StringBuilder();
            msg.append("noise fuer dim ").append(dim).append(": ").append(noiseIDs.size());
            for (Cluster<Model> c : res.getAllClusters()) {
                if (c.getModel() instanceof LinearEquationModel) {
                    msg.append("\n Cluster: Dim: ").append(((LinearEquationModel)c.getModel()).getLes().subspacedim());
                } else {
                    msg.append("\n Cluster: ").append(c.getModel().getClass().getName());
                }
                msg.append(" size: ").append(c.size());
            }
            LOG.debugFine(msg.toString());
        }
        if (progress != null) {
            progress.setProcessed(this.processedIDs.size(), LOG);
        }
        return res;
    }

    private static int dimensionality(Relation<ParameterizationFunction> relation) {
        return relation.get(relation.iterDBIDs()).getDimensionality();
    }

    private void initHeap(ObjectHeap<CASHInterval> heap, Relation<ParameterizationFunction> relation, int dim, DBIDs ids) {
        CASHIntervalSplit split = new CASHIntervalSplit(relation, this.minPts);
        double[] minMax = this.determineMinMaxDistance(relation, dim);
        double d_min = minMax[0];
        double d_max = minMax[1];
        double dIntervalLength = d_max - d_min;
        int numDIntervals = (int)FastMath.ceil(dIntervalLength / this.jitter);
        double dIntervalSize = dIntervalLength / (double)numDIntervals;
        double[] d_mins = new double[numDIntervals];
        double[] d_maxs = new double[numDIntervals];
        if (LOG.isVerbose()) {
            LOG.verbose("d_min " + d_min + "\nd_max " + d_max + "\nnumDIntervals " + numDIntervals + "\ndIntervalSize " + dIntervalSize);
        }
        double[] alphaMin = new double[dim - 1];
        double[] alphaMax = new double[dim - 1];
        Arrays.fill(alphaMax, Math.PI);
        for (int i = 0; i < numDIntervals; ++i) {
            d_mins[i] = i == 0 ? d_min : d_maxs[i - 1];
            d_maxs[i] = i < numDIntervals - 1 ? d_mins[i] + dIntervalSize : d_max - d_mins[i];
            HyperBoundingBox alphaInterval = new HyperBoundingBox(alphaMin, alphaMax);
            ModifiableDBIDs intervalIDs = split.determineIDs(ids, alphaInterval, d_mins[i], d_maxs[i]);
            if (intervalIDs == null || intervalIDs.size() < this.minPts) continue;
            heap.add(new CASHInterval(alphaMin, alphaMax, split, intervalIDs, -1, 0, d_mins[i], d_maxs[i]));
        }
        if (LOG.isDebuggingFiner()) {
            LOG.debugFiner("heap.size: " + heap.size());
        }
    }

    private MaterializedRelation<ParameterizationFunction> buildDB(int dim, double[][] basis, DBIDs ids, Relation<ParameterizationFunction> relation) {
        ProxyDatabase proxy = new ProxyDatabase(ids);
        SimpleTypeInformation<ParameterizationFunction> type = new SimpleTypeInformation<ParameterizationFunction>(ParameterizationFunction.class);
        WritableDataStore<ParameterizationFunction> prep = DataStoreUtil.makeStorage(ids, 2, ParameterizationFunction.class);
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            prep.put(iter, this.project(basis, relation.get(iter)));
            iter.advance();
        }
        if (LOG.isDebugging()) {
            LOG.debugFine("db fuer dim " + (dim - 1) + ": " + ids.size());
        }
        MaterializedRelation<ParameterizationFunction> prel = new MaterializedRelation<ParameterizationFunction>(type, ids, null, prep);
        proxy.addRelation(prel);
        return prel;
    }

    private ParameterizationFunction project(double[][] basis, ParameterizationFunction f) {
        double[] m = VMath.transposeTimes(basis, f.getColumnVector());
        return new ParameterizationFunction(DoubleVector.wrap(m));
    }

    private double[][] determineBasis(double[] alpha) {
        int i;
        int dim = alpha.length;
        double[] nn = new double[dim + 1];
        for (int i2 = 0; i2 < nn.length; ++i2) {
            double alpha_i = i2 == alpha.length ? 0.0 : alpha[i2];
            nn[i2] = ParameterizationFunction.sinusProduct(0, i2, alpha) * FastMath.cos(alpha_i);
        }
        VMath.timesEquals(nn, 1.0 / VMath.euclideanLength(nn));
        double[][] basis = new double[dim][];
        int found = 0;
        for (i = 0; i < nn.length && found < dim; ++i) {
            double[] e_i = new double[nn.length];
            e_i[i] = 1.0;
            VMath.minusTimesEquals(e_i, nn, VMath.scalarProduct(e_i, nn));
            double len = VMath.euclideanLength(e_i);
            for (int j = 0; j < found && !(len < 1.0E-9); ++j) {
                VMath.minusTimesEquals(e_i, basis[j], VMath.scalarProduct(e_i, basis[j]));
                len = VMath.euclideanLength(e_i);
            }
            if (len < 1.0E-9) continue;
            VMath.timesEquals(e_i, 1.0 / len);
            basis[found++] = e_i;
        }
        if (found < dim) {
            for (i = found; i < dim; ++i) {
                basis[i] = new double[nn.length];
            }
        }
        return VMath.transpose(basis);
    }

    private CASHInterval determineNextIntervalAtMaxLevel(ObjectHeap<CASHInterval> heap) {
        CASHInterval next = this.doDetermineNextIntervalAtMaxLevel(heap);
        while (next == null) {
            if (heap.isEmpty()) {
                return null;
            }
            next = this.doDetermineNextIntervalAtMaxLevel(heap);
        }
        return next;
    }

    private CASHInterval doDetermineNextIntervalAtMaxLevel(ObjectHeap<CASHInterval> heap) {
        CASHInterval interval = heap.poll();
        int dim = interval.getDimensionality();
        while (interval.getLevel() < this.maxLevel || interval.getMaxSplitDimension() != dim - 1) {
            CASHInterval bestInterval;
            if (heap.size() % 10000 == 0 && LOG.isVerbose()) {
                LOG.verbose("heap size " + heap.size());
            }
            if (heap.size() >= 40000) {
                LOG.warning("Heap size > 40.000! Stopping.");
                heap.clear();
                return null;
            }
            if (LOG.isDebuggingFiner()) {
                LOG.debugFiner("split " + interval.toString() + " " + interval.getLevel() + "-" + interval.getMaxSplitDimension());
            }
            interval.split();
            if (!interval.hasChildren()) {
                return null;
            }
            if (interval.getLeftChild() != null && interval.getRightChild() != null) {
                int comp = interval.getLeftChild().compareTo(interval.getRightChild());
                if (comp < 0) {
                    bestInterval = interval.getRightChild();
                    heap.add(interval.getLeftChild());
                } else {
                    bestInterval = interval.getLeftChild();
                    heap.add(interval.getRightChild());
                }
            } else {
                bestInterval = interval.getLeftChild() == null ? interval.getRightChild() : interval.getLeftChild();
            }
            interval = bestInterval;
        }
        return interval;
    }

    private double[] determineMinMaxDistance(Relation<ParameterizationFunction> relation, int dimensionality) {
        double[] min = new double[dimensionality - 1];
        double[] max = new double[dimensionality - 1];
        Arrays.fill(max, Math.PI);
        HyperBoundingBox box = new HyperBoundingBox(min, max);
        double d_min = Double.POSITIVE_INFINITY;
        double d_max = Double.NEGATIVE_INFINITY;
        DBIDIter iditer = relation.iterDBIDs();
        while (iditer.valid()) {
            ParameterizationFunction f = relation.get(iditer);
            HyperBoundingBox minMax = f.determineAlphaMinMax(box);
            double f_min = f.function(SpatialUtil.getMin(minMax));
            double f_max = f.function(SpatialUtil.getMax(minMax));
            d_min = Math.min(d_min, f_min);
            d_max = Math.max(d_max, f_max);
            iditer.advance();
        }
        return new double[]{d_min, d_max};
    }

    private double[][] runDerivator(Relation<ParameterizationFunction> relation, int dim, CASHInterval interval, ModifiableDBIDs ids) {
        Database derivatorDB = this.buildDerivatorDB(relation, interval);
        PCARunner pca = new PCARunner(new StandardCovarianceMatrixBuilder());
        FirstNEigenPairFilter filter = new FirstNEigenPairFilter(dim - 1);
        DependencyDerivator derivator = new DependencyDerivator(null, FormatUtil.NF4, pca, filter, 0, false);
        CorrelationAnalysisSolution model = (CorrelationAnalysisSolution)derivator.run(derivatorDB);
        double[][] weightMatrix = model.getSimilarityMatrix();
        double[] centroid = model.getCentroid();
        double eps = 0.25;
        ids.addDBIDs(interval.getIDs());
        DBIDIter iditer = relation.iterDBIDs();
        while (iditer.valid()) {
            double[] v = VMath.minusEquals(relation.get(iditer).getColumnVector(), centroid);
            double d = VMath.transposeTimesTimes(v, weightMatrix, v);
            if (d <= eps) {
                ids.add(iditer);
            }
            iditer.advance();
        }
        double[][] basis = model.getStrongEigenvectors();
        return VMath.getMatrix(basis, 0, basis.length, 0, dim - 1);
    }

    private Database buildDerivatorDB(Relation<ParameterizationFunction> relation, CASHInterval interval) {
        ModifiableDBIDs ids = interval.getIDs();
        ProxyDatabase proxy = new ProxyDatabase(ids);
        int dim = CASH.dimensionality(relation);
        VectorFieldTypeInformation<DoubleVector> type = new VectorFieldTypeInformation<DoubleVector>(DoubleVector.FACTORY, dim);
        WritableDataStore<DoubleVector> prep = DataStoreUtil.makeStorage(ids, 2, DoubleVector.class);
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            prep.put(iter, DoubleVector.wrap(relation.get(iter).getColumnVector()));
            iter.advance();
        }
        if (LOG.isDebugging()) {
            LOG.debugFine("db fuer derivator : " + ids.size());
        }
        MaterializedRelation<DoubleVector> prel = new MaterializedRelation<DoubleVector>(type, ids, null, prep);
        proxy.addRelation(prel);
        return proxy;
    }

    private LinearEquationSystem runDerivator(Relation<ParameterizationFunction> relation, int dimensionality, DBIDs ids) {
        try {
            Database derivatorDB = this.buildDerivatorDB(relation, ids);
            PCARunner pca = new PCARunner(new StandardCovarianceMatrixBuilder());
            FirstNEigenPairFilter filter = new FirstNEigenPairFilter(dimensionality);
            DependencyDerivator derivator = new DependencyDerivator(null, FormatUtil.NF4, pca, filter, 0, false);
            CorrelationAnalysisSolution model = (CorrelationAnalysisSolution)derivator.run(derivatorDB);
            LinearEquationSystem les = model.getNormalizedLinearEquationSystem(null);
            return les;
        }
        catch (NonNumericFeaturesException e) {
            throw new IllegalStateException("Error during normalization" + e);
        }
    }

    private Database buildDerivatorDB(Relation<ParameterizationFunction> relation, DBIDs ids) {
        ProxyDatabase proxy = new ProxyDatabase(ids);
        int dim = CASH.dimensionality(relation);
        VectorFieldTypeInformation<DoubleVector> type = new VectorFieldTypeInformation<DoubleVector>(DoubleVector.FACTORY, dim);
        MaterializedRelation<DoubleVector> prep = new MaterializedRelation<DoubleVector>(type, ids);
        proxy.addRelation(prep);
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            prep.insert(iter, DoubleVector.wrap(relation.get(iter).getColumnVector()));
            iter.advance();
        }
        return proxy;
    }

    @Override
    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array(TypeUtil.NUMBER_VECTOR_FIELD);
    }

    @Override
    protected Logging getLogger() {
        return LOG;
    }

    public static class Parameterizer
    extends AbstractParameterizer {
        public static final OptionID MINPTS_ID = new OptionID("cash.minpts", "Threshold for minimum number of points in a cluster.");
        public static final OptionID MAXLEVEL_ID = new OptionID("cash.maxlevel", "The maximum level for splitting the hypercube.");
        public static final OptionID MINDIM_ID = new OptionID("cash.mindim", "The minimum dimensionality of the subspaces to be found.");
        public static final OptionID JITTER_ID = new OptionID("cash.jitter", "The maximum jitter for distance values.");
        public static final OptionID ADJUST_ID = new OptionID("cash.adjust", "Flag to indicate that an adjustment of the applied heuristic for choosing an interval is performed after an interval is selected.");
        protected int minPts;
        protected int maxLevel;
        protected int minDim;
        protected double jitter;
        protected boolean adjust;

        @Override
        protected void makeOptions(Parameterization config) {
            Flag adjustF;
            DoubleParameter jitterP;
            IntParameter mindimP;
            IntParameter maxlevelP;
            super.makeOptions(config);
            IntParameter minptsP = (IntParameter)new IntParameter(MINPTS_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT);
            if (config.grab(minptsP)) {
                this.minPts = (Integer)minptsP.getValue();
            }
            if (config.grab(maxlevelP = (IntParameter)new IntParameter(MAXLEVEL_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT))) {
                this.maxLevel = (Integer)maxlevelP.getValue();
            }
            if (config.grab(mindimP = (IntParameter)new IntParameter(MINDIM_ID, 1).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT))) {
                this.minDim = (Integer)mindimP.getValue();
            }
            if (config.grab(jitterP = (DoubleParameter)new DoubleParameter(JITTER_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE))) {
                this.jitter = (Double)jitterP.getValue();
            }
            if (config.grab(adjustF = new Flag(ADJUST_ID))) {
                this.adjust = (Boolean)adjustF.getValue();
            }
        }

        @Override
        protected CASH<NumberVector> makeInstance() {
            return new CASH<NumberVector>(this.minPts, this.maxLevel, this.minDim, this.jitter, this.adjust);
        }
    }
}

