/*
 * Decompiled with CFR 0.152.
 */
package de.lmu.ifi.dbs.elki.algorithm.outlier;

import de.lmu.ifi.dbs.elki.algorithm.AbstractDistanceBasedAlgorithm;
import de.lmu.ifi.dbs.elki.algorithm.outlier.OutlierAlgorithm;
import de.lmu.ifi.dbs.elki.data.type.TypeInformation;
import de.lmu.ifi.dbs.elki.data.type.TypeUtil;
import de.lmu.ifi.dbs.elki.database.Database;
import de.lmu.ifi.dbs.elki.database.datastore.DataStoreUtil;
import de.lmu.ifi.dbs.elki.database.datastore.WritableDataStore;
import de.lmu.ifi.dbs.elki.database.datastore.WritableDoubleDataStore;
import de.lmu.ifi.dbs.elki.database.datastore.WritableIntegerDataStore;
import de.lmu.ifi.dbs.elki.database.ids.ArrayModifiableDBIDs;
import de.lmu.ifi.dbs.elki.database.ids.DBIDIter;
import de.lmu.ifi.dbs.elki.database.ids.DBIDMIter;
import de.lmu.ifi.dbs.elki.database.ids.DBIDRef;
import de.lmu.ifi.dbs.elki.database.ids.DBIDUtil;
import de.lmu.ifi.dbs.elki.database.ids.DBIDs;
import de.lmu.ifi.dbs.elki.database.ids.DoubleDBIDList;
import de.lmu.ifi.dbs.elki.database.ids.DoubleDBIDListIter;
import de.lmu.ifi.dbs.elki.database.ids.KNNList;
import de.lmu.ifi.dbs.elki.database.ids.ModifiableDBIDs;
import de.lmu.ifi.dbs.elki.database.query.distance.DistanceQuery;
import de.lmu.ifi.dbs.elki.database.query.knn.KNNQuery;
import de.lmu.ifi.dbs.elki.database.query.range.RangeQuery;
import de.lmu.ifi.dbs.elki.database.relation.MaterializedDoubleRelation;
import de.lmu.ifi.dbs.elki.database.relation.Relation;
import de.lmu.ifi.dbs.elki.distance.distancefunction.DistanceFunction;
import de.lmu.ifi.dbs.elki.logging.Logging;
import de.lmu.ifi.dbs.elki.logging.progress.FiniteProgress;
import de.lmu.ifi.dbs.elki.logging.progress.IndefiniteProgress;
import de.lmu.ifi.dbs.elki.logging.progress.StepProgress;
import de.lmu.ifi.dbs.elki.math.DoubleMinMax;
import de.lmu.ifi.dbs.elki.math.Mean;
import de.lmu.ifi.dbs.elki.result.outlier.InvertedOutlierScoreMeta;
import de.lmu.ifi.dbs.elki.result.outlier.OutlierResult;
import de.lmu.ifi.dbs.elki.utilities.documentation.Description;
import de.lmu.ifi.dbs.elki.utilities.documentation.Reference;
import de.lmu.ifi.dbs.elki.utilities.documentation.Title;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.OptionID;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.constraints.CommonConstraints;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.constraints.ParameterConstraint;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameterization.Parameterization;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.DoubleParameter;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.IntParameter;

@Title(value="DWOF: Dynamic Window Outlier Factor")
@Description(value="Algorithm to compute dynamic-window outlier factors in a database based on the neighborhood size parameter 'k'")
@Reference(authors="R. Momtaz, N. Mohssen, M. A. Gowayyed", title="DWOF: A Robust Density-Based Outlier Detection Approach", booktitle="Proc. 6th Iberian Conf. Pattern Recognition and Image Analysis (IbPRIA 2013)", url="https://doi.org/10.1007/978-3-642-38628-2_61", bibkey="DBLP:conf/ibpria/MomtazMG13")
public class DWOF<O>
extends AbstractDistanceBasedAlgorithm<O, OutlierResult>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(DWOF.class);
    protected int k;
    private double delta = 1.1;

    public DWOF(DistanceFunction<? super O> distanceFunction, int k, double delta) {
        super(distanceFunction);
        this.k = k + 1;
        this.delta = delta;
    }

    public OutlierResult run(Database database, Relation<O> relation) {
        IndefiniteProgress clusEvalProgress;
        DBIDs ids = relation.getDBIDs();
        DistanceQuery<O> distFunc = database.getDistanceQuery(relation, this.getDistanceFunction(), new Object[0]);
        KNNQuery<O> knnq = database.getKNNQuery(distFunc, this.k, "heavy");
        RangeQuery<O> rnnQuery = database.getRangeQuery(distFunc, "heavy");
        StepProgress stepProg = LOG.isVerbose() ? new StepProgress("DWOF", 2) : null;
        WritableDoubleDataStore dwofs = DataStoreUtil.makeDoubleStorage(ids, 30, 0.0);
        if (stepProg != null) {
            stepProg.beginStep(1, "Initializing objects' Radii", LOG);
        }
        WritableDoubleDataStore radii = DataStoreUtil.makeDoubleStorage(ids, 3, 0.0);
        this.initializeRadii(ids, knnq, distFunc, radii);
        WritableIntegerDataStore oldSizes = DataStoreUtil.makeIntegerStorage(ids, 2, 1);
        WritableIntegerDataStore newSizes = DataStoreUtil.makeIntegerStorage(ids, 2, 1);
        int countUnmerged = relation.size();
        if (stepProg != null) {
            stepProg.beginStep(2, "Clustering-Evaluating Cycles.", LOG);
        }
        IndefiniteProgress indefiniteProgress = clusEvalProgress = LOG.isVerbose() ? new IndefiniteProgress("Evaluating DWOFs", LOG) : null;
        while (countUnmerged > 0) {
            LOG.incrementProcessed(clusEvalProgress);
            DBIDIter iter = ids.iter();
            while (iter.valid()) {
                radii.putDouble(iter, radii.doubleValue(iter) * this.delta);
                iter.advance();
            }
            WritableDataStore<ModifiableDBIDs> labels = DataStoreUtil.makeStorage(ids, 1, ModifiableDBIDs.class);
            this.clusterData(ids, rnnQuery, radii, labels);
            WritableIntegerDataStore temp = newSizes;
            newSizes = oldSizes;
            oldSizes = temp;
            countUnmerged = this.updateSizes(ids, labels, newSizes);
            labels.destroy();
            DBIDIter iter2 = ids.iter();
            while (iter2.valid()) {
                double newScore = newSizes.intValue(iter2) > 0 ? (double)(oldSizes.intValue(iter2) - 1) / (double)newSizes.intValue(iter2) : 0.0;
                dwofs.putDouble(iter2, dwofs.doubleValue(iter2) + newScore);
                iter2.advance();
            }
        }
        LOG.setCompleted(clusEvalProgress);
        LOG.setCompleted(stepProg);
        DoubleMinMax minmax = new DoubleMinMax();
        DBIDIter iter = relation.iterDBIDs();
        while (iter.valid()) {
            minmax.put(dwofs.doubleValue(iter));
            iter.advance();
        }
        InvertedOutlierScoreMeta meta = new InvertedOutlierScoreMeta(minmax.getMin(), minmax.getMax(), 0.0, Double.POSITIVE_INFINITY);
        MaterializedDoubleRelation rel = new MaterializedDoubleRelation("Dynamic-Window Outlier Factors", "dwof-outlier", dwofs, ids);
        return new OutlierResult(meta, rel);
    }

    private void initializeRadii(DBIDs ids, KNNQuery<O> knnq, DistanceQuery<O> distFunc, WritableDoubleDataStore radii) {
        FiniteProgress avgDistProgress = LOG.isVerbose() ? new FiniteProgress("Calculating average kNN distances-", ids.size(), LOG) : null;
        double absoluteMinDist = Double.POSITIVE_INFINITY;
        double minAvgDist = Double.POSITIVE_INFINITY;
        Mean mean = new Mean();
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            KNNList iterNeighbors = knnq.getKNNForDBID(iter, this.k);
            mean.reset();
            DoubleDBIDListIter neighbor1 = iterNeighbors.iter();
            while (neighbor1.valid()) {
                if (!DBIDUtil.equal(neighbor1, iter)) {
                    DoubleDBIDListIter neighbor2 = iterNeighbors.iter();
                    while (neighbor2.valid()) {
                        if (!DBIDUtil.equal(neighbor1, neighbor2) && !DBIDUtil.equal(neighbor2, iter)) {
                            double distance = distFunc.distance((DBIDRef)neighbor1, (DBIDRef)neighbor2);
                            mean.put(distance);
                            if (distance > 0.0 && distance < absoluteMinDist) {
                                absoluteMinDist = distance;
                            }
                        }
                        neighbor2.advance();
                    }
                }
                neighbor1.advance();
            }
            double currentMean = mean.getMean();
            radii.putDouble(iter, currentMean);
            if (currentMean < minAvgDist) {
                minAvgDist = currentMean;
            }
            LOG.incrementProcessed(avgDistProgress);
            iter.advance();
        }
        LOG.ensureCompleted(avgDistProgress);
        iter = ids.iter();
        while (iter.valid()) {
            radii.putDouble(iter, minAvgDist > 0.0 ? absoluteMinDist * radii.doubleValue(iter) / minAvgDist : Double.POSITIVE_INFINITY);
            iter.advance();
        }
    }

    private void clusterData(DBIDs ids, RangeQuery<O> rnnQuery, WritableDoubleDataStore radii, WritableDataStore<ModifiableDBIDs> labels) {
        FiniteProgress clustProg = LOG.isVerbose() ? new FiniteProgress("Density-Based Clustering", ids.size(), LOG) : null;
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            if (labels.get(iter) == null) {
                ArrayModifiableDBIDs newCluster = DBIDUtil.newArray();
                newCluster.add(iter);
                labels.put(iter, newCluster);
                LOG.incrementProcessed(clustProg);
                ArrayModifiableDBIDs nChain = DBIDUtil.newArray();
                nChain.add(iter);
                DBIDMIter toGetNeighbors = nChain.iter();
                while (toGetNeighbors.valid()) {
                    double range = radii.doubleValue(toGetNeighbors);
                    DoubleDBIDList nNeighbors = rnnQuery.getRangeForDBID(toGetNeighbors, range);
                    DoubleDBIDListIter iter2 = nNeighbors.iter();
                    while (iter2.valid()) {
                        if (!DBIDUtil.equal(toGetNeighbors, iter2)) {
                            if (labels.get(iter2) == null) {
                                newCluster.add(iter2);
                                labels.put(iter2, newCluster);
                                nChain.add(iter2);
                                LOG.incrementProcessed(clustProg);
                            } else if (labels.get(iter2) != newCluster) {
                                ModifiableDBIDs toBeDeleted = (ModifiableDBIDs)labels.get(iter2);
                                newCluster.addDBIDs(toBeDeleted);
                                DBIDMIter iter3 = toBeDeleted.iter();
                                while (iter3.valid()) {
                                    labels.put(iter3, newCluster);
                                    iter3.advance();
                                }
                                toBeDeleted.clear();
                            }
                        }
                        iter2.advance();
                    }
                    toGetNeighbors.advance();
                }
            }
            iter.advance();
        }
        LOG.ensureCompleted(clustProg);
    }

    private int updateSizes(DBIDs ids, WritableDataStore<ModifiableDBIDs> labels, WritableIntegerDataStore newSizes) {
        int countUnmerged = 0;
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            int newClusterSize = ((ModifiableDBIDs)labels.get(iter)).size();
            newSizes.putInt(iter, newClusterSize);
            if (newClusterSize == 1) {
                ++countUnmerged;
            }
            iter.advance();
        }
        return countUnmerged;
    }

    @Override
    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array(this.getDistanceFunction().getInputTypeRestriction());
    }

    @Override
    protected Logging getLogger() {
        return LOG;
    }

    public static class Parameterizer<O>
    extends AbstractDistanceBasedAlgorithm.Parameterizer<O> {
        public static final OptionID K_ID = OptionID.getOrCreateOptionID("dwof.k", "Number of neighbors to get for DWOF score outlier detection.");
        public static final OptionID DELTA_ID = OptionID.getOrCreateOptionID("dwof.delta", "Radius increase factor.");
        protected int k;
        protected double delta = 1.1;

        @Override
        protected void makeOptions(Parameterization config) {
            DoubleParameter deltaP;
            super.makeOptions(config);
            IntParameter kP = (IntParameter)new IntParameter(K_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ONE_INT);
            if (config.grab(kP)) {
                this.k = (Integer)kP.getValue();
            }
            if (config.grab(deltaP = (DoubleParameter)((DoubleParameter)new DoubleParameter(DELTA_ID).setDefaultValue((Object)1.1)).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ONE_DOUBLE))) {
                this.delta = (Double)deltaP.getValue();
            }
        }

        @Override
        protected DWOF<O> makeInstance() {
            return new DWOF(this.distanceFunction, this.k, this.delta);
        }
    }
}

