/*
 * Decompiled with CFR 0.152.
 */
package de.lmu.ifi.dbs.elki.algorithm.outlier.lof;

import de.lmu.ifi.dbs.elki.algorithm.AbstractDistanceBasedAlgorithm;
import de.lmu.ifi.dbs.elki.algorithm.outlier.OutlierAlgorithm;
import de.lmu.ifi.dbs.elki.data.type.TypeInformation;
import de.lmu.ifi.dbs.elki.data.type.TypeUtil;
import de.lmu.ifi.dbs.elki.database.Database;
import de.lmu.ifi.dbs.elki.database.datastore.DataStoreUtil;
import de.lmu.ifi.dbs.elki.database.datastore.WritableDataStore;
import de.lmu.ifi.dbs.elki.database.datastore.WritableDoubleDataStore;
import de.lmu.ifi.dbs.elki.database.ids.DBIDIter;
import de.lmu.ifi.dbs.elki.database.ids.DBIDs;
import de.lmu.ifi.dbs.elki.database.ids.DoubleDBIDList;
import de.lmu.ifi.dbs.elki.database.ids.DoubleDBIDListIter;
import de.lmu.ifi.dbs.elki.database.query.distance.DistanceQuery;
import de.lmu.ifi.dbs.elki.database.query.range.RangeQuery;
import de.lmu.ifi.dbs.elki.database.relation.MaterializedDoubleRelation;
import de.lmu.ifi.dbs.elki.database.relation.Relation;
import de.lmu.ifi.dbs.elki.distance.distancefunction.DistanceFunction;
import de.lmu.ifi.dbs.elki.logging.Logging;
import de.lmu.ifi.dbs.elki.logging.progress.FiniteProgress;
import de.lmu.ifi.dbs.elki.math.DoubleMinMax;
import de.lmu.ifi.dbs.elki.math.MeanVariance;
import de.lmu.ifi.dbs.elki.result.outlier.OutlierResult;
import de.lmu.ifi.dbs.elki.result.outlier.QuotientOutlierScoreMeta;
import de.lmu.ifi.dbs.elki.utilities.Alias;
import de.lmu.ifi.dbs.elki.utilities.datastructures.arrays.DoubleIntegerArrayQuickSort;
import de.lmu.ifi.dbs.elki.utilities.documentation.Description;
import de.lmu.ifi.dbs.elki.utilities.documentation.Reference;
import de.lmu.ifi.dbs.elki.utilities.documentation.Title;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.OptionID;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameterization.Parameterization;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.DoubleParameter;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.IntParameter;
import java.util.Arrays;

@Title(value="LOCI: Fast Outlier Detection Using the Local Correlation Integral")
@Description(value="Algorithm to compute outliers based on the Local Correlation Integral")
@Reference(authors="S. Papadimitriou, H. Kitagawa, P. B. Gibbons, C. Faloutsos", title="LOCI: Fast Outlier Detection Using the Local Correlation Integral", booktitle="Proc. 19th IEEE Int. Conf. on Data Engineering (ICDE '03)", url="https://doi.org/10.1109/ICDE.2003.1260802", bibkey="DBLP:conf/icde/PapadimitriouKGF03")
@Alias(value={"de.lmu.ifi.dbs.elki.algorithm.outlier.LOCI"})
public class LOCI<O>
extends AbstractDistanceBasedAlgorithm<O, OutlierResult>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(LOCI.class);
    private double rmax;
    private int nmin = 0;
    private double alpha = 0.5;

    public LOCI(DistanceFunction<? super O> distanceFunction, double rmax, int nmin, double alpha) {
        super(distanceFunction);
        this.rmax = rmax;
        this.nmin = nmin;
        this.alpha = alpha;
    }

    public OutlierResult run(Database database, Relation<O> relation) {
        DistanceQuery<O> distFunc = database.getDistanceQuery(relation, this.getDistanceFunction(), new Object[0]);
        RangeQuery<O> rangeQuery = database.getRangeQuery(distFunc, new Object[0]);
        DBIDs ids = relation.getDBIDs();
        WritableDataStore<DoubleIntArrayList> interestingDistances = DataStoreUtil.makeStorage(relation.getDBIDs(), 9, DoubleIntArrayList.class);
        this.precomputeInterestingRadii(ids, rangeQuery, interestingDistances);
        FiniteProgress progressLOCI = LOG.isVerbose() ? new FiniteProgress("LOCI scores", relation.size(), LOG) : null;
        WritableDoubleDataStore mdef_norm = DataStoreUtil.makeDoubleStorage(relation.getDBIDs(), 4);
        WritableDoubleDataStore mdef_radius = DataStoreUtil.makeDoubleStorage(relation.getDBIDs(), 4);
        DoubleMinMax minmax = new DoubleMinMax();
        MeanVariance mv_n_r_alpha = new MeanVariance();
        DBIDIter iditer = ids.iter();
        while (iditer.valid()) {
            DoubleIntArrayList cdist = (DoubleIntArrayList)interestingDistances.get(iditer);
            double maxdist = cdist.getDouble(cdist.size() - 1);
            int maxneig = cdist.getInt(cdist.size() - 1);
            double maxmdefnorm = 0.0;
            double maxnormr = 0.0;
            if (maxneig >= this.nmin) {
                DoubleDBIDList maxneighbors = rangeQuery.getRangeForDBID(iditer, maxdist);
                int size = cdist.size();
                for (int i = 0; i < size; ++i) {
                    double sigma_nhat_r_alpha;
                    double sigmamdef;
                    if (cdist.getInt(i) < this.nmin) continue;
                    double r = cdist.getDouble(i);
                    double alpha_r = this.alpha * r;
                    int n_alphar = cdist.getInt(cdist.find(alpha_r));
                    mv_n_r_alpha.reset();
                    DoubleDBIDListIter neighbor = maxneighbors.iter();
                    while (neighbor.valid() && !(neighbor.doubleValue() > r)) {
                        DoubleIntArrayList cdist2 = (DoubleIntArrayList)interestingDistances.get(neighbor);
                        int rn_alphar = cdist2.getInt(cdist2.find(alpha_r));
                        mv_n_r_alpha.put(rn_alphar);
                        neighbor.advance();
                    }
                    double nhat_r_alpha = mv_n_r_alpha.getMean();
                    double mdef = nhat_r_alpha - (double)n_alphar;
                    double mdefnorm = mdef / (sigmamdef = (sigma_nhat_r_alpha = mv_n_r_alpha.getNaiveStddev()));
                    if (!(mdefnorm > maxmdefnorm)) continue;
                    maxmdefnorm = mdefnorm;
                    maxnormr = r;
                }
            } else {
                maxmdefnorm = Double.POSITIVE_INFINITY;
                maxnormr = maxdist;
            }
            mdef_norm.putDouble(iditer, maxmdefnorm);
            mdef_radius.putDouble(iditer, maxnormr);
            minmax.put(maxmdefnorm);
            LOG.incrementProcessed(progressLOCI);
            iditer.advance();
        }
        LOG.ensureCompleted(progressLOCI);
        MaterializedDoubleRelation scoreResult = new MaterializedDoubleRelation("LOCI normalized MDEF", "loci-mdef-outlier", mdef_norm, relation.getDBIDs());
        QuotientOutlierScoreMeta scoreMeta = new QuotientOutlierScoreMeta(minmax.getMin(), minmax.getMax(), 0.0, Double.POSITIVE_INFINITY, 0.0);
        OutlierResult result = new OutlierResult(scoreMeta, scoreResult);
        result.addChildResult(new MaterializedDoubleRelation("LOCI MDEF Radius", "loci-critical-radius", mdef_radius, relation.getDBIDs()));
        return result;
    }

    protected void precomputeInterestingRadii(DBIDs ids, RangeQuery<O> rangeQuery, WritableDataStore<DoubleIntArrayList> interestingDistances) {
        FiniteProgress progressPreproc = LOG.isVerbose() ? new FiniteProgress("LOCI preprocessing", ids.size(), LOG) : null;
        DBIDIter iditer = ids.iter();
        while (iditer.valid()) {
            DoubleDBIDList neighbors = rangeQuery.getRangeForDBID(iditer, this.rmax);
            DoubleIntArrayList cdist = new DoubleIntArrayList(neighbors.size() << 1);
            int i = 0;
            DoubleDBIDListIter ni = neighbors.iter();
            while (ni.valid()) {
                double ri;
                double curdist = ni.doubleValue();
                ++i;
                ni.advance();
                if (ni.valid() && curdist == ni.doubleValue()) continue;
                cdist.append(curdist, i);
                if (this.alpha == 1.0 || !((ri = curdist / this.alpha) <= this.rmax)) continue;
                cdist.append(ri, Integer.MIN_VALUE);
            }
            cdist.sort();
            int lastk = 0;
            int size = cdist.size();
            for (int i2 = 0; i2 < size; ++i2) {
                int k = cdist.getInt(i2);
                if (k == Integer.MIN_VALUE) {
                    cdist.setValue(i2, lastk);
                    continue;
                }
                lastk = k;
            }
            interestingDistances.put(iditer, cdist);
            LOG.incrementProcessed(progressPreproc);
            iditer.advance();
        }
        LOG.ensureCompleted(progressPreproc);
    }

    @Override
    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array(this.getDistanceFunction().getInputTypeRestriction());
    }

    @Override
    protected Logging getLogger() {
        return LOG;
    }

    public static class Parameterizer<O>
    extends AbstractDistanceBasedAlgorithm.Parameterizer<O> {
        public static final OptionID RMAX_ID = new OptionID("loci.rmax", "The maximum radius of the neighborhood to be considered.");
        public static final OptionID NMIN_ID = new OptionID("loci.nmin", "Minimum neighborhood size to be considered.");
        public static final OptionID ALPHA_ID = new OptionID("loci.alpha", "Scaling factor for averaging neighborhood");
        protected double rmax;
        protected int nmin = 0;
        protected double alpha = 0.5;

        @Override
        protected void makeOptions(Parameterization config) {
            DoubleParameter alphaP;
            IntParameter nminP;
            super.makeOptions(config);
            DoubleParameter rmaxP = new DoubleParameter(RMAX_ID);
            if (config.grab(rmaxP)) {
                this.rmax = rmaxP.doubleValue();
            }
            if (config.grab(nminP = new IntParameter(NMIN_ID, 20))) {
                this.nmin = nminP.intValue();
            }
            if (config.grab(alphaP = new DoubleParameter(ALPHA_ID, 0.5))) {
                this.alpha = (Double)alphaP.getValue();
            }
        }

        @Override
        protected LOCI<O> makeInstance() {
            return new LOCI(this.distanceFunction, this.rmax, this.nmin, this.alpha);
        }
    }

    private static class DoubleIntArrayList {
        double[] keys;
        int[] vals;
        int size = 0;

        public DoubleIntArrayList(int alloc) {
            this.keys = new double[alloc];
            this.vals = new int[alloc];
            this.size = 0;
        }

        public int size() {
            return this.size;
        }

        public double getDouble(int i) {
            return this.keys[i];
        }

        public int getInt(int i) {
            return this.vals[i];
        }

        public void setValue(int i, int val) {
            this.vals[i] = val;
        }

        public void append(double key, int val) {
            if (this.size == this.keys.length) {
                this.keys = Arrays.copyOf(this.keys, this.size << 1);
                this.vals = Arrays.copyOf(this.vals, this.size << 1);
            }
            this.keys[this.size] = key;
            this.vals[this.size] = val;
            ++this.size;
        }

        public int find(double search) {
            int a = 0;
            int b = this.size - 1;
            while (a <= b) {
                int mid = a + b >>> 1;
                double cur = this.keys[mid];
                if (cur > search) {
                    b = mid - 1;
                    continue;
                }
                a = mid + 1;
            }
            return b;
        }

        public void sort() {
            DoubleIntegerArrayQuickSort.sort(this.keys, this.vals, this.size);
        }
    }
}

