/*
 * Decompiled with CFR 0.152.
 */
package de.lmu.ifi.dbs.elki.algorithm.timeseries;

import de.lmu.ifi.dbs.elki.algorithm.AbstractAlgorithm;
import de.lmu.ifi.dbs.elki.algorithm.timeseries.ChangePoints;
import de.lmu.ifi.dbs.elki.data.DoubleVector;
import de.lmu.ifi.dbs.elki.data.type.TypeInformation;
import de.lmu.ifi.dbs.elki.data.type.TypeUtil;
import de.lmu.ifi.dbs.elki.database.ids.ArrayDBIDs;
import de.lmu.ifi.dbs.elki.database.ids.DBIDArrayIter;
import de.lmu.ifi.dbs.elki.database.relation.Relation;
import de.lmu.ifi.dbs.elki.database.relation.RelationUtil;
import de.lmu.ifi.dbs.elki.logging.Logging;
import de.lmu.ifi.dbs.elki.utilities.documentation.Description;
import de.lmu.ifi.dbs.elki.utilities.documentation.Reference;
import de.lmu.ifi.dbs.elki.utilities.documentation.References;
import de.lmu.ifi.dbs.elki.utilities.documentation.Title;
import de.lmu.ifi.dbs.elki.utilities.exceptions.AbortException;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.AbstractParameterizer;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.OptionID;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.constraints.CommonConstraints;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.constraints.ParameterConstraint;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameterization.Parameterization;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.DoubleParameter;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.IntParameter;
import de.lmu.ifi.dbs.elki.utilities.optionhandling.parameters.RandomParameter;
import de.lmu.ifi.dbs.elki.utilities.pairs.DoubleIntPair;
import de.lmu.ifi.dbs.elki.utilities.random.RandomFactory;
import java.util.Random;

@Title(value="Off-line Change Point Detection")
@Description(value="Detects multiple change points in a time series")
@References(value={@Reference(authors="D. Picard", title="Testing and Estimating Change-Points in Time Series ", booktitle="Advances in Applied Probability Vol. 17", url="https://doi.org/10.2307/1427090", bibkey="doi:10.2307/1427090"), @Reference(authors="E. S. Page", title="On Problems in which a Change in a Parameter Occurs at an Unknown Point", booktitle="Biometrika Vol. 44", url="https://doi.org/10.2307/2333258", bibkey="doi:10.2307/2333258"), @Reference(authors="M. Basseville, I. V. Nikiforov", title="Section 2.6: Off-line Change Detection", booktitle="Detection of Abrupt Changes - Theory and Application", url="http://people.irisa.fr/Michele.Basseville/kniga/kniga.pdf", bibkey="books/prentice/BassevilleN93/C2")})
public class OfflineChangePointDetectionAlgorithm
extends AbstractAlgorithm<ChangePoints> {
    private static final Logging LOG = Logging.getLogger(OfflineChangePointDetectionAlgorithm.class);
    int bootstrapSamples;
    double minConfidence;
    RandomFactory rnd;

    public OfflineChangePointDetectionAlgorithm(double confidence, int bootstrapSteps, RandomFactory rnd) {
        this.minConfidence = confidence;
        this.bootstrapSamples = bootstrapSteps;
        this.rnd = rnd;
    }

    public ChangePoints run(Relation<DoubleVector> relation) {
        if (!(relation.getDBIDs() instanceof ArrayDBIDs)) {
            throw new AbortException("This implementation may only be used on static databases, with ArrayDBIDs to provide a clear order.");
        }
        return new Instance(this.rnd.getSingleThreadedRandom()).run(relation);
    }

    public static void cusum(double[] data, double[] out, int begin, int end) {
        assert (out.length >= data.length);
        double m = 0.0;
        double carry = 0.0;
        for (int i = begin; i < end; ++i) {
            double v = data[i] - carry;
            double n = out[i] = m + v;
            carry = n - m - v;
            m = n;
        }
    }

    public static DoubleIntPair bestChangeInMean(double[] sums, int begin, int end) {
        int len = end - begin;
        int last = end - 1;
        double suml = begin > 0 ? sums[begin - 1] : 0.0;
        double sumr = sums[last];
        int bestpos = begin;
        double bestscore = Double.NEGATIVE_INFINITY;
        int j = begin;
        int km1 = 1;
        while (j < last) {
            assert (km1 < len);
            double sumj = sums[j];
            double lmean = (sumj - suml) / (double)km1;
            double rmean = (sumr - sumj) / (double)(len - km1);
            double dm = lmean - rmean;
            double score = (double)km1 * (double)(len - km1) * dm * dm;
            if (score > bestscore) {
                bestpos = j + 1;
                bestscore = score;
            }
            ++j;
            ++km1;
        }
        return new DoubleIntPair(bestscore, bestpos);
    }

    public static void shuffle(double[] bstrap, int len, Random rnd) {
        int i = len;
        while (i > 0) {
            int r = rnd.nextInt(i);
            double tmp = bstrap[r];
            bstrap[r] = bstrap[--i];
            bstrap[i] = tmp;
        }
    }

    @Override
    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array(TypeUtil.NUMBER_VECTOR_VARIABLE_LENGTH);
    }

    @Override
    protected Logging getLogger() {
        return LOG;
    }

    public static class Parameterizer
    extends AbstractParameterizer {
        public static final OptionID BOOTSTRAP_ID = new OptionID("changepointdetection.bootstrap.samples", "Number of samples to draw for bootstrapping the confidence estimate.");
        public static final OptionID CONFIDENCE_ID = new OptionID("changepointdetection.bootstrap.confidence", "Confidence level to use with bootstrap sampling.");
        public static final OptionID RANDOM_ID = new OptionID("changepointdetection.seed", "Random generator seed for bootstrap sampling.");
        int bootstrapSamples = 1000;
        double minConfidence;
        RandomFactory rnd;

        @Override
        protected void makeOptions(Parameterization config) {
            RandomParameter rndP;
            DoubleParameter confidenceP;
            super.makeOptions(config);
            IntParameter bootstrapsamplesP = (IntParameter)((IntParameter)new IntParameter(BOOTSTRAP_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).setDefaultValue((Object)1000);
            if (config.grab(bootstrapsamplesP)) {
                this.bootstrapSamples = (Integer)bootstrapsamplesP.getValue();
            }
            if (config.grab(confidenceP = (DoubleParameter)((DoubleParameter)((DoubleParameter)new DoubleParameter(CONFIDENCE_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ZERO_DOUBLE)).addConstraint((ParameterConstraint)CommonConstraints.LESS_THAN_ONE_DOUBLE)).setDefaultValue((Object)(1.0 - 2.5 / (double)this.bootstrapSamples)))) {
                this.minConfidence = confidenceP.doubleValue();
            }
            if (config.grab(rndP = new RandomParameter(RANDOM_ID))) {
                this.rnd = (RandomFactory)rndP.getValue();
            }
        }

        @Override
        protected OfflineChangePointDetectionAlgorithm makeInstance() {
            return new OfflineChangePointDetectionAlgorithm(this.minConfidence, this.bootstrapSamples, this.rnd);
        }
    }

    class Instance {
        double[] column;
        double[] sums;
        double[] bstrap;
        DBIDArrayIter iter;
        ChangePoints result;
        int columnnr;
        Random rnd;

        public Instance(Random rnd) {
            this.rnd = rnd;
        }

        public ChangePoints run(Relation<DoubleVector> relation) {
            ArrayDBIDs ids = (ArrayDBIDs)relation.getDBIDs();
            int dim = RelationUtil.dimensionality(relation);
            int size = ids.size();
            this.iter = ids.iter();
            this.column = new double[size];
            this.sums = new double[size];
            this.bstrap = new double[size];
            this.result = new ChangePoints("CUSUM Changepoints", "cusum-changepoints");
            this.columnnr = 0;
            while (this.columnnr < dim) {
                this.iter.seek(0);
                while (this.iter.valid()) {
                    this.column[this.iter.getOffset()] = relation.get(this.iter).doubleValue(this.columnnr);
                    this.iter.advance();
                }
                OfflineChangePointDetectionAlgorithm.cusum(this.column, this.sums, 0, size);
                this.multipleChangepointsWithConfidence(0, size);
                ++this.columnnr;
            }
            return this.result;
        }

        private int multipleChangepointsWithConfidence(int begin, int end) {
            if (end - begin <= 3) {
                return begin;
            }
            DoubleIntPair change = OfflineChangePointDetectionAlgorithm.bestChangeInMean(this.sums, begin, end);
            double confidence = this.bootstrapConfidence(begin, end, change.first);
            if (confidence < OfflineChangePointDetectionAlgorithm.this.minConfidence) {
                return begin;
            }
            this.multipleChangepointsWithConfidence(begin, change.second);
            this.result.add(this.iter.seek(change.second), this.columnnr, confidence);
            return this.multipleChangepointsWithConfidence(change.second, end);
        }

        private double bootstrapConfidence(int begin, int end, double thresh) {
            int len = end - begin;
            int pos = 0;
            for (int i = 0; i < OfflineChangePointDetectionAlgorithm.this.bootstrapSamples; ++i) {
                System.arraycopy(this.column, begin, this.bstrap, 0, len);
                OfflineChangePointDetectionAlgorithm.shuffle(this.bstrap, len, this.rnd);
                OfflineChangePointDetectionAlgorithm.cusum(this.bstrap, this.bstrap, 0, len);
                double score = OfflineChangePointDetectionAlgorithm.bestChangeInMean((double[])this.bstrap, (int)0, (int)len).first;
                if (!(score < thresh)) continue;
                ++pos;
            }
            return (double)pos / (double)OfflineChangePointDetectionAlgorithm.this.bootstrapSamples;
        }
    }
}

