/*
 * Decompiled with CFR 0.152.
 */
package de.lmu.ifi.dbs.elki.math.geometry;

import de.lmu.ifi.dbs.elki.data.spatial.Polygon;
import de.lmu.ifi.dbs.elki.logging.Logging;
import de.lmu.ifi.dbs.elki.math.DoubleMinMax;
import de.lmu.ifi.dbs.elki.utilities.datastructures.BitsUtil;
import de.lmu.ifi.dbs.elki.utilities.datastructures.arrays.DoubleIntegerArrayQuickSort;
import de.lmu.ifi.dbs.elki.utilities.documentation.Reference;
import de.lmu.ifi.dbs.elki.utilities.pairs.IntIntPair;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

@Reference(authors="D. Sinclair", title="S-hull: a fast sweep-hull routine for Delaunay triangulation", booktitle="Online", url="http://s-hull.org/", bibkey="web/Sinclair16")
public class SweepHullDelaunay2D {
    private static final Logging LOG = Logging.getLogger(SweepHullDelaunay2D.class);
    private List<double[]> points;
    private ArrayList<Triangle> tris = null;
    private LinkedList<IntIntPair> hull = null;

    public SweepHullDelaunay2D() {
        this(new ArrayList<double[]>());
    }

    public SweepHullDelaunay2D(List<double[]> points) {
        this.points = points;
    }

    public void add(double ... point) {
        this.points.add(point);
        this.hull = null;
        this.tris = null;
    }

    void run(boolean hullonly) {
        int i;
        if (this.points.size() < 3) {
            throw new UnsupportedOperationException("There is no delaunay triangulation for less than three objects!");
        }
        int len = this.points.size() - 1;
        this.hull = new LinkedList();
        this.tris = hullonly ? null : new ArrayList(len);
        boolean seedid = false;
        double[] sortd = new double[len];
        int[] sorti = new int[len];
        Arrays.fill(sorti, -42);
        Iterator<double[]> iter = this.points.iterator();
        double[] seed = iter.next();
        int i2 = 0;
        int j = 1;
        while (iter.hasNext()) {
            double dist = SweepHullDelaunay2D.quadraticEuclidean(seed, iter.next());
            if (dist <= 0.0) {
                --len;
                --i2;
            } else {
                sortd[i2] = dist;
                sorti[i2] = j;
            }
            ++j;
            ++i2;
        }
        DoubleIntegerArrayQuickSort.sort(sortd, sorti, len);
        if (len < 2) {
            this.hull.add(new IntIntPair(0, -1));
            if (len == 1) {
                this.hull.add(new IntIntPair(sorti[0], -1));
            }
            return;
        }
        assert (sortd[0] > 0.0);
        int seed2id = sorti[0];
        Triangle besttri = this.findSmallest(0, seed2id, sortd, sorti, len);
        if (besttri == null) {
            this.hull.add(new IntIntPair(0, -1));
            this.hull.add(new IntIntPair(seed2id, -1));
            return;
        }
        int start = 2;
        besttri.makeClockwise(this.points);
        if (!hullonly) {
            this.tris.add(besttri);
        }
        this.hull.add(new IntIntPair(besttri.a, 0));
        this.hull.add(new IntIntPair(besttri.b, 0));
        this.hull.add(new IntIntPair(besttri.c, 0));
        if (LOG.isDebuggingFinest()) {
            this.debugHull();
        }
        double[] center = besttri.m;
        for (i = start; i < len; ++i) {
            sortd[i] = SweepHullDelaunay2D.quadraticEuclidean(center, this.points.get(sorti[i]));
        }
        DoubleIntegerArrayQuickSort.sort(sortd, sorti, start, len);
        for (i = start; i < len; ++i) {
            int p;
            ListIterator<IntIntPair> iter2;
            int firsttri;
            int lasttri;
            int pointId = sorti[i];
            double[] newpoint = this.points.get(pointId);
            LinkedList<Triangle> newtris = hullonly ? null : new LinkedList<Triangle>();
            int hstart = -1;
            int hend = -1;
            Iterator<IntIntPair> iter3 = this.hull.descendingIterator();
            IntIntPair next = this.hull.getFirst();
            double[] nextV = this.points.get(next.first);
            int pos = this.hull.size() - 1;
            while (iter3.hasNext()) {
                Triangle tri;
                IntIntPair prev = iter3.next();
                double[] prevV = this.points.get(prev.first);
                if (hend < 0) {
                    if (this.leftOf(prevV, nextV, newpoint)) {
                        hstart = hend = pos;
                        if (!hullonly) {
                            tri = new Triangle(pointId, next.first, prev.first);
                            assert (tri.isClockwise(this.points));
                            assert (prev.second >= 0);
                            tri.updateCircumcircle(this.points);
                            tri.bc = prev.second;
                            newtris.addFirst(tri);
                        }
                    }
                } else {
                    if (!this.leftOf(prevV, nextV, newpoint)) break;
                    hstart = pos;
                    if (!hullonly) {
                        tri = new Triangle(pointId, next.first, prev.first);
                        assert (tri.isClockwise(this.points));
                        assert (prev.second >= 0);
                        tri.updateCircumcircle(this.points);
                        tri.bc = prev.second;
                        newtris.addFirst(tri);
                    }
                }
                next = prev;
                nextV = prevV;
                --pos;
            }
            if (hend == this.hull.size() - 1) {
                iter3 = this.hull.iterator();
                IntIntPair prev = iter3.next();
                double[] prevV = this.points.get(prev.first);
                while (iter3.hasNext()) {
                    IntIntPair next2 = iter3.next();
                    double[] nextV2 = this.points.get(next2.first);
                    if (!this.leftOf(prevV, nextV2, newpoint)) break;
                    ++hend;
                    if (!hullonly) {
                        Triangle tri = new Triangle(pointId, next2.first, prev.first);
                        assert (tri.isClockwise(this.points));
                        assert (prev.second >= 0);
                        tri.updateCircumcircle(this.points);
                        tri.bc = prev.second;
                        newtris.addLast(tri);
                    }
                    prev = next2;
                    prevV = nextV2;
                }
            }
            assert (hstart >= 0 && hend >= hstart);
            if (hullonly) {
                lasttri = -1;
                firsttri = -1;
            } else {
                int tristart;
                firsttri = tristart = this.tris.size();
                lasttri = tristart + newtris.size() - 1;
            }
            int hullsize = this.hull.size();
            if (LOG.isDebuggingFinest()) {
                LOG.debugFinest("Size: " + hullsize + " start: " + hstart + " end: " + hend);
            }
            if (hend < hullsize) {
                iter2 = this.hull.listIterator();
                for (p = 0; p <= hstart; ++p) {
                    iter2.next();
                }
                while (p <= hend) {
                    iter2.next();
                    iter2.remove();
                    ++p;
                }
                iter2.add(new IntIntPair(pointId, lasttri));
                iter2.previous();
                if (!hullonly) {
                    (iter2.hasPrevious() ? (IntIntPair)iter2.previous() : this.hull.getLast()).second = firsttri;
                }
            } else {
                iter2 = this.hull.listIterator();
                for (p = hullsize; p <= hend; ++p) {
                    iter2.next();
                    iter2.remove();
                }
                iter2.add(new IntIntPair(pointId, lasttri));
                p -= hullsize;
                IntIntPair pre = null;
                while (p <= hstart) {
                    pre = (IntIntPair)iter2.next();
                    ++p;
                }
                assert (pre != null);
                pre.second = firsttri;
                while (iter2.hasNext()) {
                    iter2.next();
                    iter2.remove();
                }
            }
            if (LOG.isDebuggingFinest()) {
                this.debugHull();
            }
            if (hullonly) continue;
            int tristart = this.tris.size();
            Iterator iter4 = newtris.iterator();
            int o = 0;
            while (iter4.hasNext()) {
                Triangle cur = (Triangle)iter4.next();
                cur.ca = o > 0 ? tristart + o - 1 : -1;
                int n = cur.ab = iter4.hasNext() ? tristart + o + 1 : -1;
                assert (cur.bc >= 0);
                Triangle other = this.tris.get(cur.bc);
                Orientation orient = cur.findOrientation(other);
                assert (orient != null) : "Inconsistent triangles: " + cur + " " + other;
                switch (orient) {
                    case ORIENT_BC_BA: {
                        assert (other.ab == -1) : "Inconsistent triangles: " + cur + " " + other;
                        other.ab = tristart + o;
                        break;
                    }
                    case ORIENT_BC_CB: {
                        assert (other.bc == -1) : "Inconsistent triangles: " + cur + " " + other;
                        other.bc = tristart + o;
                        break;
                    }
                    case ORIENT_BC_AC: {
                        assert (other.ca == -1) : "Inconsistent triangles: " + cur + " " + other;
                        other.ca = tristart + o;
                        break;
                    }
                    default: {
                        assert (cur.isClockwise(this.points));
                        assert (other.isClockwise(this.points));
                        throw new RuntimeException("Inconsistent triangles: " + cur + " " + other + " size:" + this.tris.size());
                    }
                }
                this.tris.add(cur);
                ++o;
            }
            assert (this.tris.size() == lasttri + 1);
        }
        if (!hullonly) {
            int size = this.tris.size();
            long[] flippedA = BitsUtil.zero(size);
            long[] flippedB = BitsUtil.zero(size);
            if (this.flipTriangles(flippedA) > 0) {
                for (int iterations = 1; iterations < 1000; iterations += 2) {
                    if (LOG.isDebuggingFinest()) {
                        this.debugHull();
                    }
                    if (this.flipTriangles(flippedA, flippedB) == 0) break;
                    if (LOG.isDebuggingFinest()) {
                        this.debugHull();
                    }
                    if (this.flipTriangles(flippedB, flippedA) == 0) break;
                }
            }
        }
    }

    public Triangle findSmallest(int seedid, int seed2id, double[] sortd, int[] sorti, int len) {
        int i;
        Triangle besttri = new Triangle(seedid, seed2id, -1);
        besttri.r2 = Double.MAX_VALUE;
        Triangle testtri = new Triangle(seedid, seed2id, -1);
        int besti = -1;
        for (i = 1; i < len; ++i) {
            testtri.c = sorti[i];
            if (!testtri.updateCircumcircle(this.points)) continue;
            assert (testtri.r2 > 0.0);
            if (testtri.r2 < besttri.r2) {
                besttri.copyFrom(testtri);
                besti = i;
                continue;
            }
            if (besttri.r2 * 4.0 < sortd[i]) break;
        }
        if (besti == -1) {
            this.hull.add(new IntIntPair(0, sorti[len - 1]));
            return null;
        }
        assert (besti >= 1);
        if (besti > 1) {
            i = sorti[besti];
            System.arraycopy(sorti, 1, sorti, 2, besti - 1);
            sorti[1] = i;
        }
        return besttri;
    }

    void debugHull() {
        StringBuilder buf = new StringBuilder(this.hull.size() * 20);
        for (IntIntPair p : this.hull) {
            buf.append(p.first).append(" (").append(p.second).append(") ");
        }
        LOG.debugFinest(buf);
    }

    int flipTriangles(long[] flippedB) {
        int size = this.tris.size();
        int numflips = 0;
        BitsUtil.zeroI(flippedB);
        for (int i = 0; i < size; ++i) {
            if (BitsUtil.get(flippedB, i) || this.flipTriangle(i, flippedB) < 0) continue;
            numflips += 2;
        }
        if (LOG.isDebuggingFinest()) {
            LOG.debugFinest("Flips: " + numflips);
        }
        return numflips;
    }

    int flipTriangles(long[] flippedA, long[] flippedB) {
        int numflips = 0;
        BitsUtil.zeroI(flippedB);
        int i = BitsUtil.nextSetBit(flippedA, 0);
        while (i > -1) {
            if (!BitsUtil.get(flippedB, i) && this.flipTriangle(i, flippedB) >= 0) {
                numflips += 2;
            }
            i = BitsUtil.nextSetBit(flippedA, i + 1);
        }
        if (LOG.isDebuggingFinest()) {
            LOG.debugFinest("Flips: " + numflips);
        }
        return numflips;
    }

    int flipTriangle(int i, long[] flipped) {
        int rig;
        int lef;
        int opp;
        Orientation orient;
        Triangle oth;
        int ot;
        Triangle cur = this.tris.get(i);
        if (cur.ab >= 0) {
            ot = cur.ab;
            oth = this.tris.get(ot);
            orient = cur.findOrientation(oth);
            switch (orient) {
                case ORIENT_AB_BA: {
                    opp = oth.c;
                    lef = oth.bc;
                    rig = oth.ca;
                    break;
                }
                case ORIENT_AB_CB: {
                    opp = oth.a;
                    lef = oth.ca;
                    rig = oth.ab;
                    break;
                }
                case ORIENT_AB_AC: {
                    opp = oth.b;
                    lef = oth.ab;
                    rig = oth.bc;
                    break;
                }
                default: {
                    throw new RuntimeException("Neighbor triangles not aligned?");
                }
            }
            if (cur.inCircle(this.points.get(opp))) {
                int a = cur.c;
                int b = cur.a;
                int c = opp;
                int d = cur.b;
                int ab = cur.ca;
                int bc = lef;
                int cd = rig;
                int da = cur.bc;
                int ca = ot;
                int ac = i;
                cur.set(a, ab, b, bc, c, ca);
                cur.updateCircumcircle(this.points);
                oth.set(c, cd, d, da, a, ac);
                oth.updateCircumcircle(this.points);
                if (bc >= 0) {
                    this.tris.get(bc).replaceEdge(c, b, ot, i);
                }
                if (da >= 0) {
                    this.tris.get(da).replaceEdge(a, d, i, ot);
                }
                BitsUtil.setI(flipped, i);
                BitsUtil.setI(flipped, ot);
                return ot;
            }
        }
        if (cur.bc >= 0) {
            ot = cur.bc;
            oth = this.tris.get(ot);
            orient = cur.findOrientation(oth);
            switch (orient) {
                case ORIENT_BC_BA: {
                    opp = oth.c;
                    lef = oth.bc;
                    rig = oth.ca;
                    break;
                }
                case ORIENT_BC_CB: {
                    opp = oth.a;
                    lef = oth.ca;
                    rig = oth.ab;
                    break;
                }
                case ORIENT_BC_AC: {
                    opp = oth.b;
                    lef = oth.ab;
                    rig = oth.bc;
                    break;
                }
                default: {
                    throw new RuntimeException("Neighbor triangles not aligned? " + (Object)((Object)orient));
                }
            }
            if (cur.inCircle(this.points.get(opp))) {
                int a = cur.a;
                int b = cur.b;
                int c = opp;
                int d = cur.c;
                int ab = cur.ab;
                int bc = lef;
                int cd = rig;
                int da = cur.ca;
                int ca = ot;
                int ac = i;
                cur.set(a, ab, b, bc, c, ca);
                cur.updateCircumcircle(this.points);
                oth.set(c, cd, d, da, a, ac);
                oth.updateCircumcircle(this.points);
                if (bc >= 0) {
                    this.tris.get(bc).replaceEdge(c, b, ot, i);
                }
                if (da >= 0) {
                    this.tris.get(da).replaceEdge(a, d, i, ot);
                }
                BitsUtil.setI(flipped, i);
                BitsUtil.setI(flipped, ot);
                return ot;
            }
        }
        if (cur.ca >= 0) {
            ot = cur.ca;
            oth = this.tris.get(ot);
            orient = cur.findOrientation(oth);
            switch (orient) {
                case ORIENT_CA_BA: {
                    opp = oth.c;
                    lef = oth.bc;
                    rig = oth.ca;
                    break;
                }
                case ORIENT_CA_CB: {
                    opp = oth.a;
                    lef = oth.ca;
                    rig = oth.ab;
                    break;
                }
                case ORIENT_CA_AC: {
                    opp = oth.b;
                    lef = oth.ab;
                    rig = oth.bc;
                    break;
                }
                default: {
                    throw new RuntimeException("Neighbor triangles not aligned?");
                }
            }
            if (cur.inCircle(this.points.get(opp))) {
                int a = cur.b;
                int b = cur.c;
                int c = opp;
                int d = cur.a;
                int ab = cur.bc;
                int bc = lef;
                int cd = rig;
                int da = cur.ab;
                int ca = ot;
                int ac = i;
                cur.set(a, ab, b, bc, c, ca);
                cur.updateCircumcircle(this.points);
                oth.set(c, cd, d, da, a, ac);
                oth.updateCircumcircle(this.points);
                if (bc >= 0) {
                    this.tris.get(bc).replaceEdge(c, b, ot, i);
                }
                if (da >= 0) {
                    this.tris.get(da).replaceEdge(a, d, i, ot);
                }
                BitsUtil.setI(flipped, i);
                BitsUtil.setI(flipped, ot);
                return ot;
            }
        }
        return -1;
    }

    public Polygon getHull() {
        if (this.hull == null) {
            this.run(true);
        }
        DoubleMinMax minmaxX = new DoubleMinMax();
        DoubleMinMax minmaxY = new DoubleMinMax();
        ArrayList<double[]> hullp = new ArrayList<double[]>(this.hull.size());
        for (IntIntPair pair : this.hull) {
            double[] v = this.points.get(pair.first);
            hullp.add(v);
            minmaxX.put(v[0]);
            minmaxY.put(v[1]);
        }
        return new Polygon(hullp, minmaxX.getMin(), minmaxX.getMax(), minmaxY.getMin(), minmaxY.getMax());
    }

    public ArrayList<Triangle> getDelaunay() {
        if (this.tris == null) {
            this.run(false);
        }
        return this.tris;
    }

    public static double quadraticEuclidean(double[] v1, double[] v2) {
        double d1 = v1[0] - v2[0];
        double d2 = v1[1] - v2[1];
        return d1 * d1 + d2 * d2;
    }

    boolean leftOf(double[] a, double[] b, double[] d) {
        double bax = b[0] - a[0];
        double day = d[1] - a[1];
        double bay = b[1] - a[1];
        double dax = d[0] - a[0];
        double cross = bax * day - bay * dax;
        return cross > 1.0E-10 * Math.max(Math.max(bax, bay), Math.max(dax, day));
    }

    public static class Triangle {
        public int a;
        public int b;
        public int c;
        public int ab = -1;
        public int ca = -1;
        public int bc = -1;
        public double r2 = -1.0;
        public double[] m = new double[2];

        public Triangle(int x, int y, int z) {
            this.a = x;
            this.b = y;
            this.c = z;
        }

        void replaceEdge(int a, int b, int ol, int ne) {
            if (this.a == a && this.b == b) {
                assert (this.ab == ol) : "Edge doesn't match: " + this + " " + a + " " + b + " " + ol + " " + ne;
                this.ab = ne;
                return;
            }
            if (this.b == a && this.c == b) {
                assert (this.bc == ol) : "Edge doesn't match: " + this + " " + a + " " + b + " " + ol + " " + ne;
                this.bc = ne;
                return;
            }
            if (this.c == a && this.a == b) {
                assert (this.ca == ol) : "Edge doesn't match: " + this + " " + a + " " + b + " " + ol + " " + ne;
                this.ca = ne;
                return;
            }
        }

        void set(int a, int ab, int b, int bc, int c, int ca) {
            assert (ab != bc || ab == -1);
            assert (ab != ca || ab == -1);
            assert (bc != ca || bc == -1);
            this.a = a;
            this.ab = ab;
            this.b = b;
            this.bc = bc;
            this.c = c;
            this.ca = ca;
        }

        public boolean inCircle(double[] opp) {
            double dx = opp[0] - this.m[0];
            double dy = opp[1] - this.m[1];
            return dx * dx + dy * dy < this.r2;
        }

        Orientation findOrientation(Triangle oth) {
            if (this.a == oth.a) {
                if (this.b == oth.c) {
                    return Orientation.ORIENT_AB_AC;
                }
                if (this.c == oth.b) {
                    return Orientation.ORIENT_CA_BA;
                }
            }
            if (this.a == oth.b) {
                if (this.b == oth.a) {
                    return Orientation.ORIENT_AB_BA;
                }
                if (this.c == oth.c) {
                    return Orientation.ORIENT_CA_CB;
                }
            }
            if (this.a == oth.c) {
                if (this.b == oth.b) {
                    return Orientation.ORIENT_AB_CB;
                }
                if (this.c == oth.a) {
                    return Orientation.ORIENT_CA_AC;
                }
            }
            if (this.b == oth.b && this.c == oth.a) {
                return Orientation.ORIENT_BC_BA;
            }
            if (this.b == oth.c && this.c == oth.b) {
                return Orientation.ORIENT_BC_CB;
            }
            if (this.b == oth.a && this.c == oth.c) {
                return Orientation.ORIENT_BC_AC;
            }
            return null;
        }

        void makeClockwise(List<double[]> points) {
            if (!this.isClockwise(points)) {
                int t = this.b;
                this.b = this.c;
                this.c = t;
                t = this.ab;
                this.ab = this.ca;
                this.ca = t;
            }
        }

        boolean isClockwise(List<double[]> points) {
            double mX;
            double max;
            double aby;
            double[] pc;
            double mY;
            double may;
            double[] pa = points.get(this.a);
            double[] pb = points.get(this.b);
            double abx = pb[0] - pa[0];
            return -abx * (may = pa[1] - (mY = (pa[1] + pb[1] + (pc = points.get(this.c))[1]) * 0.3333333333333333)) + (aby = pb[1] - pa[1]) * (max = pa[0] - (mX = (pa[0] + pb[0] + pc[0]) * 0.3333333333333333)) <= 0.0;
        }

        void copyFrom(Triangle o) {
            this.a = o.a;
            this.b = o.b;
            this.c = o.c;
            this.r2 = o.r2;
            this.m[0] = o.m[0];
            this.m[1] = o.m[1];
        }

        private boolean updateCircumcircle(List<double[]> points) {
            double bcx;
            double aby;
            double[] pc;
            double bcy;
            double[] pa = points.get(this.a);
            double[] pb = points.get(this.b);
            double abx = pb[0] - pa[0];
            double D = abx * (bcy = (pc = points.get(this.c))[1] - pb[1]) - (aby = pb[1] - pa[1]) * (bcx = pc[0] - pb[0]);
            if (D == 0.0) {
                this.m[0] = Double.NaN;
                this.m[1] = Double.NaN;
                this.r2 = Double.NaN;
                return false;
            }
            double mabx = (pa[0] + pb[0]) * 0.5;
            double maby = (pa[1] + pb[1]) * 0.5;
            double mbcx = (pb[0] + pc[0]) * 0.5;
            double mbcy = (pb[1] + pc[1]) * 0.5;
            double beta = (abx * (mbcx - mabx) + aby * (mbcy - maby)) / D;
            this.m[0] = mbcx - bcy * beta;
            this.m[1] = mbcy + bcx * beta;
            double rx = pa[0] - this.m[0];
            double ry = pa[1] - this.m[1];
            this.r2 = rx * rx + ry * ry;
            return true;
        }

        public String toString() {
            return "Triangle [a=" + this.a + ", b=" + this.b + ", c=" + this.c + ", ab=" + this.ab + ", bc=" + this.bc + ", ca=" + this.ca + "]";
        }
    }

    private static enum Orientation {
        ORIENT_AB_BA,
        ORIENT_AB_CB,
        ORIENT_AB_AC,
        ORIENT_BC_BA,
        ORIENT_BC_CB,
        ORIENT_BC_AC,
        ORIENT_CA_BA,
        ORIENT_CA_CB,
        ORIENT_CA_AC;

    }
}

