library("mcmcse")
library("tidyverse")
library("ggplot2")

reps <- letters[1:10] ## labeled a through j

readResultsOneRep <- function(rep_label) {
  MCMCresults <- list()
  files <- Sys.glob(paste0("SVexample_output_rep_", rep_label, "*.RDS"))
  for (iFile in seq_along(files)) {
    resultsfile <- files[iFile]
    MCMCresults[[iFile]] <- readRDS(resultsfile)
  }
  MCMCresults
}

processResultsOneRep <- function(MCMCresults, rep_label, testing = FALSE) {
  # If testing is TRUE, we use a faster ess calculation
  nCases <- length(MCMCresults)
  params <- c("mu", "phi", "sigma")
  nParams <- length(params)
  nPoints <- nCases * nParams
  ess <- numeric(nPoints)
  time <- numeric(nPoints)
  N <- numeric(nPoints)
  h <- numeric(nPoints)
  Param <- character(nPoints)
  iCase <- 1
  
  require("mcmcse")
  iResult <- 1
  for(iCase in 1:nCases) {
    for(iParam in 1:nParams) {
      ess[iResult] <-
        if (isTRUE(testing))
          ess(MCMCresults[[iCase]]$samples[, params[iParam]])
      else
        ess(MCMCresults[[iCase]]$samples[, params[iParam]],
            method = "obm", size = 500)
      time[iResult] <- sum(MCMCresults[[iCase]]$runtime[1:2])
      N[iResult] <- MCMCresults[[iCase]]$N
      h[iResult] <- MCMCresults[[iCase]]$covSDs[1] / 0.089
      Param[iResult] <- params[iParam]
      iResult <- iResult + 1
    }
  }
  
  results <- data.frame(Param = Param,
                        time = time,
                        N = N,
                        h = factor(h),
                        ess = ess,
                        efficiency = ess / time,
                        rep = rep(rep_label, length(Param)))
  results
}

allMCMCresults <- lapply(reps, readResultsOneRep)
## This step can be slow due to obm method of mcmcse::ess
allProcessedResults <- mapply(processResultsOneRep,
                              MCMCresults = allMCMCresults,
                              rep_label = reps,
                              SIMPLIFY = FALSE)
## This is a useful step to save for later use, because the ess
## calculations in processResultsOneRep are slow
## save(allProcessedResults, file = "SVexample_allProcessedResults.RData")
comboProcessedResults <- do.call("rbind", allProcessedResults)

comboMeanResults <- group_by(comboProcessedResults, Param, N, h) %>%
  gather(key = "metric", value = "value", ess, efficiency) %>%
  mutate(metric = factor(metric, levels = c("ess", "efficiency"))) %>%
  group_by(Param, N, h, metric) %>%
  summarize(time = mean(time), se = sd(value)/sqrt(length(value)),
            mean = mean(value))

PMCMCplot <-  ggplot(aes(x = N, y = mean, group = h),
                     data = comboMeanResults) + geom_line(aes(color = h)) +
  geom_point(aes(color = h)) +
  facet_grid(rows = vars(metric), cols = vars(Param),
             scales = "free_y", switch = "y",
             labeller = labeller(metric = c("ess" = "Effective sample size", "efficiency" = "Efficiency")))

PMCMCplot <- PMCMCplot +
  geom_linerange(aes(ymin = mean - se, ymax = mean + se, color = h),
                 position = position_dodge(width = 5)) +
  theme_minimal() + xlab("Number of particles, M") + ylab("")

## View figure directly
PMCMCplot

## Save figure
# ggsave(plot = PMCMCplot, filename = "./../../Figures/PMCMCplot.pdf")
