set.seed(1)

## 
library("nimble")
library("nimbleSMC")
exampleCode <- nimbleCode({
  x0 ~ dnorm(0, var = 1)
  x[1] ~ dnorm(.8 * x0, var = 1)
  y[1] ~ dnorm(x[1], var =  0.5)
  for (t in 2:10) {
    x[t] ~ dnorm(.8 * x[t - 1], var = 1)
    y[t] ~ dnorm(x[t], var = 0.5)
  }
})


## 
simulatedData <- c(-0.9,  1.6,  0.6,  1.3,  1.5, 0.3, -0.8, -1.3,  0.5,  1.1)
exampleModel <- nimbleModel(code = exampleCode,
  data = list(y = simulatedData), inits = list(x0 = 0))

##
exampleModel$y[3]


## 
exampleModel$simulate(nodes = exampleModel$getNodeNames(includeData = FALSE))
exampleModel$x


## 
exampleModel$calculate(nodes = "y")


## 
C_exampleModel <- compileNimble(exampleModel)


## 
exampleBootstrapFilter <- buildBootstrapFilter(exampleModel, nodes = "x",
  control = list(saveAll = TRUE, thresh = .9))


##
exampleBootstrapFilter$run(100)

## 
CexampleBootstrapFilter <- compileNimble(exampleBootstrapFilter,
  project = exampleModel)
CexampleBootstrapFilter$run(10000)
bootstrapFilterSamples <- as.matrix(CexampleBootstrapFilter$mvEWSamples)


##
exampleAuxiliaryFilter <- buildAuxiliaryFilter(exampleModel, nodes = "x",
  control = list(saveAll = TRUE, lookahead = "mean"))
CexampleAuxiliaryFilter <- compileNimble(exampleAuxiliaryFilter,
  project = exampleModel, resetFunctions = TRUE)
CexampleAuxiliaryFilter$run(10000)
auxiliaryFilterSamples <- as.matrix(CexampleAuxiliaryFilter$mvEWSamples)


## 
exampleEnsembleKF <- buildEnsembleKF(exampleModel, nodes = "x",
                                     control = list(saveAll = TRUE))
CexampleEnsembleKF <- compileNimble(exampleEnsembleKF,
                                    project = exampleModel, resetFunctions = TRUE)
CexampleEnsembleKF$run(10000)
EnKFSamples <- as.matrix(CexampleEnsembleKF$mvSamples)


## 
library("dlm")
exampleDlm <- dlm(FF = 1, V = 0.5, GG = .8, W = 1, m0 = 0, C0 = 1)
kalmanF <- dlmFilter(y = exampleModel$y, mod = exampleDlm)
var <- unlist(dlmSvd2var(u = kalmanF$U.C, d = kalmanF$D.C))[-1]
kalmanQuants <- matrix(c(kalmanF$m[-1] - 1.96 * sqrt(var), kalmanF$m[-1],
                         kalmanF$m[-1] + 1.96 * sqrt(var)), nrow = 3, byrow = TRUE)
bootstrapQuants <- apply(bootstrapFilterSamples, MARGIN = 2,
                         FUN = quantile, probs = c(.025, 0.5, .975))[, 1:10]
auxiliaryQuants <- apply(auxiliaryFilterSamples, MARGIN = 2,
                         FUN = quantile, probs = c(.025, 0.5, .975))[, 1:10]
EnKFQuants <- apply(EnKFSamples, MARGIN = 2,
                    FUN = quantile, probs = c(.025, 0.5, .975))[,1:10]
matplot(y = t(kalmanQuants), x = 1:10, type = "l", col = 1, lty = 1,
        xlab = "Time", ylim = c(-2,4.2), ylab = "")
matlines(y=cbind(t(bootstrapQuants), t(auxiliaryQuants), t(EnKFQuants)),
         type = "l", col = 1, lty = c(2, 2, 2, 5, 5, 5, 5, 4, 4))
legend("topleft", lty = c(1, 2, 5, 4), pch = c(NA, NA, NA, NA), bty = "n",
       legend = c("Kalman quantiles", "Bootstrap quantiles", "Auxiliary quantiles", "EnKF quantiles"))


## 
stochVCode <- nimbleCode({
  x[1] ~ dnorm(0, sd = sigma / sqrt(1 - phi * phi))
  y[1] ~ dnorm(0, sd = beta * exp(0.5 * x[1]))
  for (t in 2:T) {
    x[t] ~ dnorm(phi * x[t - 1], sd = sigma)
    y[t] ~ dnorm(0, sd = beta * exp(0.5 * x[t]))
  }
  phi <- 2 * phiStar - 1
  phiStar ~ dbeta(20, 1.1)
  logsigma2 ~ dgammalog(shape = 0.5, rate = 1 / (2 * 0.1)) ## This is Omega
  sigma <- exp(0.5 * logsigma2)
  mu ~ dnorm(-10, sd = 1) ## It matters whether data are converted to % or not.
  beta <- exp(0.5 * mu)
})


## 
dgammalog <- nimbleFunction(
  run = function(x = double(), shape = double(),
                 rate = double(),log = integer(0, default = 0)) {
    logProb <- shape * log(rate) + shape * x - rate * exp(x) - lgamma(shape)
    if (log) return(logProb)
    else return(exp(logProb))
    returnType(double())
  }
)

rgammalog <- nimbleFunction(
  run = function(n = integer(), shape = double(), rate = double()) {
    xg <- rgamma(1, shape = shape, rate = rate)
    return(log(xg))
    returnType(double())
  }
)


## 
library("stochvol")
data("exrates", package = "stochvol")
y <- logret(exrates$USD[exrates$date > "2010-01-01"], demean = TRUE)
T <- length(y)


## 
stochVolModel <- nimbleModel(code = stochVCode, constants = list(T = T),
  data = list(y = y), inits = list(mu = -10, phiStar = .99, logsigma2 = log(.004)))
CstochVolModel <- compileNimble(stochVolModel)


## 
stochVolMCMCConf <- configureMCMC(stochVolModel, nodes = NULL,
  monitors = c("mu", "beta", "phiStar", "phi", "logsigma2", "sigma"))
auxpf <- buildAuxiliaryFilter(stochVolModel, "x",
  control = list(saveAll = FALSE, smoothing = FALSE, initModel = FALSE))
h <- 1
propSD <- h * c(0.089, 0.039, 1.45)
m <- 100
stochVolMCMCConf$addSampler(target = c("mu", "phiStar", "logsigma2"),
  type = "RW_PF_block", control = list(propCov = diag(propSD^2),
    pf = auxpf, adaptive = FALSE, pfNparticles = m, latents = "x"))


## 
stochVolMCMC <- buildMCMC(stochVolMCMCConf)
cMCMC <- compileNimble(stochVolMCMC, project = stochVolModel,
  resetFunctions = TRUE)
cMCMC$run(50000) ## Or see nimble::runMCMC to manage MCMC runs
samples <- as.matrix(cMCMC$mvSamples)

## Code to recreate figure 2 found in code/PMCMCexample/SVexample_make_traceplot.R

nileCode <- nimbleCode({
  for (t in 1:n)
    y[t] ~ dnorm(x[t], sd = sigmaMeasurements)
  x[1] ~ dnorm(x0, sd = sigmaInnovations)
  for (t in 2:n)
    x[t] ~ dnorm((t - 1 == 28) * meanShift1899 + x[t - 1],
                 sd = sigmaInnovations)
  logSigmaInnovations ~ dnorm(0, sd = 100)       ## Prior is not used by IF2
  logSigmaMeasurements ~ dnorm(0, sd = 100)      ## Prior is not used by IF2
  sigmaInnovations <- exp(logSigmaInnovations)
  sigmaMeasurements <- exp(logSigmaMeasurements)
  x0 ~ dnorm(1120, var = 100)                    ## Prior is not used by IF2
  meanShift1899 ~ dnorm(0, sd = 100)             ## Prior is not used by IF2
})


##
y <- Nile
nileModel <- nimbleModel(nileCode, data = list(y = y),
  constants = list(n = length(y)),
  inits = list(logSigmaInnovations = log(sd(y)),
    logSigmaMeasurements = log(sd(y)),
    meanShift1899 = -100))

perturbThetaSD <- c(0.1, 0.1, 5)
initParamSigma <- c(0.1, 0.1, 5)

ff <- buildIteratedFilter2(model = nileModel, nodes = "x",
  params = c("logSigmaInnovations", "logSigmaMeasurements",
    "meanShift1899"), baselineNode = "x0",
  control = list(sigma = perturbThetaSD, initParamSigma = initParamSigma))
cNileModel <- compileNimble(nileModel)
cff <- compileNimble(ff, project = nileModel)


## 
numParticles <- 1000
numPFruns <- 100
alpha <- 0.2

est <- cff$run(m = numParticles, niter = numPFruns, alpha = alpha)


## 
bootstrapFilter <- nimbleFunction(setup = function(model, latentNodes) {
  my_initializeModel <- initializeModel(model)
  latentNodes <- model$expandNodeNames(latentNodes, sort = TRUE)
  dims <- lapply(latentNodes, function(n) nimDim(model[[n]]))
  mvWSpec <- modelValuesConf(vars = c("x", "wts"), types = c("double", "double"),
                             sizes = list(x = dims[[1]], wts = 1))
  mvWSamples <- modelValues(mvWSpec)
  mvEWSpec <- modelValuesConf(vars = c("x"), types = c("double"), sizes = list(x = dims[[1]]))
  mvEWSamples <- modelValues(mvEWSpec)
  bootStepFunctions <- nimbleFunctionList(bootstrapStepVirtual)
  timePoints <- length(latentNodes)
  for (t in 1:timePoints)
    bootStepFunctions[[t]] <- bootstrapStep(model, mvWSamples, mvEWSamples, latentNodes, t)
  },
  run = function(M = integer()) {
    my_initializeModel$run()
    resize(mvWSamples, M)
    resize(mvEWSamples, M)
    for (t in 1:timePoints)
      bootStepFunctions[[t]]$run(M)
})


## 
bootstrapStepVirtual <- nimbleFunctionVirtual(
  run = function(M = integer()) {}
)


## 
bootstrapStep <- nimbleFunction(contains = bootstrapStepVirtual,
  setup = function(model, mvWSamples, mvEWSamples, latentNodes, timePoint) {
    notFirst <- timePoint != 1
    prevNode <- latentNodes[if (notFirst)
      timePoint - 1 else timePoint]
    thisNode <- latentNodes[timePoint]
    prevDeterm <- model$getDependencies(prevNode, determOnly = TRUE)
    thisDeterm <- model$getDependencies(thisNode, determOnly = TRUE)
    thisData <- model$getDependencies(thisNode, dataOnly = TRUE)
    },
  run = function(M = integer()) {
  ids <- integer(M, 0)
  wts <- numeric(M, 0)
  for (m in 1:M) {
    if (notFirst) {
      copy(from = mvEWSamples, to = model, nodes = "x", nodesTo = prevNode,
           row = m)
      model$calculate(prevDeterm)
    }
    model$simulate(thisNode)
    copy(from = model, to = mvWSamples, nodes = thisNode, nodesTo = "x", row = m)
    model$calculate(thisDeterm)
    wts[m] <- exp(model$calculate(thisData))
    mvWSamples["wts", m][1] <<- wts[m]
  }
  rankSample(wts, M, ids)
  for (m in 1:M) {
    copy(from = mvWSamples, to = mvEWSamples, nodes = "x", nodesTo = "x", row = ids[m],
         rowTo = m)
  }
})

##
myBootstrap <- bootstrapFilter(exampleModel, "x")
cmyBootstrap <- compileNimble(myBootstrap, project = exampleModel,
  resetFunctions = TRUE)
cmyBootstrap$run(1000)
filterSamps <- as.matrix(cmyBootstrap$mvEWSamples, "x")
hist(filterSamps, main = "", xlab = "")
