library("nimble")
library("stochvol")
library("nimbleSMC")
set.seed(seed)
stochVCode <- nimbleCode({
  x[1] ~ dnorm(0, sd = sigma / sqrt(1 - phi * phi))
  y[1] ~ dnorm(0, sd = beta * exp(0.5 * x[1]))
  for (t in 2:T) {
    x[t] ~ dnorm(phi * x[t - 1], sd = sigma)
    y[t] ~ dnorm(0, sd = beta * exp(0.5 * x[t]))
  }
  phi <- 2 * phiStar - 1
  phiStar ~ dbeta(20, 1.1)
  logsigma2 ~ dgammalog(shape = 0.5, rate = 1 / (2 * 0.1))
  sigma <- exp(0.5 * logsigma2)
  mu ~ dnorm(-10, sd = 1) ## It matters whether data are converted to % or not.
  beta <- exp(0.5 * mu)
})

dgammalog <- nimbleFunction(
  run = function(x = double(),
                 shape = double(),
                 rate = double(),
                 log = integer(0, default = 0)) {
    logProb <- shape * log(rate) + shape * x - rate * exp(x) - lgamma(shape)
    if (log)
      return(logProb)
    else
      return(exp(logProb))
    returnType(double())
  }
)

rgammalog <- nimbleFunction(
  run = function(n = integer(),
                 shape = double(),
                 rate = double()) {
    xg <- rgamma(1, shape = shape, rate = rate)
    return(log(xg))
    returnType(double())
  }
)

data("exrates", package = "stochvol")
y <- logret(exrates$USD[exrates$date > "2010-01-01"], demean = TRUE)
T <- length(y)

stochVolModel <- nimbleModel(code = stochVCode,
                             constants = list(T = T), data = list(y = y),
                             inits = list(mu = -10, phiStar = .99,
                                          logsigma2 = log(.004)))
CstochVolModel <- compileNimble(stochVolModel)

stochVolMCMCConf <- configureMCMC(stochVolModel, nodes = NULL,
                                  monitors = c("mu", "beta",
                                               "phiStar", "phi",
                                               "logsigma2", "sigma"))

auxpf <- buildAuxiliaryFilter(stochVolModel, "x",
                              control = list(saveAll = FALSE,
                                             smoothing = FALSE,
                                             initModel = FALSE))

## Fill in default settings
if (!exists("N")) N <- 1000 ## SMC particles
if (!exists("m")) m <- 1000 ## MCMC samples
if (!exists("covSDs")) covSDs <- 0.5 * c(0.089, 0.039, 1.45)
stochVolMCMCConf$addSampler(target = c("mu", "phiStar", "logsigma2"), type = "RW_PF_block",
                            control = list(propCov = diag(covSDs^2),
                                           pf = auxpf,
                                           adaptive = FALSE,
                                           pfNparticles = N,
                                           latents = "x",
                                           pfResample = FALSE))

stochVolMCMC <- buildMCMC(stochVolMCMCConf)
cMCMC <- compileNimble(stochVolMCMC, project = stochVolModel)

runtime <- system.time(cMCMC$run(m))
samples <- as.matrix(cMCMC$mvSamples)
