###################################################################################################################
# Replication materials of "pexm: a JAGS module for applications involving the piecewise exponential distribution".
# Authors: Vinicius D. Mayrink, Joao D. N. Duarte and Fabio N. Demarqui.
# Departamento de Estatistica, ICEx, Universidade Federal de Minas Gerais.
###################################################################################################################
# Commands to reproduce results in the Section 3 of the paper
###################################################################################################################

# Important remarks: 
#   (1) If JAGS models are written in a separate .R file, make sure that the R working directory is the same one where this R script is located.
#       This is important for the correct identification of the files containing the Bayesian model written in the JAGS syntax.
#   (2) The flag below is set as FALSE to avoid the execution of a Monte Carlo (MC) scheme shown in Section 3.3 of the paper.
#       The MC simulation with nMC = 100 replications will take hours to run. 
#       For faster inspection, the final results (folder 'MCresults') were provided together with this replication material.
#       If you want to run the MC scheme, please set MCflag = TRUE and specify the number of MC replications in the object nMC.
#       In the paper, we set nMC = 100. You may choose a smaller value (say nMC = 2) for a fast inspection of the MC simulation code.

MCflag <- FALSE # Flag indicating: TRUE = run the MC scheme or FALSE = do not run the MC scheme.

# If you choose to run the MC scheme, the results in the folder 'MCresults' will be ignored.    
if(MCflag == TRUE){ 
  nMC <- 2 # You may change the value of nMC here.
} 
# If you choose not to run the MC scheme, the results provided in the folder 'MCresults' (assuming nMC = 100) will be used for analysis. 
if(MCflag == FALSE){ 
  nMC <- 100 
} # Do not change the value of nMC here.

################################

set.seed(1981)

# Load some required R libraries.
if(!require("plotrix")){ install.packages("plotrix"); library("plotrix")}
if(!require("msm")){ install.packages("msm"); library("msm")}
if(!require("rjags")){ install.packages("rjags"); library("rjags")}
library("pexm")
loadpexm()

# Generate the data:
lambda <- c(0.3, 0.6, 0.8, 1.3) # Real values.

# Set the fixed time grid vector.
# Requirements: 
#   - 1st element must be 0, 
#   - values must be in ascending order.
#   - length(tau) must match length(lambda).
tau <- c(0.0, 2.0, 3.0, 5.0)
m <- length(tau)

# Generate observations from the Piecewise Exponential model (via R package "msm").
n <- 1000
t <- rpexp(n, lambda, tau)

# Probabilities related to target quantiles
pq <- c(1, 2, seq(5, 95, 5), 98, 99) / 100
nq <- length(pq) 

# Code chunk CC.1 (see the paper):
data <- list(t = t, n = n, tau = tau, m = m, pq = pq, nq = nq)
parameters <- c("lambda", "ht100", "Ht100", paste0("q[",1:nq,"]"), 
                paste0("loglik[",1:n,"]"), paste0("St[",1:n,"]"))
inits1 <- list( lambda = c(0.1, 0.5, 1, 2), 
                .RNG.name = "base::Super-Duper", .RNG.seed = 1 ) 
inits2 <- list( lambda = c(0.5, 1.0, 1.5, 2.5), 
                .RNG.name = "base::Wichmann-Hill", .RNG.seed = 2 )
# MCMC setup
burnin <- 1000
lag <- 1
npost <- 2000

# Model 
Model_pex <- "model{
  for (i in 1:n) {
    t[i] ~ dpex(lambda[], tau[])
    St[i] <- 1 - ppex(t[i], lambda[], tau[])
    loglik[i] <- log(dpex(t[i], lambda[], tau[]))
  }
  #
  Ht100 <- hcpex(t[100], lambda[], tau[])
  ht100 <- hpex(t[100], lambda[], tau[])
  #
  for (j in 1:m) { lambda[j] ~ dgamma(0.01, 0.01) }
  for (k in 1:nq) { q[k] <- qpex(pq[k], lambda[], tau[]) }  
}
"

# Compile the model.
Mjags <- rjags::jags.model(textConnection(Model_pex), data = data,  
                           inits = list(inits1, inits2), 
                           n.chains = 2, n.adapt = burnin)
# Run JAGS.
output <- rjags::coda.samples(Mjags, variable.names = parameters, 
                              n.iter = npost, thin = lag)

# Number of observations in each interval defined by the time grid in tau.
# This is discussed in Section 3.3.
nI1 = sum(t <= tau[2])
nI2 = sum(t > tau[2] & t <= tau[3])
nI3 = sum(t > tau[3] & t <= tau[4])
nI4 = sum(t > tau[4])
c(nI1, nI2, nI3, nI4)

# Check the values of hazard and cumulative hazard functions for t[100].
# This result is discussed in a paragraph near Figure 1.
labels <- c("ht100", "Ht100")
hazjags <- summary(output[, labels])$statistics[,"Mean"]
round(hazjags, 3)

# True value of ht100 and Ht100. 
ht100_true = lambda[3]
Ht100_true = lambda[3] * (t[100] - tau[3]) + lambda[1] * (tau[2] - tau[1]) + lambda[2] * (tau[3] - tau[2])
round(c(ht100_true, Ht100_true), 3)


################################
# Code to replicate Figure 1 (a)
################################

# log-likelihood from 'jags' and package 'msm'.
labels <- paste0("loglik[",1:n,"]")
aux = as.matrix(output[, labels])
loglikjags <- apply(aux, 2, mean)
loglikmsm <- log(dpexp(t, lambda, tau))
hpd <- HPDinterval(coda::as.mcmc(aux))
hpdll1 <- hpd[,1] 
hpdll2 <- hpd[,2]

# S(t) from 'jags' and package 'msm'.
labels <- paste0("St[",1:n,"]")
aux = as.matrix(output[, labels])
Stjags <- apply(aux, 2, mean)
Stmsm <- 1-ppexp(t, lambda, tau)
hpd <- HPDinterval(coda::as.mcmc(aux))
hpdst1 <- hpd[,1]
hpdst2 <- hpd[,2]

# Quantiles from 'jags' and package 'msm'.
labels <- paste0("q[",1:nq,"]")
aux = as.matrix(output[, labels])
qjags <- apply(aux, 2, mean)
qmsm <- qpexp(pq, lambda, tau)
hpd <- HPDinterval(coda::as.mcmc(aux))
hpdqt1 <- hpd[,1]
hpdqt2 <- hpd[,2]

# Figure 1 (a)
par(mfrow=c(1,3))
plotCI(t, loglikjags, li = hpdll1, ui = hpdll2, pch = 19, cex = 1.5, 
       col = "black", cex.axis = 2, cex.lab = 2, xlab = "time", ylab = "log-density")
points(t, loglikmsm, pch = 19, cex = 1.5, col = "gray60")
plotCI(t, Stjags, li = hpdst1, ui = hpdst2, pch = 19,cex = 1.5,
       col = "black", cex.axis = 2, cex.lab = 2, xlab = "time", ylab = "survival function")
points(t, Stmsm, pch = 19, cex = 1.5, col = "gray60")
plotCI(pq, qjags, li = hpdqt1, ui = hpdqt2, pch = 19, cex = 1.5, 
       col = "black", cex.axis = 2, cex.lab = 2, xlab = "probability", ylab = "quantile")
points(pq, qmsm, pch = 19, cex = 1.5, col = "gray60")


#################################
# Code to replicate Figure 1 (b) 
#################################

par(mfrow = c(2, 2))
co1 <- "black"; co2 = "tomato"; co3 = "royalblue"; 
aux1 = as.matrix(output[, paste0("lambda[",1:4,"]")][1])
aux2 = as.matrix(output[, paste0("lambda[",1:4,"]")][2])
plot(aux1[,1], ylim = c(lambda[1]-0.06, lambda[1]+0.06), ylab = expression(lambda[1]), 
     xlab = "iterations", cex.axis = 2, cex.lab = 2, type = "l", col = co1, lwd = 1.5)
lines(aux2[,1], type = "l", col = co2, lwd = 1.5)
abline(h = lambda[1], col = co3, lwd = 5)
plot(aux1[,2], ylim = c(lambda[2]-0.15, lambda[2]+0.15), ylab = expression(lambda[2]),
     xlab = "iterations", cex.axis = 2, cex.lab = 2, type = "l", col = co1, lwd = 1.5)
lines(aux2[,2], type = "l", col = co2, lwd = 1.5)
abline(h = lambda[2], col = co3, lwd = 5)
plot(aux1[,3], ylim = c(lambda[3]-0.2, lambda[3]+0.2), ylab = expression(lambda[3]), 
     xlab = "iterations", cex.axis = 2, cex.lab = 2, type = "l", col = co1, lwd = 1.5)
lines(aux2[,3], type = "l", col = co2, lwd = 1.5)
abline(h = lambda[3], col = co3, lwd = 5)
plot(aux1[,4], ylim = c(lambda[4]-0.7, lambda[4]+0.7), ylab = expression(lambda[4]), 
     xlab = "iterations", cex.axis = 2, cex.lab = 2, type = "l", col = co1, lwd = 1.5)
lines(aux2[,4], type = "l", col = co2, lwd = 1.5)
abline(h = lambda[4], col = co3, lwd = 5)


# Table with some descriptive results related to Figure 1 (b).
# These results are discussed in a paragraph near Figure 1.
tab = array(0,c(4, 6))
aux = as.matrix(output[, paste0("lambda[",1:4,"]")])
hpd = HPDinterval(coda::as.mcmc(aux))
tab[,1] = lambda
tab[,2] = round(apply(aux, 2, mean), 3)
tab[,3] = round(apply(aux, 2, median), 3)
tab[,4] = round(apply(aux, 2, sd), 3)
tab[,5] = round(hpd[,1], 3)
tab[,6] = round(hpd[,2], 3)
colnames(tab) = c("real", "mean", "median", "s.d.", "HPD inf.", "HPD sup.")
rownames(tab) = c("lambda[1]", "lambda[2]", "lambda[3]", "lambda[4]")
tab


##############################################################
# Monte Carlo (MC) simulation study with artificial data sets.
# Section 3.3 of the paper.
##############################################################

# Important remarks: 
#  - Make sure that the working directory is the same one where this R script is located.
#    This is important for the correct identification of the files containing the Bayesian model written in the JAGS syntax.
#  - The Monte Carlo simulation proposed here (assuming nMC = 100) can take hours to run. 
#    The subfolders 'MCresults_s#_n#' (within the folder 'MCresults') contain the final results for faster inspection.  

# Function to run the MC scheme. Time grid tau is fixed here.
# Sample size n, the true configuration of lambda, and the number of MC replications nMC are 
# treated as input arguments.
Model_pex_MCpexm <- "model{
  for (i in 1:n) {
    t[i] ~ dpex(lambda[], tau[])
  }
  for (j in 1:m) { lambda[j] ~ dgamma(0.01, 0.01) }
}
"

Model_pex_MCpois <- "model {
  C <- 10000
  for (i in 1:n) {
    zeros[i] ~ dpois(zero_mean[i])
    zero_mean[i] <- -loglik[i] + C
    loglik[i] <- log(lambda[id[i]]) - inprod(lambda[], Tnm[i,])
  }
  for (j in 1:m) { 
    lambda[j] ~ dgamma(0.01, 0.01) 
  }
}
"

runMC <- function(n, lambda, nMC){
  #
  if(sum(diff(lambda)) > 0){ sce = "s1" }
  if(sum(diff(lambda)) == 0){ sce = "s2" }
  if(sum(diff(lambda)) < 0){ sce = "s3" }
  tau <- c(0.0, 2.0, 3.0, 5.0)
  m <- length(tau)
  burnin <- 1000; lag <- 1; npost <- 2000
  #
  for(step in 1:nMC){
    # Set the same starting values for both types of implementations (pexm and Poisson-zeros).
    inits1 <- list( lambda = c(0.1, 0.5, 1, 2), .RNG.name = "base::Super-Duper", .RNG.seed = 1 ) 
    inits2 <- list( lambda = c(0.5, 1.0, 1.5, 2.5), .RNG.name = "base::Wichmann-Hill", .RNG.seed = 2 )
    #
    # Gernerate the artificial data.
    t <- rpexp(n, lambda, tau)
    #
    # Prepare MCMC for the pexm case.
    data = list(t = t, n = n, tau = tau, m = m)
    # Compile and run the model using pexm.
    Mjags1 <- rjags::jags.model(textConnection(Model_pex_MCpexm), data = data, 
                                inits = list(inits1, inits2), n.chains = 2, n.adapt = burnin, quiet = TRUE)
    time0 <- proc.time()
    invisible(capture.output(output_pexm <- rjags::coda.samples(Mjags1, variable.names = "lambda", n.iter = npost, thin = lag) ))
    time_pexm <- proc.time() - time0
    #
    # Prepare MCMC for the Poisson-zeros strategy.
    # Identify the grid interval for each t[i].
    id <- as.numeric(cut(t, c(tau, max(t)), include.lowest = TRUE))
    # Matrix Tnm (nxm) is such that Tnm[i,j] contains the amplitude of the j-th interval with respect to the position of t[i].
    Tnm <- matrix(nrow = n, ncol = m); 
    a <- c(tau, max(t))
    for(i in 1:n){ 
      for(j in 1:m){ 
        Tnm[i,j] <- (min(t[i], a[j+1]) - a[j]) * (t[i] > a[j]) 
      } 
    }
    # The vectors "t" and "tau" are replaced by the matrix "Tnm" and the vector "id". 
    # "Tnm" and "id" account for the location of each t[i] in the grid.
    data <- list(zeros = numeric(n), Tnm = Tnm, id = id, n = n, m = m)
    Mjags2 <- rjags::jags.model(textConnection(Model_pex_MCpois), data = data, inits = list(inits1, inits2), n.chains = 2, n.adapt = burnin, quiet = TRUE)
    time0 <- proc.time()
    invisible(capture.output( output_pois <- rjags::coda.samples(Mjags2, variable.names = "lambda", n.iter = npost, thin = lag) ))
    time_pois <- proc.time() - time0
    #
    # Extract and save the main results.
    fld_out <- paste0("MCresults_",sce,"_n",n)
    # pexm:
    labels = paste0("lambda[",1:4,"]")
    aux1 = as.matrix(output_pexm[, labels])
    me <- apply(aux1, 2, mean)
    aux2 <- coda::as.mcmc(aux1)
    hpd <- HPDinterval(aux2)
    hpd1 <- hpd[,1]; hpd2 <- hpd[,2]
    aux3 <- output_pexm[, labels][1] # Autocorrelation and effective sample size (only the 1st chain for each parameter). 
    aco <- coda::autocorr.diag(aux3, lags = c(1), relative=FALSE)
    eff <- coda::effectiveSize(aux3)
    write.table(matrix(round(me, 5), nrow = 1), file = file.path(fld_out, "pexm_mean.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(hpd1, 5), nrow = 1), file = file.path(fld_out, "pexm_hpd1.txt"),
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(hpd2, 5), nrow = 1), file = file.path(fld_out, "pexm_hpd2.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(time_pexm[1], 5),nrow = 1), file = file.path(fld_out, "pexm_time.txt"),
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(aco, 5),nrow = 1), file = file.path(fld_out, "pexm_autocor.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(eff, 5),nrow = 1), file = file.path(fld_out, "pexm_effsize.txt"),
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    # Poisson-zeros:
    aux1 = as.matrix(output_pois[, labels])
    me <- apply(aux1, 2, mean)
    aux2 <- coda::as.mcmc(aux1)
    hpd <- HPDinterval(aux2)
    hpd1 <- hpd[,1]; hpd2 <- hpd[,2]
    aux3 <- output_pois[, labels][1] # Autocorrelation and effective sample size (only the 1st chain for each parameter). 
    aco <- coda::autocorr.diag(aux3, lags = c(1), relative=FALSE)
    eff <- coda::effectiveSize(aux3)
    write.table(matrix(round(me, 5), nrow = 1), file = file.path(fld_out, "pois_mean.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(hpd1, 5),nrow = 1), file = file.path(fld_out, "pois_hpd1.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(hpd2, 5),nrow = 1), file = file.path(fld_out, "pois_hpd2.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)    
    write.table(matrix(round(time_pois[1], 5), nrow = 1), file = file.path(fld_out, "pois_time.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(aco, 5),nrow = 1), file = file.path(fld_out, "pois_autocor.txt"),
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    write.table(matrix(round(eff, 5),nrow = 1), file = file.path(fld_out, "pois_effsize.txt"), 
                row.names = FALSE, col.names = FALSE, quote = FALSE, append = TRUE)
    #
    cat("\014"); cat("Scenario = ",sce,"; n = ",n,"; MC progress (in %): ", round(100*step/nMC, 2), "\r");
  } # end MC iterations.
} # end function runMC.   

# Run the MC scheme using the commands below.
# Warning: It will take hours, especially for n = 1000, to run each scenario with nMC = 100 replications. 
# The object nMC can be configured at the top of this R script. 
# For faster inspection, the final results (assuming nMC = 100) were provided in the subfolders 
# 'MCresults_s#_n#' (within the folder 'MCresults'). 
# Set MCflag = FALSE (at the top of this R script) if planning to use the results from the given folder 'MCresults'.

############## MC scheme starts here ##############
if(MCflag == TRUE){
  
  set.seed(1981)
  
  # Scenario 1 (increasing lambda)
  lambda <- c(0.3, 0.6, 0.8, 1.3)
  dir.create("MCresults_s1_n100"); runMC(n = 100, lambda, nMC)
  dir.create("MCresults_s1_n1000"); runMC(n = 1000, lambda, nMC)
  
  set.seed(1981)
  
  # Scenario 2 (constant lambda)
  lambda <- c(0.7, 0.7, 0.7, 0.7)
  dir.create("MCresults_s2_n100"); runMC(n = 100, lambda, nMC)
  dir.create("MCresults_s2_n1000"); runMC(n = 1000, lambda, nMC)
  
  set.seed(1981)
  
  # Scenario 3 (decreasing lambda)
  lambda <- c(1.3, 0.8, 0.6, 0.3)
  dir.create("MCresults_s3_n100"); runMC(n = 100, lambda, nMC)
  dir.create("MCresults_s3_n1000"); runMC(n = 1000, lambda, nMC)
  
  # After executing the commands above, the results will be saved in 
  # 6 folders ('MCresults_s#_n#') created in the main working directy. The names of these
  # folders indicate the scenario (s1, s2 or s3) and the sample
  # size (n = 100 or 1000). 
  
}
############## MC scheme ends here ##############

# Consider the next commands to build graphs intended to explore the MC simulation results.

# Warning: In order to reproduce the results shown in the paper, you must consider the 
# MC scheme with nMC = 100 replications.
# If you have specified MCflag = TRUE and nMC not equal to 100 (these elements are defined at the top of this R script), 
# the code below will not replicate the results from the paper.

if(MCflag == TRUE & nMC != 100){ 
  message("Warning: You must consider either 'MCflag = FALSE' or 'MCflag = TRUE and nMC = 100' 
          to be able to reproduce Figure 2 in the paper.") 
}

if(MCflag == TRUE){ wd1 = NULL }
if(MCflag == FALSE){ wd1 = "MCresults/" }

# Load the results from the MC simulation.
c_name = c("pexm", "pois")
s_name = c("s1", "s2", "s3")
n_name = c("n100", "n1000")
real1 = matrix(rep(c(0.3, 0.6, 0.8, 1.3),nMC), nMC, 4, byrow = TRUE)
real2 = matrix(rep(c(0.7, 0.7, 0.7, 0.7),nMC), nMC, 4, byrow = TRUE)
real3 = matrix(rep(c(1.3, 0.8, 0.6, 0.3),nMC), nMC, 4, byrow = TRUE)
for(i in c_name){
  for(j in s_name){
    for(k in n_name){
      fld_out <- paste0(wd1,"MCresults_",j,"_",k)
      me = read.table(file.path(fld_out,paste0(i,"_mean.txt")))
      hpd1 = read.table(file.path(fld_out, paste0(i,"_hpd1.txt")))
      hpd2 = read.table(file.path(fld_out, paste0(i,"_hpd2.txt")))
      time = read.table(file.path(fld_out,paste0(i,"_time.txt")))
      aco = read.table(file.path(fld_out,paste0(i,"_autocor.txt")))
      eff = read.table(file.path(fld_out,paste0(i,"_effsize.txt")))    
      if(j == "s1"){real = real1}
      if(j == "s2"){real = real2}
      if(j == "s3"){real = real3}
      rb = 100 * (me - real) / abs(real)
      cp = 100 * apply( (hpd1 < real & hpd2 > real), 2, sum ) / nMC
      assign(paste0(i,"_rb_",j,"_",k), rb)
      assign(paste0(i,"_cp_",j,"_",k), cp)
      assign(paste0(i,"_ti_",j,"_",k), time)
      assign(paste0(i,"_au_",j,"_",k), aco)
      assign(paste0(i,"_ef_",j,"_",k), eff)
    }
  }
}

###################################################
# Code to replicate Figure 2 (a) in Section 3.3.
# Comparison "pexm vs. Poisson-zeros".
# Computational times.
# Important remark: the magnitude of the
# results will vary depending on the configuration
# of the computer used in MC simulations.
# The conclusions are expected to be the same.
###################################################
par(mfrow = c(3, 2))
# Panel 1:
aux = cbind(pexm_ti_s1_n100, pois_ti_s1_n100)
co = c("powderblue","mistyrose")
boxplot(aux, names = c("pexm","zeros-trick"), ylim = c(0, 10), 
        cex.axis = 2, cex.lab = 2, ylab = "Time", main = "S1, n = 100",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
# Panel 2:
aux = cbind(pexm_ti_s1_n1000, pois_ti_s1_n1000)
co = c("powderblue","mistyrose")
boxplot(aux, names = c("pexm","zeros-trick"), ylim = c(0, 60), 
        cex.axis = 2, cex.lab = 2, ylab = "Time", main = "S1, n = 1000",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
# Panel 3:
aux = cbind(pexm_ti_s2_n100, pois_ti_s2_n100)
co = c("powderblue","mistyrose")
boxplot(aux, names = c("pexm","zeros-trick"), ylim = c(0, 10), 
        cex.axis = 2, cex.lab = 2, ylab = "Time", main = "S2, n = 100",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
# Panel 4:
aux = cbind(pexm_ti_s2_n1000, pois_ti_s2_n1000)
co = c("powderblue","mistyrose")
boxplot(aux, names = c("pexm","zeros-trick"), ylim = c(0, 60), 
        cex.axis = 2, cex.lab = 2, ylab = "Time", main = "S2, n = 1000",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
# Panel 5:
aux = cbind(pexm_ti_s3_n100, pois_ti_s3_n100)
co = c("powderblue","mistyrose")
boxplot(aux, names = c("pexm","zeros-trick"), ylim = c(0, 10), 
        cex.axis = 2, cex.lab = 2, ylab = "Time", main = "S3, n = 100",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
# Panel 6:
aux = cbind(pexm_ti_s3_n1000, pois_ti_s3_n1000)
co = c("powderblue","mistyrose")
boxplot(aux, names = c("pexm","zeros-trick"), ylim = c(0, 60), 
        cex.axis = 2, cex.lab = 2, ylab = "Time", main = "S3, n = 1000",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)

#################################################
# Code to replicate Figure 2 (b) in Section 3.3.
# Comparison "pexm vs. Poisson-zeros".
# Effective sample sizes of the chains.
#################################################
par(mfrow = c(3, 2))
# Panel 1:
aux = cbind(pexm_ef_s1_n100, rep(NA, nMC), pois_ef_s1_n100)
co = c(rep("powderblue", 4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-10, 2000), cex.axis = 2, 
        cex.lab = 2, ylab = "effective size", xlab = "lambda", main = "S1, n = 100",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 2:
aux = cbind(pexm_ef_s1_n1000, rep(NA, nMC), pois_ef_s1_n1000)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-10, 2000), cex.axis = 2, 
        cex.lab = 2, ylab = "effective size", xlab = "lambda", main = "S1, n = 1000",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 3:
aux = cbind(pexm_ef_s2_n100, rep(NA, nMC), pois_ef_s2_n100)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-10, 2000), cex.axis = 2, 
        cex.lab = 2, ylab = "effective size", xlab = "lambda", main = "S2, n = 100",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 4:
aux = cbind(pexm_ef_s2_n1000, rep(NA, nMC), pois_ef_s2_n1000)
co = c(rep("powderblue", 4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-10, 2000), cex.axis = 2, 
        cex.lab = 2, ylab = "effective size", xlab = "lambda", main = "S2, n = 1000",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 5:
aux = cbind(pexm_ef_s3_n100, rep(NA, nMC), pois_ef_s3_n100)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-10, 2000), cex.axis = 2, 
        cex.lab = 2, ylab = "effective size", xlab = "lambda", main = "S3, n = 100",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 6:
aux = cbind(pexm_ef_s3_n1000, rep(NA, nMC), pois_ef_s3_n1000)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-10, 2000), cex.axis = 2, 
        cex.lab = 2, ylab = "effective size", xlab = "lambda", main = "S3, n = 1000",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)

#############################################
# Additional graph not included in the paper.
# Comparison "pexm vs. Poisson-zeros".
# Autocorrelations of the chains.
#############################################
par(mfrow = c(3, 2))
# Panel 1:
aux = cbind(pexm_au_s1_n100, rep(NA, nMC), pois_au_s1_n100)
co = c(rep("powderblue", 4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(0, 1), main = "S1, n = 100",
        cex.axis = 2, cex.lab = 2, ylab = "autocorrelation", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 2:
aux = cbind(pexm_au_s1_n1000, rep(NA, nMC), pois_au_s1_n1000)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(0, 1), main = "S1, n = 1000", 
        cex.axis = 2, cex.lab = 2, ylab = "autocorrelation", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 3:
aux = cbind(pexm_au_s2_n100, rep(NA, nMC), pois_au_s2_n100)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(0, 1), main = "S2, n = 100", 
        cex.axis = 2, cex.lab = 2, ylab = "autocorrelation", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 4:
aux = cbind(pexm_au_s2_n1000, rep(NA, nMC), pois_au_s2_n1000)
co = c(rep("powderblue", 4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(0, 1), main = "S2, n = 1000", 
        cex.axis = 2, cex.lab = 2, ylab = "autocorrelation", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 5:
aux = cbind(pexm_au_s3_n100, rep(NA, nMC), pois_au_s3_n100)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(0, 1), main = "S3, n = 100", 
        cex.axis = 2, cex.lab = 2, ylab = "autocorrelation", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)
# Panel 6:
aux = cbind(pexm_au_s3_n1000, rep(NA, nMC), pois_au_s3_n1000)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(0, 1), main = "S3, n = 1000", 
        cex.axis = 2, cex.lab = 2, ylab = "autocorrelation", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1)

##################################################
# Additional graph not included in the paper.
# Comparison "pexm vs. Poisson-zeros".
# Relative bias (RB) for each parameter.
# Remark: RB = 100* (estimate - true) / true.
##################################################
par(mfrow = c(3, 2))
# Panel 1:
aux = cbind(pexm_rb_s1_n100, rep(NA, nMC), pois_rb_s1_n100)
co = c(rep("powderblue", 4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-150, 400), main = "S1, n = 100", 
        cex.axis = 2, cex.lab = 2, ylab = "relative bias", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1); abline(h = 0, lwd = 3, col = "royalblue")
# Panel 2:
aux = cbind(pexm_rb_s1_n1000, rep(NA, nMC), pois_rb_s1_n1000)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-150, 400), main = "S1, n = 1000", 
        cex.axis = 2, cex.lab = 2, ylab = "relative bias", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1); abline(h = 0, lwd = 3, col = "royalblue")
# Panel 3:
aux = cbind(pexm_rb_s2_n100, rep(NA, nMC), pois_rb_s2_n100)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-150, 400), main = "S2, n = 100", 
        cex.axis = 2, cex.lab = 2, ylab = "relative bias", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1); abline(h = 0, lwd = 3, col = "royalblue")
# Panel 4:
aux = cbind(pexm_rb_s2_n1000, rep(NA, nMC), pois_rb_s2_n1000)
co = c(rep("powderblue", 4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-150, 400), main = "S2, n = 1000", 
        cex.axis = 2, cex.lab = 2, ylab = "relative bias", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1); abline(h = 0, lwd = 3, col = "royalblue")
# Panel 5:
aux = cbind(pexm_rb_s3_n100, rep(NA, nMC), pois_rb_s3_n100)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-150, 400), main = "S3, n = 100", 
        cex.axis = 2, cex.lab = 2, ylab = "relative bias", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1); abline(h = 0, lwd = 3, col = "royalblue")
# Panel 6:
aux = cbind(pexm_rb_s3_n1000, rep(NA, nMC), pois_rb_s3_n1000)
co = c(rep("powderblue",4), NA, rep("mistyrose", 4))
boxplot(aux, names = c(1, 2, 3, 4, NA, 1, 2, 3, 4), ylim = c(-150, 400), main = "S3, n = 1000", 
        cex.axis = 2, cex.lab = 2, ylab = "relative bias", xlab = "lambda",
        whisklty = 1, whisklwd = 2, boxlwd = 2, staplelwd = 2, col = co)
abline(v = 5, lwd = 1); abline(h = 0, lwd = 3, col = "royalblue")

##################################################
# Additional graph not included in the paper.
# Comparison "pexm vs. Poisson-zeros".
# Coverage percentage based on 95% HPD intervals.
# For each parameter, this is the proportion of 
# MC replications that provided an HPD interval 
# containing the true value.
##################################################
par(mfrow = c(3, 2))
co = c(rep("blue", 4), NA, rep("red", 4))
# Panel 1:
aux = c(pexm_cp_s1_n100, NA, pois_cp_s1_n100)
plot(aux, xaxt = "n", pch = 19, cex = 3, ylim = c(50, 100), col = co, main = "S1, n = 100",
     cex.axis = 2, cex.lab = 2, ylab = "coverage percentage", xlab = "lambda")
axis(1, at = 1:9, labels = c(1, 2, 3, 4, NA, 1, 2, 3, 4), cex.axis = 2)
abline(v = 5, lwd = 1); abline(h = 95, lwd = 3, col = "royalblue")
# Panel 2:
aux = c(pexm_cp_s1_n1000, NA, pois_cp_s1_n1000)
plot(aux, xaxt = "n", pch = 19, cex = 3, ylim = c(50, 100), col = co, main = "S1, n = 1000",
     cex.axis = 2, cex.lab = 2, ylab = "coverage percentage", xlab = "lambda")
axis(1, at = 1:9, labels = c(1, 2, 3, 4, NA, 1, 2, 3, 4), cex.axis = 2)
abline(v = 5, lwd = 1); abline(h = 95, lwd = 3, col = "royalblue")
# Panel 3:
aux = c(pexm_cp_s2_n100, NA, pois_cp_s2_n100)
plot(aux, xaxt = "n", pch = 19, cex = 3, ylim = c(50, 100), col = co, main = "S2, n = 100",
     cex.axis = 2, cex.lab = 2, ylab = "coverage percentage", xlab = "lambda")
axis(1, at = 1:9, labels = c(1, 2, 3, 4, NA, 1, 2, 3, 4), cex.axis = 2)
abline(v = 5, lwd = 1); abline(h = 95, lwd = 3, col = "royalblue")
# Panel 4:
aux = c(pexm_cp_s2_n1000, NA, pois_cp_s2_n1000)
plot(aux, xaxt = "n", pch = 19, cex = 3, ylim = c(50, 100), col = co, main = "S2, n = 1000",
     cex.axis = 2, cex.lab = 2, ylab = "coverage percentage", xlab = "lambda")
axis(1, at = 1:9, labels = c(1, 2, 3, 4, NA, 1, 2, 3, 4), cex.axis = 2)
abline(v = 5, lwd = 1); abline(h = 95, lwd = 3, col = "royalblue")
# Panel 5:
aux = c(pexm_cp_s3_n100, NA, pois_cp_s3_n100)
plot(aux, xaxt = "n", pch = 19, cex = 3, ylim = c(50, 100), col = co, main = "S3, n = 100",
     cex.axis = 2, cex.lab = 2, ylab = "coverage percentage", xlab = "lambda")
axis(1, at = 1:9, labels = c(1, 2, 3, 4, NA, 1, 2, 3, 4), cex.axis = 2)
abline(v = 5, lwd = 1); abline(h = 95, lwd = 3, col = "royalblue")
# Panel 6:
aux = c(pexm_cp_s3_n1000, NA, pois_cp_s3_n1000)
plot(aux, xaxt = "n", pch = 19, cex = 3, ylim = c(50, 100), col = co, main = "S3, n = 1000",
     cex.axis = 2, cex.lab = 2, ylab = "coverage percentage", xlab = "lambda")
axis(1, at = 1:9, labels = c(1, 2, 3, 4, NA, 1, 2, 3, 4), cex.axis = 2)
abline(v = 5, lwd = 1); abline(h = 95, lwd = 3, col = "royalblue")

