# Hermite quadrature for true probabilities
library("statmod")
library("TruncatedNormal")
library("tlrmvnmvt")
library("mvtnorm")
nnode <- 200
nodeWeight <- gauss.quad(nnode, "hermite")
intfct <- function(x, b, rho) {
  y <- rep(0, length(x))
  for (i in 1:length(x)) y[i] <- 1/sqrt(pi) * prod(pnorm((b + 
    sqrt(2 * rho) * x[i])/sqrt(1 - rho)))
  return(y)
}
constRhoProb <- function(b, rho) {
  sum(nodeWeight$weights * intfct(nodeWeight$nodes, b, rho))
}
# simulation parameters
problemSZ <- c(16, 64, 128, 512, 1024, 2048, 4096, 16384)
pkgs <- c("mvtnorm", "TruncatedNormal", "pmvn_dense", "pmvn_tlr")
rhoVec <- c(0.5, 0.5, 0.8)
b0Vec <- c(0, -1, -1)
niter <- 10  
# result var
time_mvtnorm <- rep(NA_real_, length(problemSZ))
err_mvtnorm <- rep(NA_real_, length(problemSZ))
time_Trunc <- rep(NA_real_, length(problemSZ))
err_Trunc <- rep(NA_real_, length(problemSZ))
time_dense <- rep(NA_real_, length(problemSZ))
err_dense <- rep(NA_real_, length(problemSZ))
time_tlr <- rep(NA_real_, length(problemSZ))
err_tlr <- rep(NA_real_, length(problemSZ))
# tmp var
tmpTime <- rep(0, niter)
tmpErr <- rep(0, niter)
# simulation
for (k in 1:3) {
  i <- 1
  rho <- rhoVec[k]
  b0 <- b0Vec[k]
  set.seed(123)
  for (n in problemSZ) {
    b <- rep(b0, n)
    covM <- matrix(rho, n, n)
    diag(covM) <- 1
    probTrue <- constRhoProb(b, rho)
    # mvtnorm
    if (n < 1024) {
      for (j in 1:niter) {
        tmpTime[j] <- system.time(prob <- mvtnorm::pmvnorm(upper = b, 
          corr = covM)[[1]])[[3]]
        tmpErr[j] <- abs(log2(prob) - log2(probTrue))/log2(probTrue)
      }
      time_mvtnorm[i] <- mean(tmpTime)
      err_mvtnorm[i] <- mean(tmpErr)
    }
    # TruncatedNormal
    if (n < 4096) {
      for (j in 1:niter) {
        tmpTime[j] <- system.time(prob <- TruncatedNormal::pmvnorm(mu = rep(0, 
          n), sigma = covM, ub = b)[[1]])[[3]]
        tmpErr[j] <- abs(log2(prob) - log2(probTrue))/log2(probTrue)
      }
      time_Trunc[i] <- mean(tmpTime)
      err_Trunc[i] <- mean(tmpErr)
    }
    # tlrmvtmvn dense
    if (n < 16384) {
      for (j in 1:niter) {
        tmpTime[j] <- system.time(prob <- tlrmvnmvt::pmvn(lower = rep(-Inf, 
          n), upper = b, sigma = covM))[[3]]
        tmpErr[j] <- abs(log2(prob) - log2(probTrue))/log2(probTrue)
      }
      time_dense[i] <- mean(tmpTime)
      err_dense[i] <- mean(tmpErr)
    }
    # tlrmvnmvt tlr
    {
      for (j in 1:niter) {
        tmpTime[j] <- system.time(prob <- tlrmvnmvt::pmvn(lower = rep(-Inf, 
          n), upper = b, sigma = covM, algorithm = tlrmvnmvt::TLRQMC(m = round(sqrt(n)))))[[3]]
        tmpErr[j] <- abs(log2(prob) - log2(probTrue))/log2(probTrue)
      }
      time_tlr[i] <- mean(tmpTime)
      err_tlr[i] <- mean(tmpErr)
    }
    i <- i + 1
  }
  rm(covM, b)
  save.image(paste0("const_corr_", rho, "_", -b0, ".RData"))
}
rm(list = ls())

## Random corr
# simulation parameters
n <- 1000
nProblem <- 20
nIter <- 10
b0 <- 0.5
nu <- 7
a <- rep(-Inf, n)
b <- rep(b0, n)
# result var
time_mvtnorm <- rep(NA_real_, nProblem)
err_mvtnorm <- rep(NA_real_, nProblem)
logerr_mvtnorm <- rep(NA_real_, nProblem)
time_Trunc <- rep(NA_real_, nProblem)
err_Trunc <- rep(NA_real_, nProblem)
logerr_Trunc <- rep(NA_real_, nProblem)
time_tlr <- rep(NA_real_, nProblem)
err_tlr <- rep(NA_real_, nProblem)
logerr_tlr <- rep(NA_real_, nProblem)
time_mvtnorm_t <- rep(NA_real_, nProblem)
err_mvtnorm_t <- rep(NA_real_, nProblem)
logerr_mvtnorm_t <- rep(NA_real_, nProblem)
time_Trunc_t <- rep(NA_real_, nProblem)
err_Trunc_t <- rep(NA_real_, nProblem)
logerr_Trunc_t <- rep(NA_real_, nProblem)
time_tlr_t <- rep(NA_real_, nProblem)
err_tlr_t <- rep(NA_real_, nProblem)
logerr_tlr_t <- rep(NA_real_, nProblem)
# tmp var
tmpprob <- rep(0, nIter)
tmptime <- rep(0, nIter)
# simulation
set.seed(123)
for (i in 1:nProblem) {
  lambda <- runif(n)
  lambda <- lambda * n/sum(lambda)
  covM <- fungible::rGivens(lambda, Seed = i)$R
  # mvtnorm normal
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- mvtnorm::pmvnorm(lower = a, 
    upper = b, corr = covM)[[1]])[[3]]
  time_mvtnorm[i] <- mean(tmptime)
  err_mvtnorm[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_mvtnorm[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # TruncatedNormal normal
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- TruncatedNormal::pmvnorm(rep(0, 
    n), lb = a, ub = b, sigma = covM)[[1]])[[3]]
  time_Trunc[i] <- mean(tmptime)
  err_Trunc[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_Trunc[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # tlrmvnmvt normal
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- tlrmvnmvt::pmvn(lower = a, 
    upper = b, sigma = covM))[[3]]
  time_tlr[i] <- mean(tmptime)
  err_tlr[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_tlr[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # mvtnorm t
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- mvtnorm::pmvt(lower = a, 
    upper = b, df = nu, corr = covM)[[1]])[[3]]
  time_mvtnorm_t[i] <- mean(tmptime)
  err_mvtnorm_t[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_mvtnorm_t[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # TruncatedNormal t
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- TruncatedNormal::pmvt(rep(0, 
    n), lb = a, ub = b, df = nu, sigma = covM)[[1]])[[3]]
  time_Trunc_t[i] <- mean(tmptime)
  err_Trunc_t[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_Trunc_t[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # tlrmvnmvt t
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- tlrmvnmvt::pmvt(lower = a, 
    upper = b, df = nu, sigma = covM))[[3]]
  time_tlr_t[i] <- mean(tmptime)
  err_tlr_t[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_tlr_t[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
}
rm(covM, a, b)
save.image("rnd.RData")

## Whittle kernel
rm(list = ls())
# simulation parameters
nProblem <- 20
nIter <- 10
nx <- 30
ny <- 30
n <- nx * ny
vecx <- c(1:nx) - 1
vecy <- c(1:ny) - 1
geom <- cbind(kronecker(vecx, rep(1, ny)), kronecker(rep(1, nx), 
  vecy))
nu <- 7
# result var
time_mvtnorm <- rep(NA_real_, nProblem)
err_mvtnorm <- rep(NA_real_, nProblem)
logerr_mvtnorm <- rep(NA_real_, nProblem)
time_Trunc <- rep(NA_real_, nProblem)
err_Trunc <- rep(NA_real_, nProblem)
logerr_Trunc <- rep(NA_real_, nProblem)
time_tlr <- rep(NA_real_, nProblem)
err_tlr <- rep(NA_real_, nProblem)
logerr_tlr <- rep(NA_real_, nProblem)
time_mvtnorm_t <- rep(NA_real_, nProblem)
err_mvtnorm_t <- rep(NA_real_, nProblem)
logerr_mvtnorm_t <- rep(NA_real_, nProblem)
time_Trunc_t <- rep(NA_real_, nProblem)
err_Trunc_t <- rep(NA_real_, nProblem)
logerr_Trunc_t <- rep(NA_real_, nProblem)
time_tlr_t <- rep(NA_real_, nProblem)
err_tlr_t <- rep(NA_real_, nProblem)
logerr_tlr_t <- rep(NA_real_, nProblem)
# tmp var
tmpprob <- rep(0, nIter)
tmptime <- rep(0, nIter)
# simulation
set.seed(123)
for (i in 1:nProblem) {
  a <- runif(n, -5, -1)
  b <- runif(n, 1, 5)
  geomNew <- geom + matrix(runif(n * 2), n, 2) * 0.8
  geomNew <- geomNew/max(nx, ny)
  distM <- as.matrix(dist(geomNew))
  covM <- geoR::matern(distM, 0.1, 1)
  # mvtnorm normal
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- mvtnorm::pmvnorm(lower = a, 
    upper = b, corr = covM)[[1]])[[3]]
  time_mvtnorm[i] <- mean(tmptime)
  err_mvtnorm[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_mvtnorm[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # TruncatedNormal normal
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- TruncatedNormal::pmvnorm(rep(0, 
    n), lb = a, ub = b, sigma = covM)[[1]])[[3]]
  time_Trunc[i] <- mean(tmptime)
  err_Trunc[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_Trunc[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # tlrmvnmvt normal
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- tlrmvnmvt::pmvn(lower = a, 
    upper = b, sigma = covM))[[3]]
  time_tlr[i] <- mean(tmptime)
  err_tlr[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_tlr[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # mvtnorm t
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- mvtnorm::pmvt(lower = a, 
    upper = b, df = nu, corr = covM)[[1]])[[3]]
  time_mvtnorm_t[i] <- mean(tmptime)
  err_mvtnorm_t[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_mvtnorm_t[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # TruncatedNormal t
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- TruncatedNormal::pmvt(rep(0, 
    n), lb = a, ub = b, df = nu, sigma = covM)[[1]])[[3]]
  time_Trunc_t[i] <- mean(tmptime)
  err_Trunc_t[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_Trunc_t[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
  # tlrmvnmvt t
  for (j in 1:nIter) tmptime[j] <- system.time(tmpprob[j] <- tlrmvnmvt::pmvt(lower = a, 
    upper = b, df = nu, sigma = covM))[[3]]
  time_tlr_t[i] <- mean(tmptime)
  err_tlr_t[i] <- sd(tmpprob)/mean(tmpprob)
  logerr_tlr_t[i] <- abs(sd(log2(tmpprob))/mean(log2(tmpprob)))
}
rm(distM, covM, a, b)
save.image("Whittle.RData")

## Random correlation (time plot only)
rm(list = ls())
# simulation parameters
nVec <- c(10, 20, 40, 80, 160, 320, 640, 1000, 2000, 4000)
b0 <- 0.5
# result var
nLen <- length(nVec)
time_mvtnorm <- rep(NA_real_, nLen)
time_Trunc <- rep(NA_real_, nLen)
time_tlr <- rep(NA_real_, nLen)
# simulation
i <- 1
set.seed(123)
for (n in nVec) {
  lambda <- runif(n)
  lambda <- lambda * n/sum(lambda)
  covM <- fungible::rGivens(lambda, Seed = i)$R
  b <- rep(b0, n)
  if (n <= 1000) 
    time_mvtnorm[i] <- system.time(mvtnorm::pmvnorm(upper = b, 
      corr = covM))[[3]]
  time_Trunc[i] <- system.time(TruncatedNormal::pmvnorm(rep(0, 
    n), ub = b, sigma = covM))[[3]]
  time_tlr[i] <- system.time(tlrmvnmvt::pmvn(lower = rep(0, 
    n), upper = b, sigma = covM))[[3]]
  i <- i + 1
}
rm(covM, b, lambda)
save.image("rndcorr_time.RData")

## Whittle kernel (time plot only)
rm(list = ls())
# simulation parameters
mVec <- c(8, 11, 15, 21, 29, 41, 57, 80, 128, 180, 256)
# result var
mLen <- length(mVec)
time_dense <- rep(NA_real_, mLen)
time_tlr <- rep(NA_real_, mLen)
# simulation
i <- 1
set.seed(123)
for (m in mVec) {
  n <- m * m
  a <- runif(n, -5, -1)
  b <- runif(n, 1, 5)
  nx <- m
  ny <- m
  vecx <- c(1:nx) - 1
  vecy <- c(1:ny) - 1
  geom <- cbind(kronecker(vecx, rep(1, ny)), kronecker(rep(1, 
    nx), vecy))
  geom <- geom + matrix(runif(n * 2), n, 2) * 0.8
  geom <- geom/max(nx, ny)
  # dense
  if (m <= 128) {
    distM <- as.matrix(dist(geom))
    covM <- geoR::matern(distM, 0.1, 1)
    diag(covM) <- diag(covM) + 0.05
    time_dense[i] <- system.time(tlrmvnmvt::pmvn(lower = a, 
      upper = b, sigma = covM))[[3]]
  }
  # TLR
  time_tlr[i] <- system.time(tlrmvnmvt::pmvn(lower = a, upper = b, 
    geom = geom, algorithm = tlrmvnmvt::TLRQMC(m = m), kernelType = "matern", 
    para = c(1, 0.1, 1, 0.05)))[[3]]
  i <- i + 1
}
rm(distM, covM, geom, a, b, vecx, vecy)
save.image("Whittle_time.RData")














