################################################################################
## 3. Package structure and implementation
################################################################################

## 3.2. Computation with dense matrices
set.seed(123)
nx <- 20
ny <- 20
n <- nx * ny
vecx <- c(1:nx) - 1
vecy <- c(1:ny) - 1
geom <- cbind(kronecker(vecx, rep(1, ny)), kronecker(rep(1, nx), vecy))
geom <- geom + matrix(runif(n * 2), n, 2)
geom <- geom / max(nx, ny)
a <- runif(n, -5, -1)
b <- runif(n, 1, 5)

library("geoR")
distM <- as.matrix(dist(geom))
covM <- matern(distM, 0.1, 1.0)

library("tlrmvnmvt")
sprintf("Time costs: %f seconds",
        system.time(ret <- pmvn(a, b, 0, covM))[[3]])
ret

nu <- 7
sprintf("Time costs: %f seconds",
        system.time(ret <- pmvt(a, b, 0, nu, covM, uselog2 = TRUE))[[3]])
ret

sprintf("Time costs: %f seconds",
        system.time(ret <- pmvn(a, b, 0, geom = geom, kernelType = "matern", 
                                para = c(1, 0.1, 1, 0),
                                algorithm = GenzBretz(N = 737)))[[3]])
ret

## 3.3. Computation with TLR matrices
sprintf("Time costs: %f seconds",
        system.time(ret <- pmvn(a, b, 0, covM, algorithm = TLRQMC(m = 20)))[[3]])
ret

sprintf("Time costs: %f seconds",
        system.time(ret <- pmvt(a, b, 0, nu, covM, uselog2 = TRUE, 
                                algorithm = TLRQMC(m = 20)))[[3]])
ret

set.seed(123)
n <- 4000
geom <- c(0:(n - 1)) / n
geom <- geom + runif(n) / n * 0.8
distM <- as.matrix(dist(geom))
covM <- matern(distM, 0.1, 1.0)
a <- runif(n, -5, -1)
b <- runif(n, 1, 5)

sprintf("Time costs: %f seconds",
        system.time(ret <- pmvn(a, b, 0, covM, 
                                algorithm = TLRQMC(m = 64, epsl = 1e-6)))[[3]])
ret

set.seed(123)
nx <- 50
ny <- 80
n <- nx * ny
vecx <- c(1:nx) - 1
vecy <- c(1:ny) - 1
geom <- cbind(kronecker(vecx, rep(1, ny)), kronecker(rep(1, nx), vecy))
geom <- geom + matrix(runif(n * 2), n, 2) * 0.8
geom <- geom / max(nx, ny)
idxZ <- zorder(geom)
geom <- geom[idxZ, ]
distM <- as.matrix(dist(geom))
covM1 <- matern(distM, 0.1, 1.0)
covM2 <- matern(distM, 0.1, 0.5)
covM <- covM1 + covM2
a <- runif(n, -10, -2)
b <- runif(n, 2, 10)

sprintf("Time costs: %f seconds",
        system.time(ret <- pmvn(a, b, 0, covM, 
                                algorithm = TLRQMC(m = 64, epsl = 1e-5)))[[3]])
ret

################################################################################
## 4. Performance comparison
################################################################################

## Run simulation.R to obtain results.

load("const_corr_0.5_0.RData")
Table1 <- list(err = 100 * abs(rbind(err_mvtnorm, err_Trunc, err_dense, err_tlr)),
               time = rbind(time_mvtnorm, time_Trunc, time_dense, time_tlr))
colnames(Table1[[1]]) <- colnames(Table1[[2]]) <- problemSZ
load("const_corr_0.5_1.RData")
Table2 <- list(err = 100 * abs(rbind(err_mvtnorm, err_Trunc, err_dense, err_tlr)),
               time = rbind(time_mvtnorm, time_Trunc, time_dense, time_tlr))
colnames(Table2[[1]]) <- colnames(Table2[[2]]) <- problemSZ
load("const_corr_0.8_1.RData")
Table3 <- list(err = 100 * abs(rbind(err_mvtnorm, err_Trunc, err_dense, err_tlr)),
               time = rbind(time_mvtnorm, time_Trunc, time_dense, time_tlr))
colnames(Table3[[1]]) <- colnames(Table3[[2]]) <- problemSZ

Table1
Table2
Table3

load("rnd.RData")
Table4a <- rbind(mvtnorm = c(summary(logerr_mvtnorm) * 100, time = mean(time_mvtnorm)),
                 Trunc = c(summary(logerr_Trunc) * 100, time = mean(time_Trunc)),
                 tlr = c(summary(logerr_tlr) * 100, time = mean(time_tlr)))
Table4b <- rbind(mvtnorm_t = c(summary(logerr_mvtnorm_t) * 100, time = mean(time_mvtnorm_t)),
                 Trunc_t = c(summary(logerr_Trunc_t) * 100, time = mean(time_Trunc_t)),
                 tlr_t = c(summary(logerr_tlr_t) * 100, time = mean(time_tlr_t)))
Table4a
Table4b
                
load("Whittle.RData")                
Table5a <- rbind(mvtnorm = c(summary(logerr_mvtnorm) * 100, time = mean(time_mvtnorm)),
                 Trunc = c(summary(logerr_Trunc) * 100, time = mean(time_Trunc)),
                 tlr = c(summary(logerr_tlr) * 100, time = mean(time_tlr)))
Table5b <- rbind(mvtnorm_t = c(summary(logerr_mvtnorm_t) * 100, time = mean(time_mvtnorm_t)),
                 Trunc_t = c(summary(logerr_Trunc_t) * 100, time = mean(time_Trunc_t)),
                 tlr_t = c(summary(logerr_tlr_t) * 100, time = mean(time_tlr_t)))
Table5a
Table5b

library("ggplot2")

load("rndcorr_time.RData")
mydf <- data.frame(rep(nVec, 3), c(time_mvtnorm, time_Trunc, 
  time_tlr), rep(c("mvtnorm", "TruncatedNormal", "pmvn.genz"), 
  each = nLen))
colnames(mydf) <- c("n", "time", "package")
ggplot(mydf, aes(x = n, y = time, color = package, label = sprintf("%0.2f", 
  round(time, digits = 2))), show.legend = FALSE) + geom_point(aes(shape = package)) + 
  labs(y = "Computation time (s)") + scale_y_continuous(trans = "log", 
  breaks = round(2^seq(log2(min(mydf$time, na.rm = TRUE)), log2(max(mydf$time, 
    na.rm = TRUE)), length.out = 4), 2)) + scale_x_continuous(trans = "log", 
  breaks = unique(mydf$n)) + theme_classic(base_size = 16) + 
  theme(legend.position = c(0.2, 0.85), legend.margin = margin(c(2, 
    5, 2, 2)), legend.box.background = element_blank(), legend.spacing.y = unit(0, 
    "mm"), legend.text = element_text(size = 16), legend.background = element_blank(), 
    legend.title = element_blank())

load("Whittle_time.RData")
nVec <- mVec^2
mydf <- data.frame(rep(nVec, 2), c(time_dense, time_tlr), rep(c("pmvn.genz", 
  "pmvn.tlr"), each = mLen))
colnames(mydf) <- c("n", "time", "package")
ggplot(mydf, aes(x = n, y = time, color = package, label = sprintf("%0.2f", 
  round(time, digits = 2))), show.legend = FALSE) + geom_point(aes(shape = package)) + 
  labs(y = "Computation time (s)") + scale_y_continuous(trans = "log", 
  breaks = round(2^seq(log2(min(mydf$time, na.rm = TRUE)), log2(max(mydf$time, 
    na.rm = TRUE)), length.out = 4), 2)) + scale_x_continuous(trans = "log", 
  breaks = unique(mydf$n)) + theme_classic(base_size = 16) + 
  theme(legend.position = c(0.2, 0.85), legend.margin = margin(c(2, 
    5, 2, 2)), legend.box.background = element_blank(), legend.spacing.y = unit(0, 
    "mm"), legend.text = element_text(size = 16), legend.background = element_blank(), 
    legend.title = element_blank())

################################################################################
## 5. Application in finding excursion sets
################################################################################

library("geoR")
library("tlrmvnmvt")
library("fields")
# Set some parameters and create the grid
n.obs <- 1000
sigma.e <- 0.5
m <- 80
n <- m * m
x <- seq(from = 0, to = 1, length.out = m)
y <- x
geom <- cbind(kronecker(x, rep(1, m)), kronecker(rep(1, m), y))
odrMorton <- zorder(geom)
geom <- geom[odrMorton, ]
distM <- as.matrix(dist(geom))
covM <- matern(distM, phi = sqrt(2)/20, kappa = 1)
cholM <- t(chol(covM))
mu <- rep(0, n)
set.seed(120)
y0 <- as.vector(cholM %*% rnorm(n)) + mu

# observe at n.obs locations under additive Gaussian noise
obsIdx <- sample(1:n, n.obs)
Y <- y0[obsIdx] + rnorm(n.obs) * sigma.e

# a parameter matrix A
A <- matrix(0, n.obs, n)
A[cbind(seq_len(n.obs), obsIdx)] <- 1
# posterior mean and variance
Q <- chol2inv(chol(covM))
Q.post <- Q
Q.post <- Q.post + crossprod(A) / (sigma.e^2)
mu.post <- as.vector(mu + solve(Q.post, 
  (t(A) %*% (Y - mu[obsIdx]))/(sigma.e^2)))
covM <- chol2inv(chol(Q.post)) 
rm(cholM, distM, Q, A)

# marginal exceedance prob
pMar <- 1 - pnorm(0, mean = mu.post, sd = sqrt(diag(covM)))
# compute the excursion function FMar
FMar <- rep(0, n)
ttlNum <- sum(pMar > 0.96)
pMarOdr <- order(pMar, decreasing = TRUE)
numVec <- round(seq(from = 3180, to = 1000, by = -10))
for (num in numVec) {
  selectIdx <- pMarOdr[1:num]
  tmpLower <- rep(-Inf, n)
  tmpLower[selectIdx] <- 0
  tmpUpper <- rep(Inf, n)
  FMar[selectIdx] <- pmvn(lower = tmpLower, upper = tmpUpper, 
    mean = mu.post, sigma = covM, algorithm = TLRQMC(m = 80))
}
# Plot pMar and FMar
pdf("marginalprob.pdf", width = 5, height = 5)
par(mar = c(0,0,0,0))
cmap <- gray(seq(1, 0, length.out = 10))
image(x, y, matrix(pMar[order(odrMorton)], m, m), xlab = "", 
  ylab = "", col = cmap, axes = FALSE, main = NULL)
dev.off()
pdf("marginalF.pdf", width = 5, height = 5)
par(mar = c(0,0,0,0))
image(x, y, matrix(FMar[order(odrMorton)], m, m), xlab = " ", 
  ylab = " ", col = cmap, axes = FALSE, main = NULL)
dev.off()
pdf("marginallegend.pdf", width = 0.63, height = 5)
par(mar = c(0,0,0,0))
plot(c(0, 2), c(0, 10), type = "n", xlab = "", ylab = "", axes = FALSE)
for (i in 1:10) rect(0, i - 1, 1, i, col = cmap[i])
text(x = rep(1.7, 10), y = (c(1:10) - 0.5), 
  labels = as.character(c(1:10)/10), cex = 0.8)
dev.off()
### Estimation error based on Monte Carlo simulate N * ns
### samples
cholM <- t(chol(covM))
set.seed(123)
N <- 10000
ns <- 5
# used as the x-axis of Fig. 5
Funique <- sort(unique(FMar))
Funique <- Funique[Funique < 0.999]
Fsample <- matrix(0, length(Funique), ns)
# compute the excursion function
for (k in 1:ns) {
  Xtmp <- matrix(rnorm(n * N), n, N)
  X <- cholM %*% Xtmp + mu.post
  for (i in 2:length(Funique)) {
    Fsample[i, k] <- mean(1 * (apply(X[FMar >= Funique[i], 
      ], 2, function(x) min(x, na.rm = TRUE)) > 0))
  }
}
Fmean <- rowMeans(Fsample)
# plot the excursion function
pdf("mc_error.pdf", width = 8, height = 5)
par(mar = c(5, 5, 0, 0))
plot(x = Funique, y = Fmean - Funique, type = "l", xlab = expression(1 - 
  alpha), ylab = expression(1 - alpha - hat(p)(alpha)), frame.plot = FALSE)
abline(h = 0, lty = "dashed", col = "red")
dev.off()
