
# This replication script serves to replicate all small code snippets in the
# article "The JuliaConnectoR: A Functionally Oriented Interface for
# Integrating Julia in R", with the exception of the example for using neural
# ordinary differential equations, which is contained in the GitHub repository
# https://github.com/stefan-m-lenz/JuliaConnectoR-SpiralExample.

# This script has been tested with Julia versions 1.4, 1.5, and 1.6
# and JuliaConnectoR version 1.1.0.

#------------------------------------------------------------------------------

# Install the JuliaConnectoR package
if (!require("JuliaConnectoR")) {
    install.packages("JuliaConnectoR")
    library("JuliaConnectoR")
}    

# Load an environment that contains all required Julia packages
Pkg <- juliaImport("Pkg")
# For this we can use the environment specified via the files Project.toml and
# Manifest.toml, which are contained in the same directory.
# These files were created by recording the package operations in a Julia
# project.
# For more information on this, see "Working with environments" in the Julia
# documentation: https://julialang.github.io/Pkg.jl/v1.4/environments/
if (juliaEval('VERSION < v"1.6"')) {
  Pkg$activate("project_1_4") # project environment for Julia 1.4 and 1.5
} else {
  Pkg$activate("project_1_6") # project environment for Julia 1.6
}
# Instantiating the environment may take a while because
# missing packages are installed.
Pkg$instantiate()

#------------------------------------------------------------------------------

# Mixed models with R package "lme4" and Julia package "MixedModels"
if (!require("lme4")) {
    install.packages("lme4")
    library("lme4")
}    

# Prepare data set InstEval (contained in lme4 package)
data("InstEval", package = "lme4")
InstEval$dept <- as.factor(as.integer(as.character(InstEval$dept)))
InstEval$y <- as.double(InstEval$y)
saveRDS(InstEval, "InstEval.rds", version = 2)

# Fit mixed model in R
library("lme4")
InstEval <- readRDS("InstEval.rds")
formula1 <- y ~ 1 + service*dept + (1|s) + (1|d)
system.time(fm1 <- lmer(formula1, InstEval, REML = TRUE))
which.max(ranef(fm1)$d$`(Intercept)`)

# Fit mixed model in Julia
library("JuliaConnectoR")
MM <- juliaImport("MixedModels")
RData <- juliaImport("RData")
InstEval <- RData$load("InstEval.rds")
formula2 <- juliaEval("MixedModels.@formula(y ~ 1 + service*dept + (1|s) + (1|d))")
system.time(fm2 <- MM$fit(MM$LinearMixedModel, formula2, InstEval, REML = TRUE))
which.max(juliaCall("MixedModels.ranef", fm2)[[2]])

# (For evaluating the execution time in Julia, the code is measured on the
# second run to rule out the precompilation time.)

#------------------------------------------------------------------------------

# Example for installing the package Flux and defining a neural network in Flux
Pkg <- juliaImport("Pkg")
Pkg$add(Pkg$PackageSpec(name = "Flux", version = "0.11"))
Flux <- juliaImport("Flux") # this may take a while
model <- Flux$Chain(Flux$Dense(4L, 4L, Flux$relu), Flux$Dense(4L, 1L))

#------------------------------------------------------------------------------

# Loading a module
# in two lines:
juliaCall("include", normalizePath("mymodule.jl"))
MyModule <- juliaImport(".MyModule")
# or in one line:
MyModule <- juliaImport(juliaCall("include", normalizePath("mymodule.jl")))

#------------------------------------------------------------------------------

# Translating functions
f2 <- juliaEval("x -> x .^ 2")
f2 <- juliaEval('function f2(x)
                   x .^ 2
                 end')
f2 <- juliaFun("f2")
f2(2)

#------------------------------------------------------------------------------

# Translating structs into R objects
MyLibrary <- juliaImport(juliaCall("include", normalizePath("MyLibrary.jl")))
book <- juliaGet(MyLibrary$Book("Shakespeare", "Romeo and Julia", 1597L))
book
MyLibrary$cite(book)

#------------------------------------------------------------------------------

# Callbacks
juliaCall("map", function(x) {x+1}, c(1,2,3))

#------------------------------------------------------------------------------

# Let-syntax
juliaLet('Dict("x" => x, "y" => y)', x = c(1,2), y = c(2,3))

#------------------------------------------------------------------------------

# Interrupting
# juliaEval('while true; end') # (uncomment for interactive testing)

#------------------------------------------------------------------------------

# Missing values
juliaCall("+", c(1, NA, NaN), c(1, 2, 3))
juliaCall("sqrt", NA)

