## Section 3 ------------------------------------
## Package and options
# install.packages("NeuralSens")
# install.packages("NeuralSens_1.0.0.tar.gz", repos = NULL)
library("NeuralSens", quietly = TRUE)
options(prompt = "R> ", continue = "+  ", width = 70,
        useFancyQuotes = FALSE)

# Perform robustness analysis and calculate computational time only on purpose
# This prevent the user to run code that might last too long
do_robustness_analysis <- FALSE
do_comput_time_analysis <- FALSE


## Section 3.2 ----------------------------------
## SensAnalysisMLP function

# Train neural network from RSNNS package to test SensAnalysisMLP
library("RSNNS", quietly = TRUE)

# mod1 is comment out because it gives a different model for 
# different operating systems, so it would be read from a RDS file 
# to exactly reproduce the results.
# set.seed(150) # For replication
# mod1 <- mlp(simdata[, c("X1", "X2", "X3")], simdata[, "Y"],
#             maxit = 1000, size = 10, linOut = TRUE)
# saveRDS(mod1, file = "RSNNSmodel.RDS")
mod1 <- readRDS(file = "RSNNSmodel.RDS")

# SensAnalysisMLP to calculate sensitivity measures
sens <- SensAnalysisMLP(mod1, trData = simdata,
                        output_name = "Y", plot = FALSE)

class(sens)
summary(sens)

print(sens, n = 7)

## Section 3.3 ----------------------------------
## SensitivityPlots function
# Create sensitivity plots based on the sensitivities calculated
SensitivityPlots(sens)

# Using plot() method
plot(sens)


## Section 3.4 ----------------------------------
## SensTimePlot function
# Load data to test SensTimePlot

# Exploratory analysis
library("ggplot2", quietly = TRUE)
ggplot(DAILY_DEMAND_TR) + geom_point(aes(x = TEMP, y = DEM))

data("DAILY_DEMAND_TR", package = "NeuralSens")

# Scale data
DAILY_DEMAND_TR[, 4] <- DAILY_DEMAND_TR[, 4]/100
DAILY_DEMAND_TR[, 2] <- DAILY_DEMAND_TR[, 2]/100

# Train neural network from caret package to test SensTimePlot
library("caret", quietly = TRUE)
set.seed(150) # For replication
mod2 <- caret::train(form = DEM~TEMP + WD,
                    data = DAILY_DEMAND_TR,
                    method = "nnet",
                    linout = TRUE,
                    tuneGrid = data.frame(size = 5,
                                          decay = 0.1),
                    maxit = 250,
                    preProcess = c("center", "scale"),
                    trControl = trainControl(),
                    metric = "RMSE",
                    trace = FALSE)

# SensTimePlot of the first 2 years of data
SensTimePlot(mod2,
             fdata = DAILY_DEMAND_TR[1:(365*2), ],
             date.var = DAILY_DEMAND_TR[1:(365*2), 1],
             facet = TRUE,
             output_name = "DEM")

# SensTimePlot of some weeks to obtain better information of WD
SensTimePlot(mod2,
             fdata = DAILY_DEMAND_TR[75:135, ],
             date.var = DAILY_DEMAND_TR[75:135, 1],
             facet = TRUE,
             output_name = "DEM")

# Using plot() method
sens2 <- SensAnalysisMLP(mod2, trData = DAILY_DEMAND_TR[1:(365*2), ],
                         output_name = "DEM", plot = FALSE)

plot(sens2, plotType = "time", facet = TRUE,
     date.var = DAILY_DEMAND_TR[1:(365*2), 1])

## Section 3.5 ----------------------------------
## SensFeaturePlot function
# SensFeaturePlot of the first 2 years of data
SensFeaturePlot(mod2, fdata = DAILY_DEMAND_TR[1:(365*2), ], output_name = "DEM")

# Using plot() method
plot(sens2, plotType = "features")

## Section 3.7 ----------------------------------
## Robustness analysis ------------------------------
if (do_robustness_analysis) {
  library("NeuralNetTools", quietly = TRUE)
  library("dplyr", quietly = TRUE)
  neurons <- c(1, 10, 20)
  seeds <- sample(c(1:5000), 50)
  grids <- expand.grid(neurons, seeds)
  names(grids) <- c("neurons", "seeds")
  # iterate through conditions
  imp_vals <- vector("list", nrow(grids))
  imp_vals_garson <- vector("list", nrow(grids))
  imp_vals_olden <- vector("list", nrow(grids))
  for(i in 1:nrow(grids)){
    cat(i, "\t")
    # get specific conditions
    seed_val <- grids[i, "seeds"]
    set.seed(seed_val)
    neurons_val <- grids[i, "neurons"]
    # create model
    mod_robust <- nnet::nnet(Y ~ ., size = neurons_val, maxit = 300,
                             linout = TRUE, data = simdata, trace = FALSE)

    # get importance, append to output
    imp <- SensAnalysisMLP(mod_robust, trData = simdata, output_name = "Y", plot = FALSE)$sens[[1]]
    imp_vals[[i]] <- imp

    imp_vals_garson[[i]] <- garson(mod_robust, bar_plot = FALSE)
    imp_vals_olden[[i]] <- olden(mod_robust, bar_plot = FALSE)
  }

  imp_sums <- do.call("cbind", imp_vals)
  imp_sums_mean <- imp_sums[, names(imp_sums) == "mean"]
  imp_sums_std <- imp_sums[, names(imp_sums) == "std"]
  imp_sums_sq <- imp_sums[, names(imp_sums) == "meanSensSQ"]
  nms_mean <- row.names(imp_sums_mean)
  nms_std <- row.names(imp_sums_std)
  nms_sq <- row.names(imp_sums_sq)
  imp_sums_mean <- t(imp_sums_mean)
  imp_sums_std <- t(imp_sums_std)
  imp_sums_sq <- t(imp_sums_sq)
  names(imp_sums_mean) <- nms_mean
  names(imp_sums_std) <- nms_std
  names(imp_sums_sq) <- nms_sq
  row.names(imp_sums_mean) <- 1:nrow(imp_sums_mean)
  row.names(imp_sums_std) <- 1:nrow(imp_sums_std)
  row.names(imp_sums_sq) <- 1:nrow(imp_sums_sq)

  library("tidyr", quietly = TRUE)
  imp_sums_mean <- data.frame(grids, imp_sums_mean) %>%
    gather("variable", "importance", -neurons, -seeds) %>%
    group_by(neurons, variable) %>%
    do((function(x) {
      out <- quantile(x$importance, c(0, 0.5, 1))
      out <- data.frame(t(out))
      names(out) <- c("lo", "med", "hi")
      out
    })(.)) %>%
    data.frame
  imp_sums_std <- data.frame(grids, imp_sums_std) %>%
    gather("variable", "importance", -neurons, -seeds) %>%
    group_by(neurons, variable) %>%
    do((function(x) {
      out <- quantile(x$importance, c(0, 0.5, 1))
      out <- data.frame(t(out))
      names(out) <- c("lo", "med", "hi")
      out
    })(.)) %>%
    data.frame
  imp_sums_sq <- data.frame(grids, imp_sums_sq) %>%
    gather("variable", "importance", -neurons, -seeds) %>%
    group_by(neurons, variable) %>%
    do((function(x) {
      out <- quantile(x$importance, c(0, 0.5, 1))
      out <- data.frame(t(out))
      names(out) <- c("lo", "med", "hi")
      out
    })(.)) %>%
    data.frame

  # subset by neurons so barplots can be ordered by medians
  toplo1_mean <- imp_sums_mean[imp_sums_mean$neurons == neurons[1], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo2_mean <- imp_sums_mean[imp_sums_mean$neurons == neurons[2], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo3_mean <- imp_sums_mean[imp_sums_mean$neurons == neurons[3], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo1_std <- imp_sums_std[imp_sums_std$neurons == neurons[1], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo2_std <- imp_sums_std[imp_sums_std$neurons == neurons[2], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo3_std <- imp_sums_std[imp_sums_std$neurons == neurons[3], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo1_sq <- imp_sums_sq[imp_sums_sq$neurons == neurons[1], ]%>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo2_sq <- imp_sums_sq[imp_sums_sq$neurons == neurons[2], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo3_sq <- imp_sums_sq[imp_sums_sq$neurons == neurons[3], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))

  # widths
  wd <- 0.3

  p1_mean <- ggplot(toplo1_mean, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    ggtitle(paste0(neurons[1], " neurons")) + theme(plot.title = element_text(hjust = 0.5)) +
    ylab("Mean") + scale_x_discrete(limits=c("X1","X2","X3"))

  p2_mean <- ggplot(toplo2_mean, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none")  +
    ggtitle(paste0(neurons[2], " neurons")) + theme(plot.title = element_text(hjust = 0.5))+
    ylab("Mean") + scale_x_discrete(limits=c("X1","X2","X3"))

  p3_mean <- ggplot(toplo3_mean, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none")  +
    ggtitle(paste0(neurons[3], " neurons")) + theme(plot.title = element_text(hjust = 0.5)) +
    ylab("Mean") + scale_x_discrete(limits=c("X1","X2","X3"))

  p1_std <- ggplot(toplo1_std, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    ylab("Standard Deviation") + scale_x_discrete(limits=c("X1","X2","X3"))

  p2_std <- ggplot(toplo2_std, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    ylab("Standard Deviation") + scale_x_discrete(limits=c("X1","X2","X3"))

  p3_std <- ggplot(toplo3_std, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none")+
    ylab("Standard Deviation") + scale_x_discrete(limits=c("X1","X2","X3"))

  p1_sq <- ggplot(toplo1_sq, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    ylab("Mean of Squares") + scale_x_discrete(limits=c("X1","X2","X3"))

  p2_sq <- ggplot(toplo2_sq, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none")+
    ylab("Mean of Squares") + scale_x_discrete(limits=c("X1","X2","X3"))

  p3_sq <- ggplot(toplo3_sq, aes(x = variable, y = med, fill = abs(med))) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none")+
    ylab("Mean of Squares") + scale_x_discrete(limits=c("X1","X2","X3"))

  gridExtra::grid.arrange(grobs = list(p1_mean, p2_mean, p3_mean,
                                       p1_std, p2_std, p3_std,
                                       p1_sq, p2_sq, p3_sq), nrow = 3)

  # summarize
  imp_sums <- do.call("cbind", imp_vals_olden)
  nms <- row.names(imp_sums)
  imp_sums <- t(imp_sums)
  names(imp_sums) <- nms
  row.names(imp_sums) <- 1:nrow(imp_sums)
  imp_sums <- data.frame(grids, imp_sums) %>%
    gather("variable", "importance", -neurons, -seeds) %>%
    group_by(neurons, variable) %>%
    do((function(x) {

      out <- quantile(x$importance, c(0, 0.5, 1))
      out <- data.frame(t(out))
      names(out) <- c("lo", "med", "hi")

      out

    })(.)) %>%
    data.frame
  # save(imp_sums, file = "imp_sums.RData", compress = "xz")

  # subet by neurons so barplots can be ordered by medians
  toplo1 <- imp_sums[imp_sums$neurons == neurons[1], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo2 <- imp_sums[imp_sums$neurons == neurons[2], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo3 <- imp_sums[imp_sums$neurons == neurons[3], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))

  # widths
  wd <- 0.3

  p1 <- ggplot(toplo1, aes(x = variable, y = med, fill = med)) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    scale_y_continuous("Importance")

  p2 <- ggplot(toplo2, aes(x = variable, y = med, fill = med)) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    scale_y_continuous("Importance")

  p3 <- ggplot(toplo3, aes(x = variable, y = med, fill = med)) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    scale_y_continuous("Importance")

  gridExtra::grid.arrange(grobs = list(p1, p2, p3), nrow = 1)

  imp_sums <- do.call("cbind", imp_vals_garson)
  nms <- row.names(imp_sums)
  imp_sums <- t(imp_sums)
  names(imp_sums) <- nms
  row.names(imp_sums) <- 1:nrow(imp_sums)
  imp_sums <- data.frame(grids, imp_sums) %>%
    gather("variable", "importance", -neurons, -seeds) %>%
    group_by(neurons, variable) %>%
    do((function(x) {

      out <- quantile(x$importance, c(0, 0.5, 1))
      out <- data.frame(t(out))
      names(out) <- c("lo", "med", "hi")

      out

    })(.)) %>%
    data.frame
  # save(imp_sums, file = "imp_sums.RData", compress = "xz")

  # subet by neurons so barplots can be ordered by medians
  toplo1 <- imp_sums[imp_sums$neurons == neurons[1], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo2 <- imp_sums[imp_sums$neurons == neurons[2], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))
  toplo3 <- imp_sums[imp_sums$neurons == neurons[3], ] %>%
    mutate(variable = factor(variable, levels = variable[order(med)]))

  # widths
  wd <- 0.3

  p1 <- ggplot(toplo1, aes(x = variable, y = med, fill = med)) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    scale_y_continuous("Importance")

  p2 <- ggplot(toplo2, aes(x = variable, y = med, fill = med)) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    scale_y_continuous("Importance")

  p3 <- ggplot(toplo3, aes(x = variable, y = med, fill = med)) +
    geom_bar(stat = "identity") +
    geom_errorbar(aes(ymin = lo, ymax = hi), width = wd) +
    theme_bw() +
    theme(axis.title.x = element_blank(), legend.position = "none") +
    scale_y_continuous("Importance")

  gridExtra::grid.arrange(grobs = list(p1, p2, p3), nrow = 1)
}

## Section 4 ------------------------------------
## Comparison with other packages
# Install other packages needed for the comparison section
# install.packages(c("lime", "NeuralNetTools", "pdp"), dep = TRUE)

## Section 4.1 ----------------------------------

# Exploratory analysis of the dataset
ggplot(iris) + geom_point(aes(x = Sepal.Width, y = Sepal.Length, color = Species))
ggplot(iris) + geom_point(aes(x = Petal.Width, y = Petal.Length, color = Species))


# Train neural network from caret package for the comparison
set.seed(150) # For replication
mod3 <- caret::train(Species ~ .,
              data = iris,
              method = "nnet",
              linout = FALSE,
              tuneGrid = data.frame(size = 5, decay = 0.1),
              maxit = 100,
              preProcess = c("center", "scale"),
              trControl = trainControl(),
              metric = "Accuracy",
              trace = FALSE)

# Sensitivity measures of the neural network model
sens3 <- SensAnalysisMLP(mod3, plot = FALSE)
summary(sens3)

# Create sensitivity plots of the first output
SensitivityPlots(sens3, der = FALSE, output = "setosa")


## CombineSens function
# Combine sensitivity measures for each of the output variables
combined_sens <- CombineSens(sens3)
summary(combined_sens)


## Comparison with NeuralNetTools package
# Load package
library("NeuralNetTools", quietly = TRUE)

# Garson's method
garson(mod3)

# Olden's method
olden(mod3)

## Comparison with lime package
# Load package
library("lime", quietly = TRUE)

# Linearize neural network model
explanation <- lime(iris, mod3)

# Plot explanations for one sample of each species
set.seed(150)
exp <- explain(iris[c(1, 51, 101), 1:4],
               explanation,
               n_labels = 3,
               n_features = 4)
plot_explanations(exp)

## Section 4.2 ----------------------------------
## Comparison with pdp package
# Load Boston dataset for regression
data("Boston", package = "MASS")

# Select only necessary variables
Boston <- Boston[, c("zn", "rad", "lstat", "nox")]

# Scale dataset
Boston <- as.data.frame(scale(Boston))

# Train neural network from nnet package for the comparison
set.seed(150) # For replication
mod4 <- nnet::nnet(nox ~ ., data = Boston, size = 25, decay = 0.1,
                   maxit = 150, linout = TRUE, trace = FALSE)


## Comparison with NeuralNetTools package
# Load package
library("NeuralNetTools", quietly = TRUE)


# lekprofile function performing clustering
set.seed(150)
lekprofile(mod4, group_vals = 6)

# Show values of clusters
set.seed(150)
lekprofile(mod4, group_vals = 6, group_show = TRUE)

# Plot sensitivities values related to input values
SensFeaturePlot(mod4, fdata = Boston, output_name = "nox")

# Plot sensitivities plots
SensAnalysisMLP(mod4, trData = Boston, output_name = "nox", der = FALSE)

## Comparison with pdp package
# Load package
library("pdp", quietly = TRUE)

# Calculate partial derivatives
pdps <- list()
for (i in 3:1) {
  pdps[[4-i]] <- autoplot(partial(mod4, pred.var = names(Boston)[i],
                       train = Boston, ice = TRUE), train = Boston,
                       center = TRUE, alpha = 0.1, rug = TRUE) +
    theme_bw() + ylab("nox")
}

# Plot pdps
# install.packages("gridExtra", dep = TRUE)
gridExtra::grid.arrange(grobs = pdps, nrow = 1)


## Calculate computational time of each function
if (do_comput_time_analysis) {
  library("microbenchmark", quietly = TRUE)
  
  ## Download data set from 
  ## Dua, D. and Graff, C. (2019). UCI Machine Learning Repository
  ## [http://archive.ics.uci.edu/ml].
  ## Irvine, CA: University of California, School of Information and Computer Science.
  
  # if (!file.exists("YearPredictionMSD.txt")) {
  #   download.file("https://archive.ics.uci.edu/ml/machine-learning-databases/00203/YearPredictionMSD.txt.zip", "YearPredictionMSD.txt.zip")
  #   unzip("YearPredictionMSD.txt.zip", file = "YearPredictionMSD.txt")
  # }
  train_sev_var <- read.table("YearPredictionMSD.txt", sep = ",")
  names(train_sev_var) <- c("Year", paste0('timbre_average', 1:12), paste0('timbre_covariance', 1:78))
  neurons <- c(10, 30, 50, 100)
  times <- list()
  for (nvars in seq(5, ncol(train_sev_var), 5)) {
    for (neurons_val in neurons) {
      total_times <- list()
      for (percentil in 1:10) {
        samples <- percentil * 1000
        trim_train_sev_var <- train_sev_var[1:samples, c(1:nvars)]
        mod_time <- nnet::nnet(Year ~ ., size = neurons_val, linout = TRUE,
                               data = trim_train_sev_var, trace = FALSE, MaxNWts = 100000)
        times <- microbenchmark::microbenchmark(
          NeuralSens = NeuralSens::SensAnalysisMLP(mod_time, trData = trim_train_sev_var,
                                                   output_name = "Year", plot = FALSE),
          Garson = NeuralNetTools::garson(mod_time, bar_plot = FALSE),
          Olden = NeuralNetTools::olden(mod_time, bar_plot = FALSE),
          Lek = NeuralNetTools::lekprofile(mod_time, val_out = TRUE),
          ICE = for(i in 1:nvars){pdp::partial(mod_time, pred.var = names(trim_train_sev_var)[i],
                                               train = trim_train_sev_var, ice = TRUE)},
          pdp = for(i in 1:nvars){pdp::partial(mod_time, pred.var = names(trim_train_sev_var)[i],
                                               train = trim_train_sev_var, ice = FALSE)},
          times = 1)
        cat(percentil, "\t")
        total_times[[percentil]] <- data.frame(expr = times$expr, time = times$time,
                                               percentile = percentil, nvar = nvars, neurons = neurons_val)
      }
      total_times <- do.call("rbind", total_times)
      saveRDS(total_times, file = paste0("times_", as.character(neurons_val),
                                         "_", as.character(nvars), ".rds"))
    }
  }

  total_times <- data.frame()
  for (nvars in seq(5, ncol(train_sev_var), 5)) {
    for (neurons_val in neurons) {
      times <- readRDS(file = paste0("times_", as.character(neurons_val),
                                     "_", as.character(nvars), ".rds"))
      total_times <- rbind(total_times, times)
    }
  }
  total_times$percentile <- factor(total_times$percentile)
  total_times$nvar <- factor(total_times$nvar)
  total_times$neurons <- factor(total_times$neurons)
  levels(total_times$neurons) <- paste(levels(total_times$neurons), "neurons")

  saveRDS(total_times, file = "total_times.rds")
  total_times <- readRDS("total_times.rds")
  library("ggplot2", quietly = TRUE)

  p1 <- ggplot(total_times[total_times$percentile == 1, ]) +
    geom_point(aes(x = nvar, y = time, group = expr, color = expr)) +
    geom_line(aes(x = nvar, y = time, group = expr, color = expr)) +
    facet_wrap(neurons ~ ., nrow = 1) + scale_y_log10() + annotation_logticks() +
    theme(legend.position = "none") + ylab("time (log scale)") +
    xlab("Number of input variables") +
    ggtitle("Computational time using 1000 samples") +
    scale_x_discrete(breaks = c(seq(min(as.numeric(levels(total_times$nvar))),
                                    max(as.numeric(levels(total_times$nvar))),
                                    by = 15), 90))

  p2 <- ggplot(total_times[total_times$percentile == 5, ]) +
    geom_point(aes(x = nvar, y = time, group = expr, color = expr)) +
    geom_line(aes(x = nvar, y = time, group = expr, color = expr)) +
    facet_wrap(neurons ~ ., nrow = 1) + scale_y_log10() + annotation_logticks() +
    theme(legend.position = "none") + ylab("time (log scale)") +
    xlab("Number of input variables") +
    ggtitle("Computational time using 5000 samples") +
    scale_x_discrete(breaks = c(seq(min(as.numeric(levels(total_times$nvar))),
                                    max(as.numeric(levels(total_times$nvar))),
                                    by = 15), 90))

  p3 <- ggplot(total_times[total_times$percentile == 10, ]) +
    geom_point(aes(x = nvar, y = time, group = expr, color = expr)) +
    geom_line(aes(x = nvar, y = time, group = expr, color = expr)) +
    facet_wrap(neurons ~ ., nrow = 1) + scale_y_log10() + annotation_logticks() +
    theme(legend.position = "none") + ylab("time (log scale)") +
    xlab("Number of input variables") +
    ggtitle("Computational time using 10000 samples")+
    scale_x_discrete(breaks = c(seq(min(as.numeric(levels(total_times$nvar))),
                                    max(as.numeric(levels(total_times$nvar))),
                                    by = 15), 90))
  gridExtra::grid.arrange(grobs = list(p1, p2, p3), nrow = 1)
}

## Annex A ----------------------------------
# Extending package functionalities
# monMLP
# install.packages("monmlp")
library("monmlp", quietly = TRUE)

# monmlp_model is comment out because it gives a different model 
# for different operating systems, so it would be read from a RDS file 
# to exactly reproduce the results.
# set.seed(150)
# monmlp_model <- monmlp.fit(x = data.matrix(simdata[, 1:3]),
#                            y = data.matrix(simdata[, 4]),
#                            hidden1 = 5,
#                            scale.y = FALSE,
#                            iter.max = 500,
#                            silent = TRUE)
# saveRDS(monmlp_model, file = "monmlp_model.RDS")
monmlp_model <- readRDS(file = "monmlp_model.RDS")

# Correct models to change bias to first row
W1 <- rbind(monmlp_model[[1]]$W1[4, ],
            monmlp_model[[1]]$W1[1:3, ])
W2 <- c(monmlp_model[[1]]$W2[6, ],
        monmlp_model[[1]]$W2[1:5, ])
wts <- c(as.vector(W1),
         as.vector(W2))

# Define the structure of the model
mlpstruct <- c(3, 5, 1)

# Define the activation functions and their derivatives
Actfunc <- c("linear",
             attr(monmlp_model, "Th"),
             attr(monmlp_model, "To"))

Deractfunc <- c("linear",
                function(v) {
                  diag(attr(monmlp_model, "Th.prime")(v))
                },
                function(v) {
                  diag(attr(monmlp_model, "To.prime")(v))
                })

# Perform preprocess to data
x <- data.matrix(simdata[, 1:3])
x.center <- attr(monmlp_model, "x.center")
x.scale <- attr(monmlp_model, "x.scale")
x <- sweep(x, 2, x.center, "-")
x <- sweep(x, 2, x.scale, "/")
x <- cbind(data.frame(x), Y = simdata[, 4])

# Perform sensitivity analysis
sens_monmlp <- SensAnalysisMLP(wts,
                         trData = x,
                         mlpstr = mlpstruct,
                         coefnames = c("X1", "X2", "X3"),
                         output_name = "Y",
                         actfunc = Actfunc,
                         deractfunc = Deractfunc,
                         plot = FALSE)
summary(sens_monmlp)
# TensorFlow
# install.packages(c("tensorflow", "keras", "tfdatasets", "dplyr"), dep = TRUE)
set.seed(150)
library("tensorflow", quietly = TRUE)
# install_tensorflow()
set.seed(150)
library("tfdatasets", quietly = TRUE)
set.seed(150)
library("keras", quietly = TRUE)
set_random_seed(150)
# install_keras()
library("dplyr", quietly = TRUE)

keras_model <- keras_model_sequential()  %>%
  layer_dense(units = 16, activation = "relu",
              input_shape = 3) %>%
  layer_dense(units = 8, activation = "relu") %>%
  layer_dense(units = 1) %>%
  compile(
    loss = "mse",
    optimizer = optimizer_rmsprop(),
    metrics = list("mean_absolute_error")
  )
history <- keras_model %>% fit(
  data.matrix(simdata[, 1:3]),
  array(simdata[, 4]),
  epochs = 500,
  validation_split = 0.2,
  verbose = 0
)

model_weights <- get_weights(keras_model)

wts <- c()
neural_struct <- c(nrow(model_weights[[1]]))
for (i in seq(2, length(model_weights), 2)) {
  neural_struct <- c(neural_struct, dim(model_weights[[i]]))
  lyr_wgts <- rbind(model_weights[[i]], model_weights[[i - 1]])
  wts <- c(wts, unname(do.call(c, as.data.frame(lyr_wgts))))
}

actfunc <- c("linear", "ReLU", "ReLU", "linear")
sens_keras <- SensAnalysisMLP(wts, trData = simdata, mlpstr = neural_struct,
                         coefnames = names(simdata)[1:3],
                         output_name = names(simdata)[4],
                         actfunc = actfunc)
summary(sens_keras)

