package jss.perm;

import blang.mcmc.Samplers;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * A permutation or equivalently, a bipartite perfect
 * matching.
 * 
 * The annotation "@Samplers" links the data type with the appropriate sampler.
 * 
 * The annotation "@Data" is not related to data in the statistical
 * sense but rather read as 'data class', meaning that .equals, .hashcode
 * are automatically implemented, as well as other nice defaults
 * (see the xtend documentation for details).
 */
@Samplers(PermutationSampler.class)
@Data
@SuppressWarnings("all")
public class Permutation {
  /**
   * Assume the vertices are indexed 0, 1, ..., N in the first bipartite component, and
   * also 0, 1, 2, .., N in the second bipartite component.
   * 
   * For vertex i in the first component, connections.get(i)
   * give the index in the second bipartite component.
   */
  private final List<Integer> connections;
  
  /**
   * Initialize to the identity permutation.
   */
  public Permutation(final int componentSize) {
    this.connections = IterableExtensions.<Integer>toList(new ExclusiveRange(0, componentSize, true));
  }
  
  /**
   * The number of vertices in each of the two bipartite components.
   */
  public int componentSize() {
    return this.connections.size();
  }
  
  @Override
  public String toString() {
    return this.connections.toString();
  }
  
  /**
   * Sample an independent uniform permutation in place.
   */
  public void sampleUniform(final Random random) {
    Collections.<Integer>sort(this.connections);
    Collections.shuffle(this.connections, random);
  }
  
  @Override
  @Pure
  public int hashCode() {
    return 31 * 1 + ((this.connections== null) ? 0 : this.connections.hashCode());
  }
  
  @Override
  @Pure
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    Permutation other = (Permutation) obj;
    if (this.connections == null) {
      if (other.connections != null)
        return false;
    } else if (!this.connections.equals(other.connections))
      return false;
    return true;
  }
  
  @Pure
  public List<Integer> getConnections() {
    return this.connections;
  }
}
