######################################################################
## Section 5: The oem package
######################################################################

library("mvtnorm")
library("oem")
library("microbenchmark")
set.seed(1234)

######################################################################
## 5.1. The oem() function()
######################################################################

nobs <- 1e4
nvars <- 25
rho <- 0.25
sigma <- matrix(rho, ncol = nvars, nrow = nvars)
diag(sigma) <- 1
x <- rmvnorm(n = nobs, mean = numeric(nvars), sigma = sigma)
y <- drop(x %*% c(0.5, 0.5, -0.5, -0.5, rep(0, nvars - 4))) + 
    rnorm(nobs, sd = 3)

fit <- oem(x = x, y = y, penalty = c("lasso", "mcp", "grp.lasso"),
           gamma = 2, groups = rep(1:5, each = 5),
           lambda.min.ratio = 1e-3)
par(mar = c(5, 5, 5, 3) + 0.1)
layout(matrix(1:3, ncol = 3))
plot(fit, which.model = 1, xvar = "lambda", 
     cex.main = 3, cex.axis = 1.25, cex.lab = 2)
plot(fit, which.model = 2, xvar = "lambda",
     cex.main = 3, cex.axis = 1.25, cex.lab = 2)
plot(fit, which.model = 3, xvar = "lambda",
     cex.main = 3, cex.axis = 1.25, cex.lab = 2)

fit <- oem(x = x, y = y, penalty = c("lasso", "mcp", "grp.lasso"),
           gamma = 2, groups = rep(1:5, each = 5),
           lambda.min.ratio = 1e-3,
           compute.loss = TRUE)

logLik(fit, which.model = 2)[c(1, 25, 50, 100)]

######################################################################
## 5.2. Fitting multiple penalties
######################################################################

nobs <- 1e6
nvars <- 100
rho <- 0.25
sigma <- matrix(rho, ncol = nvars, nrow = nvars)
diag(sigma) <- 1
x2 <- rmvnorm(n = nobs, mean = numeric(nvars), sigma = sigma)
y2 <- drop(x2 %*% c(0.5, 0.5, -0.5, -0.5, rep(0, nvars - 4))) + 
    rnorm(nobs, sd = 5)

mb <- microbenchmark(
    "oem[lasso]" = oem(x = x2, y = y2, 
                       penalty = c("lasso"),
                       gamma = 3,
                       groups = rep(1:20, each = 5)),
    "oem[all]" = oem(x = x2, y = y2, 
                     penalty = c("lasso", "mcp", 
                                 "grp.lasso", "scad"),
                     gamma = 3,
                     groups = rep(1:20, each = 5)),
    times = 10L)
print(mb, digits = 3)

######################################################################
## 5.3. Parallel support via OpenMP
######################################################################

nobs <- 1e5
nvars <- 500
rho <- 0.25
sigma <- rho ** abs(outer(1:nvars, 1:nvars, FUN = "-"))
x2 <- rmvnorm(n = nobs, mean = numeric(nvars), sigma = sigma)
y2 <- drop(x2 %*% c(0.5, 0.5, -0.5, -0.5, rep(0, nvars - 4))) + 
    rnorm(nobs, sd = 5)
mb <- microbenchmark(
    "oem" = oem(x = x2, y = y2, 
                penalty = c("lasso", "mcp", 
                            "grp.lasso", "scad"),
                gamma = 3,
                groups = rep(1:20, each = 25)),
    "oem[parallel]" = oem(x = x2, y = y2, 
                          ncores = 2,
                          penalty = c("lasso", "mcp", 
                                      "grp.lasso", "scad"),
                          gamma = 3,
                          groups = rep(1:20, each = 25)),
    times = 10L)
print(mb, digits = 3)

######################################################################
## 5.4. The cv.oem() function
######################################################################

cvfit <- cv.oem(x = x, y = y, 
                penalty = c("lasso", "mcp", "grp.lasso"),
                gamma = 2, groups = rep(1:5, each = 5), 
                nfolds = 10)

cvfit$best.model
cvfit$lambda.min

summary(cvfit)

predict(cvfit, newx = x[1:3, ], 
        which.model = "best.model",
        s = "lambda.min")

######################################################################
## 5.5. The xval.oem() function
######################################################################

xvalfit <- xval.oem(x = x, y = y, 
                    penalty = c("lasso", "mcp", "grp.lasso"),
                    gamma = 2,
                    groups = rep(1:5, each = 5), 
                    nfolds = 10)

yrng <- range(c(unlist(xvalfit$cvup), unlist(xvalfit$cvlo)))
layout(matrix(1:3, ncol = 3))
par(mar = c(5, 5, 5, 3) + 0.1)
plot(xvalfit, which.model = 1, ylim = yrng, 
     cex.main = 3, cex.axis = 1.25, cex.lab = 2)
plot(xvalfit, which.model = 2, ylim = yrng,
     cex.main = 3, cex.axis = 1.25, cex.lab = 2)
plot(xvalfit, which.model = 3, ylim = yrng, 
     cex.main = 3, cex.axis = 1.25, cex.lab = 2)

######################################################################
## 5.6. OEM with precomputation for linear models with the oem.xtx()
##      function
######################################################################

xtx <- crossprod(x) / nrow(x)
xty <- crossprod(x, y) / nrow(x)
fitxtx <- oem.xtx(xtx, xty, 
                  penalty = c("lasso", "mcp", "grp.lasso"),
                  gamma = 2,
                  groups = rep(1:5, each = 5))

######################################################################
## 5.7. Out-of-memory computation with the big.oem() function
######################################################################

nobs <- 1e6
nvars <- 250
bkFile <- "big_matrix.bk"
descFile <- "big_matrix.desc"
big_mat <- filebacked.big.matrix(nrow = nobs, 
                                  ncol = nvars, 
                                  type = "double",  
                                  backingfile = bkFile, 
                                  backingpath = ".", 
                                  descriptorfile = descFile,
                                  dimnames = c(NULL, NULL))

for (i in 1:nvars) {
    big_mat[, i] <- rnorm(nobs)
}

yb <- rnorm(nobs, sd = 5)

profvis::profvis({
    bigfit <- big.oem(x = big_mat, y = yb, 
                      penalty = c("lasso", "grp.lasso", "mcp", "scad"),
                      gamma = 3,
                      groups = rep(1:50, each = 5))
})

xb <- big_mat[, ]

print(object.size(xb), units = "Mb")
print(object.size(big_mat), units = "Mb")

mb <- microbenchmark(
    "big.oem" = big.oem(x = big_mat, y = yb, 
                        penalty = c("lasso", "grp.lasso", 
                                    "mcp", "scad"),
                        gamma = 3,
                        groups = rep(1:50, each = 5)),
    "oem" =     oem(x = xb, y = yb, 
                        penalty = c("lasso", "grp.lasso", 
                                    "mcp", "scad"),
                        gamma = 3,
                        groups = rep(1:50, each = 5)),
    times = 10L)

print(mb, digits = 3)

######################################################################
## 5.8. Sparse matrix support
######################################################################

library("Matrix")
n.obs <- 1e5
n.vars <- 200
true.beta <- c(runif(15, -0.25, 0.25), rep(0, n.vars - 15))
xs <- rsparsematrix(n.obs, n.vars, density = 0.01)
ys <- rnorm(n.obs, sd = 3) + as.vector(xs %*% true.beta)
x.dense <- as.matrix(xs)

mb <- microbenchmark(fit = oem(x = x.dense, y = ys, 
                                 penalty = c("lasso", "grp.lasso"), 
                                 groups = rep(1:40, each = 5)),
                     fit.s = oem(x = xs, y = ys, 
                                 penalty = c("lasso", "grp.lasso"), 
                                 groups = rep(1:40, each = 5)),
                     times = 10L)

print(mb, digits = 3)

library("Matrix")
n.obs <- 1e5
n.vars <- 200
true.beta <- c(runif(15, -0.25, 0.25), rep(0, n.vars - 15))
xs <- rsparsematrix(n.obs, n.vars, density = 0.01)
ys <- rnorm(n.obs, sd = 3) + as.vector(xs %*% true.beta)
x.dense <- as.matrix(xs)

mb <- microbenchmark(fit = oem(x = x.dense, y = ys, 
                                 penalty = c("lasso", "grp.lasso"), 
                                 groups = rep(1:40, each = 5)),
                     fit.s = oem(x = xs, y = ys, 
                                 penalty = c("lasso", "grp.lasso"), 
                                 groups = rep(1:40, each = 5)),
                     times = 10L)

print(mb, digits = 3)

######################################################################
## 5.9. API comparison with glmnet
######################################################################

library("glmnet")

oem.fit <- oem(x = x, y = y, penalty = "lasso")
glmnet.fit <- glmnet(x = x, y = y)

oem.fit.sp <- oem(x = xs, y = ys, penalty = "lasso")
glmnet.fit.sp <- glmnet(x = xs, y = ys)

preds.oem <- predict(oem.fit, newx = x)
preds.glmnet <- predict(glmnet.fit, newx = x)

plot(oem.fit, xvar = "norm")
plot(glmnet.fit, xvar = "norm")

oem.cv.fit <- cv.oem(x = x, y = y, penalty = "lasso")
oem.xv.fit <- xval.oem(x = x, y = y, penalty = "lasso")
glmnet.cv.fit <- cv.glmnet(x = x, y = y)

plot(oem.cv.fit)
plot(oem.xv.fit)
plot(glmnet.cv.fit)

preds.cv.oem <- predict(oem.cv.fit, newx = x, s = "lambda.min")
preds.xv.oem <- predict(oem.xv.fit, newx = x, s = "lambda.min")
preds.cv.glmnet <- predict(glmnet.cv.fit, newx = x, s = "lambda.min")
