library("methods")
library("Matrix")

cv.stand <- function(X, y, group, lambda = NULL, K = 10, ...)
{
  require("standGL")
  n <- nrow(X)
  
  foldid <- sample(c(rep(1:K, each = n%/%K), rep(K, n%%K)))
  
  cv <- matrix(0, nrow = K, ncol = length(lambda))
  
  for (k in 1:K) {
    ind <- which(foldid == k)
    res <- standGL(y[-ind], X[-ind, ], group, family = "linear", lam.path = lambda, ...)
    
    cv[k, ] <- colSums((y[ind] - X[ind, ] %*% res$beta)^2)
  }
  
  cvsd <- apply(cv, 2, sd) * sqrt(K)
  
  cv <- colSums(cv)
  
  return(list(cv = cv, cvsd = cvsd, lambda = lambda))
}

clustering <- function(X, dist = c("cor", "cor2", "l2"))
{  
  ##### clustering part
  require("fastcluster")
  
  D <- c()
  
  if (dist == "cor") {
    # correlation based distance
    D <- as.dist(1 - abs(cor(X)))
  } else {
    n <- nrow(X)
    if (dist == "cor2") {
      # correlation based distance
      D <- as.dist(sqrt(2*n) * sqrt(1 - abs(cor(X))))
    } else {
      X2 <- scale(X, center = TRUE, scale = FALSE)
      X2 <- scale(X2, center = FALSE, scale = sqrt(colSums(X2^2)/n))
      
      D <- dist(t(X2))
    }
  }
  
  hc <- fastcluster::hclust(D, method = "average")
  
  # number of groups maximizing the branch length
  nbGroup <- p - which.max(diff(hc$height))
  
  group <- cutree(hc, nbGroup)
  return(group)
}

# cluster representative lasso
CRLasso <- function(X, y, hc = NULL, method = c("lasso", "standglasso", "glasso"), dist = c("cor", "cor2", "l2"), ...)
{
  temps <- rep(NA, 3)
  
  # Dimension
  n <- nrow(X)
  p <- ncol(X)
  
  if (is.null(hc)) {
    t1 <- proc.time()
    method <- match.arg(method)  
    dist <- match.arg(dist)    
    
    group <- clustering(X, dist)
    t2 <- proc.time()
    
    temps[1] <- t2[3] - t1[3]
  } else {
    # number of groups maximizing the branch length
    nbGroup <- p - which.max(diff(hc$height))
    
    group <- cutree(hc, nbGroup)
  }
  
  if (method == "lasso") {
    t1 <- proc.time()
    ##### Cluster Representative Lasso
    Xbar <- do.call(cbind, tapply(1:p, group, function(x) {
        rowMeans(X[, x, drop = FALSE])
    }))
    #    require("lars")
    #    res <- lars(Xbar, y, ...)
    #    rescv <- cv.lars(Xbar, y, K=10, ...)
    
    require("glmnet")
    res <- glmnet(Xbar, y, ...)
    t2 <- proc.time()
    rescv <- cv.glmnet(Xbar, y, nfolds = 10, ...)
    t3 <- proc.time()
    
    temps[2] <- t2[3] - t1[3]
    temps[3] <- t3[3] - t2[3]
    
    return(list(group = group, CRL = res, CRLcv = rescv, method = method, temps = temps))
  } else {
    if (method == "standglasso") {
      ##### Cluster Group Lasso
      require("standGL")
      t1 <- proc.time()
      resst <- standGL(y, cbind(1, X), c(0, group), family = "linear", is.pen = c(0, rep(1, length(unique(group)))), min.lam.frac = 0.001, ...)
      # rescv <- cv.standGL(y, cbind(1, X), c(0, group), family = "linear", is.pen = c(0, rep(1, length(unique(group)))), min.lam.frac = 0.001, nfold = 10,...)
      t2 <- proc.time()
      rescv <- cv.stand(cbind(1, X), y, c(0, group), lambda = resst$lam.path, K = 10, is.pen = c(0, rep(1, length(unique(group)))))
      t3 <- proc.time()
      
      temps[2] <- t2[3] - t1[3]
      temps[3] <- t3[3] - t2[3]
      
      return(list(group = group, CRL = resst, CRLcv = rescv, method = method, temps = temps))
    } else {
      # gglasso need consecutive number in group
      groupord <- order(group)
      group2 <- group[groupord]
      
      ##### Group Lasso
      require("gglasso")
      resgg <- gglasso(X[, groupord], y, group2, ...)
      
      resgg$beta <- resgg$beta[order(groupord), ]
      resgg$group <- group
      
      return(list(group = group, CRL = resgg, method = method))
    }    
  }  
}
