set.seed(155776)
library("bootUR")

#### NOTE: performing the simulation study with the settings as below/in the
#### paper, is likely to take several hours.

#### Auxiliary Functions ####
evaluate <- function(selected_orders, true_orders){
  # Proportion of units correctly classified
  CC <- sum(selected_orders==true_orders)/length(true_orders)

  # Proportion of units incorrectly found as stationary
  size <- sum((selected_orders==0 & true_orders>0))/sum(true_orders>0)

  # Proportion of units correctly found as stationary
  # Can only be computed if there are stationary time series
  if(sum(true_orders==0)!=0){
    power <- sum((selected_orders==0 & true_orders==0))/sum(true_orders==0)
  }else{
    power <- NA
  }
  out <- c(CC, size, power)
  return(out)
}

# Simulate from a standardized noncentral t distribution
sim_nct <- function(n, df, ncp = 0, sd = 1) {
  # Ratio of gamma functions appearing in mean and variance
  gamma_ratio <- gamma((df - 1) / 2) / gamma(df / 2)
  # Mean of noncentral t
  m <- ncp * sqrt(df/2) * gamma_ratio
  # Variance of noncentral t
  v <- df * (1 + ncp^2) / (df - 2) - df * ncp^2 * gamma_ratio^2 / 2
  # Simulate from noncentral t with mean 0 and standard deviation sd
  x <- sd * (rt(n, df = df, ncp = ncp) - m) / sqrt(v)
  return(x)
}

#### General Settings ####
q0 <- c(0, 0.2, 0.5, 0.9) # Proportion of stationary time series
M <- 200 # number of MC simulations
df <- 4 # Degrees of freedom noncentral t distribution
ncp <- 10 # Noncentrality parameter t distribution

#### Micro panel ####
n <- 25 # Time series length
N <- 200 # Number of variables
ar <- 0.6 # AR parameter for stationary time series



#### Store Results ####
micro_metrics <- array(NA, c(length(q0), 3, 4, M),
                 dimnames = list(paste("q0=", q0),
                                 c("correct", "size", "power"),
                                 c("adf", "boot_ur", "boot_sqt", "boot_fdr"),
                                 paste("sim", 1:M)))

#### Start simulation ####
for(f in 1:length(q0)){
  for(sim in 1:M){
    Y <- matrix(NA, ncol = N, nrow = n)
    nbr_I0 <- round(q0[f]*N) # number of I(0) series
    nbr_I1 <- N - nbr_I0 # number of I(1) series
    correct_orders <- c(rep(0, nbr_I0), rep(1, nbr_I1))

    #### Simulate I(0) series ####
    if(nbr_I0!=0){
      for(i0 in 1:nbr_I0){
        Y[, i0] <- arima.sim(n = n, n.start = round(n/2), list(ar = ar),
                             rand.gen = sim_nct, df = df, ncp = ncp)
      }
    }

    #### Simulate I(1) series ####
    for(i1 in 1:nbr_I1){
      Y[, nbr_I0 + i1] <- cumsum(sim_nct(n, df = df, ncp = ncp))
    }


    #### Unit Root Tests ###
    out_adf <- order_integration(Y, method = "adf", max_order = 1)
    out_ur <- order_integration(Y, method = "boot_ur", show_progress = F,
                                B = 499, max_order = 1)
    out_sqt <- order_integration(Y, method = "boot_sqt", show_progress = F,
                                 B = 499, steps = 0:8/8, max_order = 1)
    out_fdr <- order_integration(Y, method = "boot_fdr", show_progress = F,
                                 B = 499, max_order = 1)

    #### Evaluation Criteria ####
    metrics_adf <- evaluate(out_adf$order_int, correct_orders)
    metrics_ur <- evaluate(out_ur$order_int, correct_orders)
    metrics_sqt <- evaluate(out_sqt$order_int, correct_orders)
    metrics_fdr <- evaluate(out_fdr$order_int, correct_orders)
    micro_metrics[f, , , sim] <- cbind(metrics_adf, metrics_ur, metrics_sqt,
                                       metrics_fdr)
  }
}
micro_results <- apply(micro_metrics, c(1,2,3), mean)


################################################################################
#### Macro panel ####
n <- 100 # Time series length
N <- 50 # Number of variables
ar <- 0.9 # AR parameter for stationary time series

#### Store Results ####
macro_metrics <- array(NA, c(length(q0), 3, 4, M),
                       dimnames = list(
                         paste("q0=", q0),
                         c("correct", "size", "power"),
                         c("adf", "boot_ur", "boot_sqt", "boot_fdr"),
                         paste("sim", 1:M)))

#### Start simulation ####
for(f in 1:length(q0)){
  for(sim in 1:M){
    Y <- matrix(NA, ncol = N, nrow = n)
    nbr_I0 <- round(q0[f]*N) # number of I(0) series
    nbr_I1 <- N - nbr_I0 # number of I(1) series
    correct_orders <- c(rep(0, nbr_I0), rep(1, nbr_I1))

    #### Simulate I(0) series ####
    if(nbr_I0!=0){
      for(i0 in 1:nbr_I0){
        Y[, i0] <- arima.sim(n = n, n.start = round(n/2), list(ar = ar))
      }
    }

    #### Simulate I(1) series ####
    for(i1 in 1:nbr_I1){
      Y[, nbr_I0 + i1] <- cumsum(rnorm(n))
    }


    #### Unit Root Tests ###
    out_adf <- order_integration(Y, method = "adf", max_order = 1)
    out_ur <- order_integration(Y, method = "boot_ur", show_progress = F,
                                B = 499, max_order = 1)
    out_sqt <- order_integration(Y, method = "boot_sqt", show_progress = F,
                                 B = 499, steps = 0:4/4, max_order = 1)
    out_fdr <- order_integration(Y, method = "boot_fdr", show_progress = F,
                                 B = 499, max_order = 1)

    #### Evaluation Criteria ####
    metrics_adf <- evaluate(out_adf$order_int, correct_orders)
    metrics_ur <- evaluate(out_ur$order_int, correct_orders)
    metrics_sqt <- evaluate(out_sqt$order_int, correct_orders)
    metrics_fdr <- evaluate(out_fdr$order_int, correct_orders)
    macro_metrics[f, , , sim] <- cbind(metrics_adf, metrics_ur, metrics_sqt,
                                       metrics_fdr)
  }
}
macro_results <- apply(macro_metrics, c(1,2,3), mean)

################################################################################

#### High-dimensional panel ####
n <- 100 # Time series length
N <- 100 # Number of variables
ar <- 0.9 # AR parameter for stationary time series

#### Store Results ####
highdim_metrics <- array(NA, c(length(q0), 3, 4, M),
                         dimnames = list(
                           paste("q0=", q0), c("correct", "size", "power"),
                           c("adf", "boot_ur", "boot_sqt", "boot_fdr"),
                           paste("sim", 1:M)))

#### Start simulation ####
for(f in 1:length(q0)){
  for(sim in 1:M){
    Y <- matrix(NA, ncol = N, nrow = n)
    nbr_I0 <- round(q0[f]*N) # number of I(0) series
    nbr_I1 <- N - nbr_I0 # number of I(1) series
    correct_orders <- c(rep(0, nbr_I0), rep(1, nbr_I1))

    #### Simulate I(0) series ####
    if(nbr_I0!=0){
      for(i0 in 1:nbr_I0){
        Y[, i0] <- arima.sim(n = n, n.start = round(n/2), list(ar = ar))
      }
    }

    #### Simulate I(1) series ####
    for(i1 in 1:nbr_I1){
      Y[, nbr_I0 + i1] <- cumsum(rnorm(n))
    }

    #### Unit Root Tests ###
    out_adf <- order_integration(Y, method = "adf", max_order = 1)
    out_ur <- order_integration(Y, method = "boot_ur", show_progress = F,
                                B = 499, max_order = 1)
    out_sqt <- order_integration(Y, method = "boot_sqt", show_progress = F,
                                 B = 499, steps = 0:8/8, max_order = 1)
    out_fdr <- order_integration(Y, method = "boot_fdr", show_progress = F,
                                 B = 499, max_order = 1)

    #### Evaluation Criteria ####
    metrics_adf <- evaluate(out_adf$order_int, correct_orders)
    metrics_ur <- evaluate(out_ur$order_int, correct_orders)
    metrics_sqt <- evaluate(out_sqt$order_int, correct_orders)
    metrics_fdr <- evaluate(out_fdr$order_int, correct_orders)
    highdim_metrics[f, , , sim] <- cbind(metrics_adf, metrics_ur, metrics_sqt,
                                         metrics_fdr)
  }
}
highdim_results <- apply(highdim_metrics, c(1,2,3), mean)


Table6 <- rbind(round(cbind(t(micro_results[1,,]), t(micro_results[2,,]), 
							t(micro_results[3,,]), t(micro_results[4,,])), 2),
                round(cbind(t(macro_results[1,,]), t(macro_results[2,,]), 
                			t(macro_results[3,,]), t(macro_results[4,,])), 2),
                round(cbind(t(highdim_results[1,,]), t(highdim_results[2,,]), 
                			t(highdim_results[3,,]), t(highdim_results[4,,])), 2))
print(Table6)
