
##############################################################################
#       ###  Computing standard deviation of imbalances ###                  #
##############################################################################  
#   (1) Three aspects of imbalances are calculated:                          #
#        (i) the overall, within-covariate-margin, and within-stratum levels #
#   (2) Seven randomization procedures are considered:                       # 
#        (i) Hu and Hu's genenral covariate-adaptive randomization procedure #
#        (ii) Pocock and Simon's minimization                                #
#        (iii) Stritified biased coin design                                 #
#        (iv) Stratified permuted block randomziation                        #
#        (v) Atkinson's D_A-optimal biased coin design                       #
#        (vi) Covariate-adjusted biased coin design                          #
#                                                                            #
#   Note that calculation for standard deviation of imbalances with complete #
# randomization is realized by the function Std_CR, while those are realized #
# by the function Std by specifying the parameter method.                    #  
##############################################################################

Std_CR = function(nvec, cov_num = 2, level_num = c(2, 2), 
                   pr = rep(0.5, 4), N = 1000){
  leng = length(nvec); 
  STD = matrix(NA, nrow = 1 + Rprod(level_num) + sum(level_num), ncol = leng); 
  for(i in 1 : leng){
    n = nvec[i]; 
    res = evalSimRand(cov_num, level_num, pr, n, N, bsize = 4); 
    DIF = res[5, 1][[1]]; 
    STD[, i] = sqrt(apply(DIF, 1, var)); 
  }
  colnames(STD) = paste("n", nvec, sep = "="); 
  rownames(STD) = nameString(cov_num, level_num, Rprod(level_num), "All", "Real")
  return(STD); 
}

Std = function(nvec, cov_num = 2, level_num = c(2, 2), 
                pr = rep(0.5, 4), method, ..., N = 1000){
  leng = length(nvec); 
  STD = matrix(NA, nrow = 1 + Rprod(level_num) + sum(level_num), ncol = leng); 
  for(i in 1 : leng){
    n = nvec[i]; 
    res = carat::evalRand.sim(n, N, TRUE, cov_num, level_num, pr, method, ...); 
    DIF = res$DIF; 
    STD[, i] = sqrt(apply(DIF, 1, var));
  }
  colnames(STD) = paste("n", nvec, sep = "="); 
  rownames(STD) = nameString(cov_num, level_num, Rprod(level_num), "All", "Real")
  return(STD); 
}


Preprocess <- function(data){
  return(Preprocess(data))
}


HuHuCAR_BT_In <- function(data, B, omega, p){
  return(HuHuCAR_BT_In(data, B, omega, p))
}

HuHuCAR_getData <- function(n, cov_num, level_num, pr,
                            type, beta, mu1, mu2, 
                            sigma, omega, p){
  return(HuHuCAR_getData(n, cov_num, level_num, pr, type, beta, mu1, mu2, 
                         sigma, omega, p))
}

#############################
## Ma et al. (2020) in JASA##
#############################
# It can be loaded from 
# https://www.tandfonline.com/doi/suppl/10.1080/01621459.2019.1635483?scroll=top


# Randomization part

phi <- function(x){(1-x)^2/((1-x)^2+(1+x)^2)}

Atkinson <- function(XZ) # Atkinson (1982)
{
  T_ls=rep(NA, dim(XZ)[1])
  burn_in=10
  T_ls[1:burn_in]=c(1,0)
  
  F=cbind(1,XZ)
  
  for (i in (burn_in+1):length(T_ls))
  {
    
    b_i=t(as.matrix(2*T_ls[1:(i-1)]-1))%*%F[1:(i-1),]
    D_i=F[i,]%*%MASS::ginv(t(F[1:(i-1),])%*%F[1:(i-1),])%*%t(b_i)
    
    p_i=phi(D_i)
    
    T_ls[i]=rbinom(1, 1, p_i)
    
    
  }
  return(T_ls)
}

# Power

# all functions


HH <- function(XZ)
{
  n=dim(XZ)[1] # number of patients
  p=dim(XZ)[2] # number of covariates
  T_ls=rep(NA,n)
  num_stratum=2^p
  burn_in=10
  T_ls[1:burn_in]=sample(c(1,0),burn_in,replace=TRUE)
  
  D_overall=0
  D_margin =array(0,dim=  c(2,p))
  D_stratum=array(0,dim=rep(2,p))
  #D_stratum=array(1:2^p,dim=rep(2,p))
  D_overall=sum(T_ls[1:burn_in]==1)-sum(T_ls[1:burn_in]==0)
  for (i in 1:burn_in)
  {
    idx=XZ[i,]+1
    if (T_ls[i]==1)
    {
      D_stratum[t(as.matrix(idx))]=D_stratum[t(as.matrix(idx))]+1
      D_margin[cbind(idx,1:p)]=D_margin[cbind(idx,1:p)]+1
    }
    if (T_ls[i]==0)
    {
      D_stratum[t(as.matrix(idx))]=D_stratum[t(as.matrix(idx))]-1 
      D_margin[cbind(idx,1:p)]=D_margin[cbind(idx,1:p)]-1
    }
  }
  #D_overall
  #D_margin
  #cbind(XZ,T_ls)[1:burn_in,]
  #D_stratum
  
  for (i in (burn_in+1):n)
  {
    D_overall_ifgoT1=D_overall_ifgoT0=D_overall
    D_margin_ifgoT1 =D_margin_ifgoT0 =D_margin
    D_stratum_ifgoT1=D_stratum_ifgoT0=D_stratum
    
    idx=XZ[i,]+1
    if (1) # if this patient goes to T1
    {
      D_overall_ifgoT1=D_overall_ifgoT1+1
      D_stratum_ifgoT1[t(as.matrix(idx))]=D_stratum_ifgoT1[t(as.matrix(idx))]+1
      D_margin_ifgoT1[cbind(idx,1:p)]=D_margin_ifgoT1[cbind(idx,1:p)]+1
    }
    if (1) # if this patient goes to T0
    {
      D_overall_ifgoT0=D_overall_ifgoT0-1
      D_stratum_ifgoT0[t(as.matrix(idx))]=D_stratum_ifgoT0[t(as.matrix(idx))]-1 
      D_margin_ifgoT0[cbind(idx,1:p)]=D_margin_ifgoT0[cbind(idx,1:p)]-1
    }
    # equal weights
    #w_overall=1/(p+2)
    #w_margin=rep(1/(p+2),p)
    #w_stratum=1/(p+2)
    if (0) # Hu and Hu AOS page 1798 example 1's weights
    {
      w_overall=1/3
      w_margin=rep(1/3/p,p)
      w_stratum=1/3
    }
    if (0) # PS, special case of Hu and Hu AOS
    {
      w_overall=0
      w_margin=rep(1/p,p)
      w_stratum=0
    }
    if (0) # Hu and Hu AOS page 1800 remark 3.3
    {
      w_overall=0.1
      w_margin=c(0.4,0.4)
      w_stratum=0.1
    }
    if (1) # Hu and Hu AOS page 1800 good weights satisfying remark 3.3
    {
      w_overall=0.4
      w_margin=c(0.1,0.1)
      w_stratum=0.4
    }
    
    Imb_ifgoT1=w_overall*(D_overall_ifgoT1^2)+
      sum(w_margin*(D_margin_ifgoT1[cbind(idx,1:p)]^2))+
      w_stratum*(D_stratum_ifgoT1[t(as.matrix(idx))]^2)
    Imb_ifgoT0=w_overall*(D_overall_ifgoT0^2)+
      sum(w_margin*(D_margin_ifgoT0[cbind(idx,1:p)]^2))+
      w_stratum*(D_stratum_ifgoT0[t(as.matrix(idx))]^2)    
    if (Imb_ifgoT1<Imb_ifgoT0)
    {
      T_ls[i]=1*(runif(1)<0.75)
    }
    if (Imb_ifgoT1>Imb_ifgoT0)
    {
      T_ls[i]=1*(runif(1)<0.25)
    }
    if (Imb_ifgoT1==Imb_ifgoT0)
    {
      T_ls[i]=1*(runif(1)<0.5)
    }
    if (T_ls[i]==1) # if this patient is finally assigned to T1
    {
      D_overall=D_overall+1
      D_stratum[t(as.matrix(idx))]=D_stratum[t(as.matrix(idx))]+1
      D_margin[cbind(idx,1:p)]=D_margin[cbind(idx,1:p)]+1
    }
    if (T_ls[i]==0) # if this patient is finally assigned to T0
    {
      D_overall=D_overall-1
      D_stratum[t(as.matrix(idx))]=D_stratum[t(as.matrix(idx))]-1 
      D_margin[cbind(idx,1:p)]=D_margin[cbind(idx,1:p)]-1
    }
  }
  return(T_ls)
}


HHMaha <- function(XZ)
{
  n=dim(XZ)[1] # number of patients
  p=dim(XZ)[2] # number of covariates
  T_ls=rep(NA,n)
  num_stratum=2^p
  burn_in=10
  #T_ls[1:burn_in]=sample(c(1,0),burn_in,replace=TRUE)
  if (burn_in %% 2 == 0)
  {
    T_ls[1:burn_in]=rep(c(1,0),burn_in/2)
  } else {
    print("burn_in is not even")
    return(0)
  }
  for (i in (burn_in+1):n)
  {
    T_ls_try=T_ls
    # If the next patient goes to T1
    T_ls_try[i]=1  
    Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
    Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
    covX=cov(XZ[!is.na(T_ls_try),])
    pw=length(which(T_ls_try==1))/sum(!is.na(T_ls_try))
    M_tryT1= sum(!is.na(T_ls_try))*pw*(1-pw) * t(Xbar_T1-Xbar_T0)%*%solve(covX)%*%(Xbar_T1-Xbar_T0)
    
    # If the next patient goes to T0
    T_ls_try[i]=0  
    Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
    Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
    covX=cov(XZ[!is.na(T_ls_try),])
    pw=length(which(T_ls_try==1))/sum(!is.na(T_ls_try))
    M_tryT0= sum(!is.na(T_ls_try))*pw*(1-pw) * t(Xbar_T1-Xbar_T0)%*%solve(covX)%*%(Xbar_T1-Xbar_T0)
    
    # assign treatment
    if (M_tryT1<M_tryT0)
    {
      T_ls[i]=1*(runif(1)<0.75)
    }
    if (M_tryT1>M_tryT0)
    {
      T_ls[i]=1*(runif(1)<0.25)
    }
    if (M_tryT1==M_tryT0)
    {
      T_ls[i]=1*(runif(1)<0.5)
    }
  }
  return(T_ls)
}

Maha <- function(XZ,T_ls)
{
  Xbar_T1=colMeans(XZ[which(T_ls==1),])
  Xbar_T0=colMeans(XZ[which(T_ls==0),])
  covX=cov(XZ[!is.na(T_ls),])
  pw=length(which(T_ls==1))/sum(!is.na(T_ls))
  M= sum(!is.na(T_ls))*pw*(1-pw) * t(Xbar_T1-Xbar_T0)%*%solve(covX)%*%(Xbar_T1-Xbar_T0)
  return(M)
}

HHMaha_balance <- function(XZ)
{
  n=dim(XZ)[1] # number of patients
  p=dim(XZ)[2] # number of covariates
  T_ls=rep(NA,n)
  num_stratum=2^p
  burn_in=p+2+(p %% 2 != 0)*1
  #T_ls[1:burn_in]=sample(c(1,0),burn_in,replace=TRUE)
  if (burn_in %% 2 == 0)
  {
    T_ls[1:burn_in]=rep(c(1,0),burn_in/2)
  } else {
    print("burn_in is not even")
    return(0)
  }
  #for (i in (burn_in+1):(n-(n %% 2 != 0)*1)) # from burn_in+1 to the largest even number (<= n)
  for (i in (burn_in+1):n) # from burn_in+1 to the largest even number (<= n)
  {
    if (i %% 2 != 0)
    {
      T_ls_try=T_ls
      # If the next patient goes to T1
      T_ls_try[i]=1  
      T_ls_try[i+1]=0  
      Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
      Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
      covX=cov(XZ[!is.na(T_ls_try),])
      pw=length(which(T_ls_try==1))/sum(!is.na(T_ls_try))
      M_tryT1= sum(!is.na(T_ls_try))*pw*(1-pw) * t(Xbar_T1-Xbar_T0)%*%solve(covX)%*%(Xbar_T1-Xbar_T0)
      
      # If the next patient goes to T0
      T_ls_try[i]=0  
      T_ls_try[i+1]=1  
      Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
      Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
      covX=cov(XZ[!is.na(T_ls_try),])
      pw=length(which(T_ls_try==1))/sum(!is.na(T_ls_try))
      M_tryT0= sum(!is.na(T_ls_try))*pw*(1-pw) * t(Xbar_T1-Xbar_T0)%*%solve(covX)%*%(Xbar_T1-Xbar_T0)
      
      # assign treatment
      if (M_tryT1<M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.75)
      }
      if (M_tryT1>M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.25)
      }
      if (M_tryT1==M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.5)
      }
      T_ls[i+1]=ifelse(T_ls[i]==1,0,1)
    }
  }
  #  if (n %% 2 != 0)
  #  {
  #      T_ls[i+1]=1*(runif(1)<0.5)
  #  }
  return(T_ls)
}

CR <- function(N)
{
  T_ls=rep(NA,N)
  idx_ls=seq(1,N)
  for (n in 1:(N/2))
  {
    this_idx=sample(idx_ls,1)
    idx_ls=idx_ls[idx_ls!=this_idx]
    T_ls[this_idx]=1
  }
  T_ls[is.na(T_ls)]=0
  # for (n in ((N/2)+1):N)
  # {
  #     if (length(idx_ls)>1)
  #     {
  #         this_idx=sample(idx_ls,1)
  #         idx_ls=idx_ls[idx_ls!=this_idx]
  #         T_ls[this_idx]=0
  #     } else {
  #         T_ls[idx_ls]=0
  #     }
  # }
  return(T_ls)
}


CAM_optimized <- function(XZ)
{
  n=dim(XZ)[1] # number of patients
  p=dim(XZ)[2] # number of covariates
  T_ls=rep(NA,n)
  #burn_in=p+2+(p %% 2 != 0)*1
  burn_in=2
  if (burn_in %% 2 == 0)
  {
    T_ls[1:burn_in]=rep(c(1,0),burn_in/2)
  } else {
    print("burn_in is not even")
    return(0)
  }
  #for (i in (burn_in+1):(n-(n %% 2 != 0)*1)) # from burn_in+1 to the largest even number (<= n)
  covX=cov(XZ)
  invCovX=solve(covX)
  pw=0.5
  for (i in (burn_in+1):n) # from burn_in+1 to the largest even number (<= n)
  {
    if (i %% 2 != 0)
    {
      T_ls_try=T_ls
      # If the next patient goes to T1
      T_ls_try[i]=1  
      T_ls_try[i+1]=0  
      Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
      Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
      M_tryT1= sum(!is.na(T_ls_try))*pw*(1-pw) * t(Xbar_T1-Xbar_T0)%*%invCovX%*%(Xbar_T1-Xbar_T0)
      
      # If the next patient goes to T0
      T_ls_try[i]=0  
      T_ls_try[i+1]=1  
      Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
      Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
      M_tryT0= sum(!is.na(T_ls_try))*pw*(1-pw) * t(Xbar_T1-Xbar_T0)%*%invCovX%*%(Xbar_T1-Xbar_T0)
      
      # assign treatment
      if (M_tryT1<M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.75)
      }
      if (M_tryT1>M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.25)
      }
      if (M_tryT1==M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.5)
      }
      T_ls[i+1]=ifelse(T_ls[i]==1,0,1)
    }
  }
  return(T_ls)
}

HHMaha_balance_realdata <- function(XZ)
{
  n=dim(XZ)[1] # number of patients
  p=dim(XZ)[2] # number of covariates
  covX=cov(XZ)
  invcovX=solve(covX)
  pt=0.5
  T_ls=rep(NA,n)
  burn_in=2
  if (burn_in %% 2 == 0)
  {
    T_ls[1:burn_in]=rep(c(1,0),burn_in/2)
  } else {
    print("burn_in is not even")
    return(0)
  }
  for (i in (burn_in+1):n)
  {
    if (i %% 2 != 0)
    {
      #print(i)
      T_ls_try=T_ls
      # If the next patient goes to T1
      T_ls_try[i]=1  
      T_ls_try[i+1]=0  
      Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
      Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
      M_tryT1= sum(!is.na(T_ls_try))*pt*(1-pt) * t(Xbar_T1-Xbar_T0)%*%invcovX%*%(Xbar_T1-Xbar_T0)
      
      # If the next patient goes to T0
      T_ls_try[i]=0  
      T_ls_try[i+1]=1  
      Xbar_T1=colMeans(XZ[which(T_ls_try==1),])
      Xbar_T0=colMeans(XZ[which(T_ls_try==0),])
      M_tryT0= sum(!is.na(T_ls_try))*pt*(1-pt) * t(Xbar_T1-Xbar_T0)%*%invcovX%*%(Xbar_T1-Xbar_T0)
      
      # assign treatment
      if (M_tryT1<M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.75)
      }
      if (M_tryT1>M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.25)
      }
      if (M_tryT1==M_tryT0)
      {
        T_ls[i]=1*(runif(1)<0.5)
      }
      T_ls[i+1]=ifelse(T_ls[i]==1,0,1)
    }
  }
  return(T_ls)
}

cv_boot_fn <- function(Y,XZ,method="")
{
  N=dim(XZ)[1]
  tval_XZ_CR_B=rep(NA,B)
  tval_X_CR_B =rep(NA,B)
  tval_Z_CR_B =rep(NA,B)
  tval_0_CR_B =rep(NA,B)
  tval_XZ_RR_B=rep(NA,B)
  tval_X_RR_B =rep(NA,B)
  tval_Z_RR_B =rep(NA,B)
  tval_0_RR_B =rep(NA,B)
  tval_XZ_HH_B=rep(NA,B)
  tval_X_HH_B =rep(NA,B)
  tval_Z_HH_B =rep(NA,B)
  tval_0_HH_B =rep(NA,B)
  tval_XZ_AT_B=rep(NA,B)
  tval_X_AT_B =rep(NA,B)
  tval_Z_AT_B =rep(NA,B)
  tval_0_AT_B =rep(NA,B)
  for (b in 1:B)
  {
    #idx=sample(1:N,replace=TRUE)
    #Y_b=Y[idx]
    #XZ_b=XZ[idx,]
    Y_b=Y
    XZ_b=XZ
    X_b=XZ_b[,1:p]
    Z_b=XZ_b[,(p+1):(p+q)]
    if (method=="CR")
    {
      T_CR_b=rbinom(N,1,0.5) #Complete randomization
      fit_XZ_CR <- lm(Y_b ~ XZ_b + T_CR_b)
      tval_XZ_CR_B[b]=summary(fit_XZ_CR)$coef[p+q+2,3]
      fit_X_CR <- lm(Y_b ~ X_b + T_CR_b)
      tval_X_CR_B[b]=summary(fit_X_CR)$coef[p+2,3]
      fit_Z_CR <- lm(Y_b ~ Z_b + T_CR_b)
      tval_Z_CR_B[b]=summary(fit_Z_CR)$coef[q+2,3]
      fit_0_CR <- lm(Y_b ~ T_CR_b)
      tval_0_CR_B[b]=summary(fit_0_CR)$coef[2,3]
    }
    if (method=="RR")
    {
      T_RR_b=sample(c(rep(0,N/2),rep(1,N/2)),N) # RR
      M_RR=Maha(XZ_b,T_RR_b)
      RR_it=1
      a=3
      while ((M_RR>a)&(RR_it<1500))
      {
        T_RR_b=sample(c(rep(0,N/2),rep(1,N/2)),N) #RR
        M_RR=Maha(XZ_b,T_RR_b)
        RR_it=RR_it+1
      }
      fit_XZ_RR <- lm(Y_b ~ XZ_b + T_RR_b)
      tval_XZ_RR_B[b]=summary(fit_XZ_RR)$coef[p+q+2,3]
      fit_X_RR <- lm(Y_b ~ X_b + T_RR_b)
      tval_X_RR_B[b]=summary(fit_X_RR)$coef[p+2,3]
      fit_Z_RR <- lm(Y_b ~ Z_b + T_RR_b)
      tval_Z_RR_B[b]=summary(fit_Z_RR)$coef[q+2,3]
      fit_0_RR <- lm(Y_b ~ T_RR_b)
      tval_0_RR_B[b]=summary(fit_0_RR)$coef[2,3]
    }
    if (method=="HH")
    {
      T_HH_b=CAM_optimized(XZ_b) #Hu and Hu AOS
      fit_XZ_HH <- lm(Y_b ~ XZ_b + T_HH_b)
      tval_XZ_HH_B[b]=summary(fit_XZ_HH)$coef[p+q+2,3]
      fit_X_HH <- lm(Y_b ~ X_b + T_HH_b)
      tval_X_HH_B[b]=summary(fit_X_HH)$coef[p+2,3]
      fit_Z_HH <- lm(Y_b ~ Z_b + T_HH_b)
      tval_Z_HH_B[b]=summary(fit_Z_HH)$coef[q+2,3]
      fit_0_HH <- lm(Y_b ~ T_HH_b)
      tval_0_HH_B[b]=summary(fit_0_HH)$coef[2,3]
    }
    if (method=="AT")
    {
      T_AT_b=Atkinson(XZ_b) #Atkinson
      fit_XZ_AT <- lm(Y_b ~ XZ_b + T_AT_b)
      tval_XZ_AT_B[b]=summary(fit_XZ_AT)$coef[p+q+2,3]
      fit_X_AT <- lm(Y_b ~ X_b + T_AT_b)
      tval_X_AT_B[b]=summary(fit_X_AT)$coef[p+2,3]
      fit_Z_AT <- lm(Y_b ~ Z_b + T_AT_b)
      tval_Z_AT_B[b]=summary(fit_Z_AT)$coef[q+2,3]
      fit_0_AT <- lm(Y_b ~ T_AT_b)
      tval_0_AT_B[b]=summary(fit_0_AT)$coef[2,3]
    }
    
  }
  return(list(
    tval_XZ_CR_B=tval_XZ_CR_B,
    tval_X_CR_B =tval_X_CR_B ,
    tval_Z_CR_B =tval_Z_CR_B ,
    tval_0_CR_B =tval_0_CR_B ,
    tval_XZ_RR_B=tval_XZ_RR_B,
    tval_X_RR_B =tval_X_RR_B ,
    tval_Z_RR_B =tval_Z_RR_B ,
    tval_0_RR_B =tval_0_RR_B ,
    tval_XZ_HH_B=tval_XZ_HH_B,
    tval_X_HH_B =tval_X_HH_B ,
    tval_Z_HH_B =tval_Z_HH_B ,
    tval_0_HH_B =tval_0_HH_B ,
    tval_XZ_AT_B=tval_XZ_AT_B,
    tval_X_AT_B =tval_X_AT_B ,
    tval_Z_AT_B =tval_Z_AT_B ,
    tval_0_AT_B =tval_0_AT_B 
  ))
  
}



Smith <- function(XZ,Q) # Smith (1984ab)
  # Q = lim (t(cbind(1,XZ))%*%cbind(1,XZ))/N
{
  T_ls=rep(NA,dim(XZ)[1])
  burn_in=10
  T_ls[1:burn_in]=c(1,0)
  
  F=cbind(1,XZ)
  
  for (i in (burn_in+1):length(T_ls))
  {
    
    b_i=t(as.matrix(2*T_ls[1:(i-1)]-1))%*%F[1:(i-1),]
    D_i=F[i,]%*%solve(Q)%*%t(b_i)/(i-1)
    
    p_i=phi(D_i)
    
    T_ls[i]=rbinom(1, 1, p_i)
    
    
  }
  return(T_ls)
}



BR <- function(n,block.size=4) # Permuted block design
{
  block.num=ceiling(n/block.size)
  cards=NULL
  i=1
  while(i<=block.num)
  {
    cards=c(cards,sample(cbind(rep(1,block.size/2),rep(0,block.size/2)),block.size))
    i=i+1
  }
  cards=cards[1:n]
  return(cards)
} 

SB <- function(XZ) # Stratified block design for 4 binary covariates coded as 0 or 1
{
  x=XZ[,1]*8+XZ[,2]*4+XZ[,3]*2+XZ[,4]+1
  m=2*2*2*2
  
  tr=rep(NA,dim(XZ)[1])
  i=1
  while(i<=m)
  {
    tr[x==i]=BR(length(x[x==i]))
    i=i+1
  }
  return(tr) 
}

# NOTE: The Pocock and Simon procedure is written in C and called by the R function PS 
# Compile the file PocockSimon.cpp to a shared library with the below command in UNIX
# R CMD SHLIB PocockSimon.cpp

condvar <- function(x,cutoff) # Calculate E[Var{X|D(X)}]
{
  x1=x[x>cutoff]
  x2=x[x<=cutoff]
  
  res=var(x1)*length(x1)+var(x2)*length(x2)
  res=res/length(x)
  return(res)
}

B = 10; p = 2; q = 2; N = 10
n = 10; u0 = 0; u1 = 0.7
betagamma = c(0.1, 0.4, 0.3, 0.1)
it_max = 25600
tval_XZ_HH=rep(NA, it_max)
pval_XZ_HH=rep(NA, it_max)

cal_fn <- function(it)
{
  X=matrix(rnorm(N*p),N,p)
  Z=matrix(rnorm(N*q),N,q)
  XZ=cbind(X,Z)
  Y0=u0+XZ%*%betagamma+rnorm(N)*2
  Y1=u1+XZ%*%betagamma+rnorm(N)*2
  
  T_ls=CAM_optimized(XZ) #Hu and Hu AOS
  Y=ifelse(T_ls==0,Y0,Y1)
  fit_XZ <- lm(Y ~ XZ+T_ls)
  tval_XZ_HH[it]=summary(fit_XZ)$coef[dim(XZ)[2]+2,3]
  pval_XZ_HH[it]=summary(fit_XZ)$coef[dim(XZ)[2]+2,4]
  
  # asymptotic distribution under CAM
  var_eps = var(fit_XZ$res)
  betagamma_hat = summary(fit_XZ)$coef[2:(p+q+1),1]
  beta_hat = betagamma_hat[1:p]
  gamma_hat = betagamma_hat[(p+1):(p+q)]
  var_XZ = apply(XZ,2,var)
  alpha_1_XZ = sqrt( var_eps / ( var_eps ) ) # is always 1
  pval_XZ_HH[it] = 2*pnorm(-abs(tval_XZ_HH[it]),mean = 0, sd = alpha_1_XZ)
  
  # bootstrap p values
  tvals=cv_boot_fn(Y,XZ,method="HH")
  pval_XZ_HH[it]=(sum(abs(tval_XZ_HH[it])<tvals$tval_XZ_HH_B)+sum(-abs(tval_XZ_HH[it])>tvals$tval_XZ_HH_B))/B
  
  cat("u1=",u1,"N=",N,"it=",it,"\n")
  return(list(
    # t test statistics
    tval_XZ_HH_this=tval_XZ_HH[it],
    
    # p values from bootstrap or traditional test
    pval_XZ_HH_this=pval_XZ_HH[it]
  ))
}