# Replication script for "ARCHModels.jl: Estimating ARCH models in Julia"
# Run in Julia 1.8.5 using `include("replication_script")`

# create Figures subdirectory
const FIG_PATH = joinpath("..", "Figures")
mkpath(FIG_PATH)

# install dependencies
using Pkg

cd(mktempdir()) do
    Pkg.activate(".")
	Pkg.add([PackageSpec(name="BenchmarkTools", version="1.3.2"),
			 PackageSpec(name="Plots", version="1.38.4"),
			 PackageSpec(name="StatsPlots", version="0.15.4"),
			 PackageSpec(name="TimeSeries", version="0.23.1"),
			 PackageSpec(name="MarketData", version="0.13.12"),
			 PackageSpec(name="KernelDensity", version="0.6.5"),
			 PackageSpec(name="Distributions", version="0.25.80"),
			 PackageSpec(name="ARCHModels", version="2.3.3")
			 ])
    Pkg.status()
end

# setup
using BenchmarkTools, Plots, StatsPlots, MarketData, TimeSeries,
	  KernelDensity, Distributions, ARCHModels
gr()
Plots.default(fontfamily="Computer Modern", linewidth=2,
 			  framestyle=:box, label=nothing, grid=false, thickness_scaling=1,
			  titlefontsize=18,
			  tickfontsize=14,
			  legendfontsize=12,
			  guidefontsize=14
			  )

wait_for_key(prompt) = (print(stdout, prompt); read(stdin, 1); nothing)

function backtest(windowsize)
	vars = similar(BG96)
	T = length(BG96)
	Threads.@threads for t = windowsize:T-1
		m = fit(GARCH{1, 1}, BG96[t-(windowsize-1):t]);
		vars[t+1] = predict(m, :VaR; level=0.05);
	end
	return vars
end

function backtest_nothreads(windowsize)
	vars = similar(BG96)
	T = length(BG96)
	for t = windowsize:T-1
		m = fit(GARCH{1, 1}, BG96[t-(windowsize-1):t]);
		vars[t+1] = predict(m, :VaR; level=0.05);
	end
	return vars
end

function main()
	# Figure 1
	println("Plotting Figure 1...")
	r = percentchange(MarketData.AAPL[Symbol("AdjClose")])
	data = values(r)
	p1 = plot(r, title="Volatility Clustering", legend=:none, ylabel="\$r_t\$", plotsize=(600, 400), left_margin = 5Plots.mm, right_margin = 5Plots.mm, guidefontsize=21, tickfontsize=17)
	p2 = plot(kde(data), label="Kernel Density", title="Heavy Tails", ylabel="\$f(r)\$", plotsize=(600, 400), left_margin = 5Plots.mm, guidefontsize=21, tickfontsize=17)
	plot!(fit(Normal, data), label="Fitted Normal")
	plot(p1, p2, size=(1300, 650))
	savefig(joinpath(FIG_PATH, "figure1.pdf"))
	println("Done.")
	println()
    # Figure 2
	println("Plotting Figure 2...")
	am = fit(GARCH{1, 1}, BG96)
	vars = VaRs(am, 0.05)
	plot(-BG96, legend=:none, xlabel="\$t\$", ylabel="\$-r_t\$", linewidth=1, guidefontsize=16)
	plot!(vars, color=:purple, linewidth=1, guidefontsize=16)
	savefig(joinpath(FIG_PATH, "figure2.pdf"))
	println("Done.")	
	# Estimation results in Section 5
	println()
	println("Estimation results in Section 5:")
	println()
	println("autocor:")
	println(autocor(BG96.^2, 1:4, demean=true)')
	println()
	println(ARCHLMTest(BG96, 1))
	println()
	println(fit(GARCH{1, 1}, BG96))
	println()
	T = length(BG96)
	windowsize = 1000
	time1 = @belapsed backtest_nothreads($windowsize)
	time2 = @belapsed backtest($windowsize)
	vars = backtest(windowsize)
	println("Fitted $(T-windowsize) models in $time1 seconds using 1 thread.")
	println("Fitted $(T-windowsize) models in $time2 seconds using $(Threads.nthreads()) threads.")
	println()
	println(DQTest(BG96[windowsize+1:end], vars[windowsize+1:end], 0.05))
	println(fit(DCC, DOW29))
	println(fit(DCC, DOW29; method=:twostep))
	# Benchmarks
	println()
	println("Benchmarks for Table 1:")
	@btime fit(GARCH{1, 1}, $BG96, meanspec=Intercept)
	println("Done.")
	println("")
	println()
	println("Benchmarks for Table 2:")
	@btime fit(DCC, DOW29)
	@btime fit(DCC, DOW29; method=:twostep)
	println("Done.")
	println("")
	wait_for_key("Press enter to exit.")

end


# check correct version and run script
println()
if !(VERSION == v"1.8.5")
	@warn("This code is tested with version 1.8.5; detected version $VERSION")
	print("Continue (y/n)? ")
	a = readline()
	if startswith(a, 'y')
		main()
	end
else
	main()
end
