#!/bin/bash
#SBATCH --job-name=markets-benchmark
#SBATCH --partition=fuchs
#SBATCH --nodes=1
#SBATCH --ntasks=1
#SBATCH --cpus-per-task=40
#SBATCH --mem-per-cpu=3072
#SBATCH --time=48:00:00
#SBATCH --no-requeue
#SBATCH --mail-type=NONE
#SBATCH --account=agmisc

# The data were produced with `./benchmark.sh full`
# Use `./benchmark.sh test` for a small-scale replication
export MODE="$1"
echo "Starting in '$MODE' mode execution."

# Benchmarked models
declare -a MODEL_ARRAY=(diseq_stochastic_adjustment\
			    diseq_deterministic_adjustment\
			    diseq_basic\
			    equilibrium_model\
			    diseq_directional)
# Number of processes to be simultaneously executed
export PROCS=`expr $(nproc --all) - 1`
# Fixed Nobs exponent when benchmarking over parameters
export FNEXP=12

if [[ $MODE == "full" ]]
then
  spack load gsl r-cairo
    
  # Repetitions for each benchmark
  export REPS=100
  # Nobs exponents used in benchmarks over observation 
  declare -a N_ARRAY=(`seq 14 -1 1`)
  # Number of extra parameters used in benchmarks over parameters
  declare -a P_ARRAY=(`seq  14 -1 1`)
elif [[ $MODE == "test" ]]
then
  export REPS=4
  declare -a N_ARRAY=(`seq 5 -1 1`)
  declare -a P_ARRAY=(`seq 2 -1 1`)
else
  echo "ERROR: Unknown mode $MODE. Please use 'full' or 'test'."
  exit 1
fi

Rscript -e 'version'

declare -a TASKS=()

# output filename
filename="time_benchmarks_$(date +%Y-%m-%d_%H-%M-%S).rds"

# Benchmarks over the number of parameters
for ADDPARS in "${P_ARRAY[@]}"
do
  for MODEL in "${MODEL_ARRAY[@]}"
  do
    ARGS="--model=$MODEL --nexp=$FNEXP --reps=$REPS  --addpars=$ADDPARS --out=$filename"
    TASKS+=( "echo $ARGS; Rscript benchmark.R $ARGS;" )
  done
done

# Benchmarks over the number of observations
for NEXP in "${N_ARRAY[@]}"
do
  for MODEL in "${MODEL_ARRAY[@]}"
  do
    ARGS="--model=$MODEL --nexp=$NEXP --reps=$REPS --out=$filename"
    TASKS+=( "echo $ARGS; Rscript benchmark.R $ARGS;" )
  done
done

printf '%s\n' "${TASKS[@]}" | xargs --max-procs=$PROCS -n 1 -I {} bash -c '{}'

# Wait for all child processes to terminate.
wait
