# run generate.r before to create data

library(holiglm)
library(abess)
library(bestglm)
library(data.table)

result_dir <- normalizePath("results")
dir.create(result_dir, showWarnings = FALSE, recursive = TRUE)

gurobi_installed <- ("ROI.plugin.gurobi" %in% ROI::ROI_installed_solvers())
mosek_installed <- ("ROI.plugin.mosek" %in% ROI::ROI_installed_solvers())

# hglm_seq 

## gaussian data
gauss <- readRDS('data/gauss.rds')
result <- data.table(i=seq_along(gauss))

# warmup solver
hglm(mpg~., data=mtcars, solver="ecos")
if (gurobi_installed) hglm(mpg~., data=mtcars, solver="gurobi")

message("Gauss:")
for (i in seq_along(gauss)) {
    message(i/length(gauss)*100, "%")
    data <- gauss[[i]]
    x <- model.matrix(y~.-1, data=data)
    y <- data[["y"]]
    xy <- as.data.frame(cbind(x, y))

    time_abess <- system.time(model_abess <- abess(x, y, family="gaussian", tune.path="sequence", tune.type="aic"))[[3]]
    time_bestglm <- system.time(model_bestglm <- bestglm(xy, family=gaussian, IC="AIC"))[[3]]

    set(result, i=i, j="p", value=ncol(data)-1L)
    set(result, i=i, j="k", value=sum(attr(data, "true_beta"))-1L)

    set(result, i=i, j="time_abess",   value=time_abess)
    set(result, i=i, j="time_bestglm", value=time_bestglm)
    
    set(result, i=i, j="model_abess",   value=list(model_abess))
    set(result, i=i, j="model_bestglm", value=list(model_bestglm))

    if (gurobi_installed) {
        time_hglm <- system.time(model_hglm <- hglm_seq(formula=y~., family=gaussian(), data=data, solver="gurobi"))[[3]]
        set(result, i=i, j="time_hglm_gurobi", value=time_hglm)
        set(result, i=i, j="model_hglm_gurobi", value=list(model_hglm))
    }

    if (ncol(data) <= 21) {
        time_hglm <- system.time(model_hglm <- hglm_seq(formula=y~., family=gaussian(), data=data, solver="ecos"))[[3]]
        set(result, i=i, j="time_hglm_ecos", value=time_hglm)
        set(result, i=i, j="model_hglm_ecos",    value=list(model_hglm))
    }
}

ofi <- file.path(result_dir, "gauss_bss.rds")
saveRDS(result, ofi)

## binomial data
binomial <- readRDS('data/binomial.rds')
result <- data.table(i=seq_along(binomial))

# warmup solver
hglm(group ~ ., data=PlantGrowth, family=binomial(), solver="ecos")
if (mosek_installed) hglm(group ~ ., data=PlantGrowth, family=binomial(), solver="mosek")

message("Binomial:")
for (i in seq_along(binomial)) {
    message(i/length(binomial)*100, "%")
    data <- binomial[[i]]
    if (ncol(data) > 21) next
    x <- model.matrix(y~.-1, data=data)
    y <- data[["y"]]
    xy <- as.data.frame(cbind(x, y))

    time_hglm <-  system.time(model_hglm <- hglm_seq(formula=y~., family=binomial(), data=data, solver="ecos"))[[3]]
    time_abess <- system.time(model_abess <- abess(x, y, family="binomial", tune.path="sequence", tune.type="aic"))[[3]]

    set(result, i=i, j="p", value=ncol(data)-1L)
    set(result, i=i, j="k", value=sum(attr(data, "true_beta"))-1L)

    set(result, i=i, j="time_hglm_ecos", value=time_hglm)
    set(result, i=i, j="time_abess",     value=time_abess)
    
    set(result, i=i, j="model_hglm_ecos", value=list(model_hglm))
    set(result, i=i, j="model_abess",     value=list(model_abess))

    if (ncol(data) <= 16) {
        time_bestglm <- system.time(model_bestglm <- bestglm(xy, family=binomial(), IC="AIC"))[[3]]
        set(result, i=i, j="time_bestglm", value=time_bestglm)
        set(result, i=i, j="model_bestglm", value=list(model_bestglm))
    }

    if (mosek_installed) {
        time_hglm <-  system.time(model_hglm <- hglm_seq(formula=y~., family=binomial(), data=data, solver="mosek"))[[3]]
        set(result, i=i, j="time_hglm_mosek", value=time_hglm)
        set(result, i=i, j="model_hglm_mosek", value=list(model_hglm))
    }
}

ofi <- file.path(result_dir, "binomial_bss.rds")
saveRDS(result, ofi)


## poisson data
poisson <- readRDS('data/poisson.rds')
result <- data.table(i=seq_along(poisson))

# warmup solver
if (mosek_installed) hglm(Assault ~ ., data=USArrests, family=poisson(), solver="mosek")

message("Poisson:")
for (i in seq_along(poisson)) {
    message(i/length(poisson)*100, "%")
    data <- poisson[[i]]
    if (ncol(data) > 21) next
    x <- model.matrix(y~.-1, data=data)
    y <- data[["y"]]
    xy <- as.data.frame(cbind(x, y))

    time_hglm <-  system.time(model_hglm <- hglm_seq(formula=y~., family=poisson(), data=data, solver="ecos", control=list(maxit=1e3L)))[[3]]
    time_abess <- system.time(model_abess <- abess(x, y, family="poisson", tune.path="sequence", tune.type="aic"))[[3]]

    set(result, i=i, j="p", value=ncol(data)-1L)
    set(result, i=i, j="k", value=sum(attr(data, "true_beta"))-1L)

    set(result, i=i, j="time_hglm_ecos", value=time_hglm)
    set(result, i=i, j="time_abess",     value=time_abess)
    
    set(result, i=i, j="model_hglm_ecos", value=list(model_hglm))
    set(result, i=i, j="model_abess",     value=list(model_abess))

    if (ncol(data) <= 16) {
        time_bestglm <- system.time(model_bestglm <- bestglm(xy, family=poisson(), IC="AIC"))[[3]]
        set(result, i=i, j="time_bestglm", value=time_bestglm)
        set(result, i=i, j="model_bestglm", value=list(model_bestglm))
    }

    if (mosek_installed) {
        time_hglm <-  system.time(model_hglm <- hglm_seq(formula=y~., family=poisson(), data=data, solver="mosek"))[[3]]
        set(result, i=i, j="time_hglm_mosek",  value=time_hglm)
        set(result, i=i, j="model_hglm_mosek", value=list(model_hglm))
    }
}

ofi <- file.path(result_dir, "poisson_bss.rds")
saveRDS(result, ofi)
