# run generate.r before to create data respectively sim_*.r to produce result data

library(holiglm)
library(abess)
library(restriktor)
library(ConsReg)
library(data.table)
library(kableExtra)
options(knitr.kable.NA = "--")

# copies solutions from other solvers if not installed to run smoothly
gurobi_installed <- ("ROI.plugin.gurobi" %in% ROI::ROI_installed_solvers())
mosek_installed <- ("ROI.plugin.mosek" %in% ROI::ROI_installed_solvers())

results_dir <- normalizePath("results")
supp_dir <- normalizePath("supp")
pn <- function(x,y,n=3) {
    s = paste0("%.", n, "f (%.", n,"f)")
    ifelse(is.na(x), NA_character_, sprintf(s, x, y))
}

latex_bf <- function(x) gsub("(\\S*) (.*)", "\\\\textbf{\\1} \\2", x)
nonneg <- function(x) all(round(coef(x), 4) >= 0)
split1 <- function(x) if (is.na(x)) x else gsub("(.*) (.*)", "\\1", x)
split2 <- function(x) if (is.na(x)) "" else gsub("(.*) (.*)", "\\2", x)

select_AIC <- function(x) UseMethod("select_AIC")

select_AIC.hglm_seq <- function(x) {
    idx <- which.min(lapply(x, AIC))
    x[[idx]]
}

select_AIC.abess <- function(x) {
    idx <- which.min(x[["tune.value"]])
    c(setNames(x[["intercept"]][idx], "(Intercept)"), x[["beta"]][,idx])
}

select_AIC.bestglm <- function(x) x$BestModel

active_coefficients.glm <- active_coefficients.lm <- function(x) round(coef(x), 4) != 0
name_coef <- function(x) names(acoef(x))[acoef(x)]

# similar to stats::glm.fit to create glm object for given coefficients
make_glm <- function(formula, coef, family, data) {
    linkinv <- family$linkinv
    mf <- model.frame(formula, data=data)
    x <- model.matrix(formula, mf)
    y <- model.response(mf)
    if (ncol(x) < length(coef)) x <- cbind(1, x)
    if (is.factor(y)) y <- as.integer(y) - 1L
    weights = rep.int(1L, NROW(x))
    ynames <- names(y)
    nobs <- NROW(y)
    nvars <- ncol(x)
    coef_indicators <- coef != 0
    is_active <- setNames(coef_indicators, colnames(x))
    n_active_vars <- sum(is_active)

    eta <- setNames(drop(x %*% coef), ynames)
    mu <- setNames(linkinv(eta), ynames)
    mu.eta.val <- family$mu.eta(eta)
    good <- mu.eta.val != 0
    residuals <- setNames((y - mu) / family$mu.eta(eta), ynames)
    dev <- sum(family$dev.resids(y, mu, weights))

    z <- eta[good] + (y - mu)[good] / mu.eta.val[good]
    w <- sqrt((weights[good] * mu.eta.val[good]^2) / family$variance(mu)[good])
    wt <- setNames(rep.int(0, nobs), ynames)
    wt[good] <- w^2
    wtdmu <- sum(weights * y) / sum(weights)
    nulldev <- sum(family$dev.resids(y, wtdmu, weights))
    n.ok <- nobs - sum(weights == 0)
    nulldf <- n.ok - 1L

    qr_tolerance <- 1e-07
    qr <- qr.default(x * w, qr_tolerance, LAPACK = FALSE)
    effects <- qr.qty(qr, z * w)
    qr$tol <- qr_tolerance

    nr <- min(sum(good), nvars)
    if (nr < nvars) {
        Rmat <- diag(nvars)
        Rmat[1L:nr, 1L:nvars] <- qr$qr[1L:nr, 1L:nvars]
    } else Rmat <- qr$qr[1L:nvars, 1L:nvars]
    Rmat <- as.matrix(Rmat)
    Rmat[row(Rmat) > col(Rmat)] <- 0
    dimnames(Rmat) <- list(colnames(qr$qr), colnames(qr$qr))

    rank <- sum(coef_indicators)

    aic.model <- family$aic(y, nobs, mu, weights, dev) + 2 * rank

    boundary <- NA 
    resdf <- n.ok - rank

    model <- list(coefficients = coef, residuals = residuals, fitted.values = mu,
         effects = effects, R = Rmat, rank = rank, qr = qr,
         family = family, linear.predictors = eta, deviance = dev,
         aic = aic.model, null.deviance = nulldev,
         weights = wt, prior.weights = setNames(weights, ynames),
         df.residual = resdf, df.null = nulldf, y = setNames(y, ynames),
         boundary = boundary)
    class(model) <- "glm"
    model
}

# sizes
gauss <- readRDS(file.path(results_dir, "gauss_size.rds"))
binomial <- readRDS(file.path(results_dir, "binomial_size.rds"))
poisson <- readRDS(file.path(results_dir, "poisson_size.rds"))

gauss_size <- melt(gauss, id.vars=c("n", "p"), measure.vars=c("size_hglm", "size_hglm_big", "size_glm"), variable.name="method", value.name="size")
gauss_size[, c("method", "size") := .(gsub("^[^_]*_", "", method), as.numeric(gsub(" .*$", "", size)))]
gauss_size

binomial_size <- melt(binomial, id.vars=c("n", "p"), measure.vars=c("size_hglm", "size_hglm_big", "size_glm"), variable.name="method", value.name="size")
binomial_size[, c("method", "size") := .(gsub("^[^_]*_", "", method), as.numeric(gsub(" .*$", "", size)))]
binomial_size

poisson_size <- melt(poisson, id.vars=c("n", "p"), measure.vars=c("size_hglm", "size_hglm_big", "size_glm"), variable.name="method", value.name="size")
poisson_size[, c("method", "size") := .(gsub("^[^_]*_", "", method), as.numeric(gsub(" .*$", "", size)))]
poisson_size

sizes <- list(gauss_size, binomial_size, poisson_size)
ofi <- file.path(supp_dir, "sizes.rds")
saveRDS(sizes, ofi)

# bss
gauss_bss <- readRDS(file.path(results_dir, "gauss_bss.rds"))
binomial_bss <- readRDS(file.path(results_dir, "binomial_bss.rds"))
poisson_bss <- readRDS(file.path(results_dir, "poisson_bss.rds"))

binomial_bss <- binomial_bss[!is.na(p)]
poisson_bss <- poisson_bss[!is.na(p)]

# change result data if gurobi not installed
if (!gurobi_installed) {
    gauss_bss[, model_hglm_gurobi := model_bestglm]
    gauss_bss[, time_hglm_gurobi := time_bestglm]
}
# change result data if mosek not installed
if (!mosek_installed) {
    binomial_bss[, model_hglm_mosek := model_hglm_ecos]
    binomial_bss[, time_hglm_mosek := time_hglm_ecos]
    poisson_bss[, model_hglm_mosek := model_hglm_ecos]
    poisson_bss[, time_hglm_mosek := time_hglm_ecos]
}


## validation

### gaussian
gauss_data <- readRDS('data/gauss.rds')
ecos_models <- lapply(gauss_bss[!is.na(time_hglm_ecos), model_hglm_ecos], select_AIC)
bestglm_models <- lapply(gauss_bss[, model_bestglm], select_AIC)
gurobi_models <- lapply(gauss_bss[, model_hglm_gurobi], select_AIC)
abess_models <- vector(mode="list", length=length(bestglm_models))
# need to recalculate AIC for abess since the one calculated by abess directly differs from `glm`
for (i in seq(bestglm_models)) {
    abess_models[[i]] <- make_glm(formula("y ~ ."), select_AIC(gauss_bss[i, model_abess][[1]]), family=gaussian(), data=gauss_data[[i]])
}

gurobi_aic <- sapply(gurobi_models, AIC)
ecos_aic <- sapply(ecos_models, AIC)
length(ecos_aic) <- length(gurobi_aic)
abess_aic <- sapply(abess_models, AIC)
bestglm_aic <- sapply(bestglm_models, AIC)

as <- as.data.table(round(cbind(gurobi_aic, ecos_aic, abess_aic, bestglm_aic), 3))
as[, minaic := apply(.SD, 1, min, na.rm=TRUE)]

ecos_acoef <- lapply(ecos_models, name_coef)
gurobi_acoef <- lapply(gurobi_models, name_coef)
abess_acoef <- lapply(abess_models, name_coef)
bestglm_acoef <- lapply(bestglm_models, name_coef)

sel_coef <- data.table(ecos = mapply(function(x,y) all(x==y), gurobi_acoef, ecos_acoef),
                       gurobi = mapply(function(x,y) all(x==y), gurobi_acoef, gurobi_acoef),
                       abess = mapply(function(x,y) all(x==y), gurobi_acoef, abess_acoef),
                       bestglm = mapply(function(x,y) all(x==y), gurobi_acoef, bestglm_acoef))

result <- cbind(gauss_bss[, .(i,p,k)], sel_coef)
gauss_val <- result[, lapply(.SD, sum), .(p, k), .SDcols=c("ecos", "gurobi", "abess", "bestglm")]
gauss_val[p > 20, ecos := NA]
gauss_val

### binomial
ecos_models <- lapply(binomial_bss[, model_hglm_ecos], select_AIC)
mosek_models <- lapply(binomial_bss[, model_hglm_mosek], select_AIC)
bestglm_models <- lapply(binomial_bss[!is.na(time_bestglm), model_bestglm], select_AIC)
abess_models <- vector(mode="list", length=length(ecos_models))
for (i in seq(ecos_models)) {
    x <- ecos_models[[i]]
    abess_models[[i]] <- make_glm(x$formula, select_AIC(binomial_bss[i, model_abess][[1]]), family=x$family, data=x$data)
}

mosek_aic <- sapply(mosek_models, AIC)
ecos_aic <- sapply(ecos_models, AIC)
abess_aic <- sapply(abess_models, AIC)
bestglm_aic <- sapply(bestglm_models, AIC)
length(bestglm_aic) <- length(ecos_aic)

as <- as.data.table(round(cbind(mosek_aic, ecos_aic, abess_aic, bestglm_aic), 3))
as[, minaic := apply(.SD, 1, min, na.rm=TRUE)]

ecos_acoef <- lapply(ecos_models, name_coef)
mosek_acoef <- lapply(mosek_models, name_coef)
abess_acoef <- lapply(abess_models, name_coef)
bestglm_acoef <- lapply(bestglm_models, name_coef)

sel_coef <- data.table(ecos = mapply(function(x,y) all(x==y), mosek_acoef, ecos_acoef),
                       mosek = mapply(function(x,y) all(x==y), mosek_acoef, mosek_acoef),
                       abess = mapply(function(x,y) all(x==y), mosek_acoef, abess_acoef),
                       bestglm = mapply(function(x,y) all(x==y), mosek_acoef, bestglm_acoef))

result <- cbind(binomial_bss[, .(i,p,k)], sel_coef)
binomial_val <- result[, lapply(.SD, sum), .(p, k), .SDcols=c("ecos", "mosek", "abess", "bestglm")]
binomial_val[p > 15, bestglm := NA]
binomial_val


### poisson
ecos_models <- lapply(poisson_bss[, model_hglm_ecos], select_AIC)
mosek_models <- lapply(poisson_bss[, model_hglm_mosek], select_AIC)
bestglm_models <- lapply(poisson_bss[!is.na(time_bestglm), model_bestglm], select_AIC)
abess_models <- vector(mode="list", length=length(ecos_models))
for (i in seq(ecos_models)) {
    x <- ecos_models[[i]]
    abess_models[[i]] <- make_glm(x$formula, select_AIC(poisson_bss[i, model_abess][[1]]), family=x$family, data=x$data)
}

mosek_aic <- sapply(mosek_models, AIC)
ecos_aic <- sapply(ecos_models, AIC)
abess_aic <- sapply(abess_models, AIC)
bestglm_aic <- sapply(bestglm_models, AIC)
length(bestglm_aic) <- length(ecos_aic)

as <- as.data.table(round(cbind(mosek_aic, ecos_aic, abess_aic, bestglm_aic), 3))
as[, minaic := apply(.SD, 1, min, na.rm=TRUE)]

ecos_acoef <- lapply(ecos_models, name_coef)
mosek_acoef <- lapply(mosek_models, name_coef)
abess_acoef <- lapply(abess_models, name_coef)
bestglm_acoef <- lapply(bestglm_models, name_coef)

sel_coef <- data.table(ecos = mapply(function(x,y) all(x==y), mosek_acoef, ecos_acoef),
                       mosek = mapply(function(x,y) all(x==y), mosek_acoef, mosek_acoef),
                       abess = mapply(function(x,y) all(x==y), mosek_acoef, abess_acoef),
                       bestglm = mapply(function(x,y) all(x==y), mosek_acoef, bestglm_acoef))

result <- cbind(poisson_bss[, .(i,p,k)], sel_coef)
poisson_val <- result[, lapply(.SD, sum), .(p, k), .SDcols=c("ecos", "mosek", "abess", "bestglm")]
poisson_val[p > 15, bestglm := NA]
poisson_val

bss_val <- Reduce(function(x,y) merge(x, y, by=c("p", "k"), all.x=TRUE), list(gauss_val, binomial_val, poisson_val))
bss_val

kbl(bss_val, format="latex", booktabs=TRUE,
    label = "bss:val", align="r", escape=FALSE, linesep = "",
    col.names = c("$p$", "$q$", "hglm(ECOS)", "hglm(GUROBI)", "\\pkg{abess}", "\\pkg{bestglm}", "hglm(ECOS)", "hglm(MOSEK)", "\\pkg{abess}", "\\pkg{bestglm}", "hglm(ECOS)", "hglm(MOSEK)", "\\pkg{abess}", "\\pkg{bestglm}"),
    caption = "Comparison of \\proglang{R} packages on best subset selection with respect to the number of times the method selected the correct coefficients of the model with minimal AIC for 10 datasets.") %>%
    kable_classic() %>%
    add_header_above(c(" " = 2 , "Linear" = 4, "Logistic" = 4, "Poisson" = 4)) %>%
    kable_styling(latex_options="scale_down")

## timings

gauss_time <- melt(gauss_bss, id.vars=c("p", "k"), measure.vars=c("time_hglm_ecos", "time_hglm_gurobi", "time_abess", "time_bestglm"), variable.name="method", value.name="time")
gauss_time[, method := gsub("^[^_]*_", "", method)] #, levels = c("hglm", "abess", "bestglm"))]

binomial_time <- melt(binomial_bss, id.vars=c("p", "k"), measure.vars=c("time_hglm_ecos", "time_hglm_mosek", "time_abess", "time_bestglm"), variable.name="method", value.name="time")
binomial_time[, method := gsub("^[^_]*_", "", method)] #, levels = c("hglm", "abess", "bestglm"))]

poisson_time <- melt(poisson_bss, id.vars=c("p", "k"), measure.vars=c("time_hglm_ecos", "time_hglm_mosek", "time_abess", "time_bestglm"), variable.name="method", value.name="time")
poisson_time[, method := gsub("^[^_]*_", "", method)]#, levels = c("hglm", "abess", "bestglm"))]

bss <- list(gauss_time, binomial_time, poisson_time)
ofi <- file.path(supp_dir, "bss.rds")
saveRDS(bss, ofi)

gauss_time_wide <- dcast(gauss_time[, .(mean = mean(time), sd = sd(time)), .(p,k,method)], p + k ~ method, value.var = c("mean", "sd"))[, .(p, k, 
                    hglm_ecos = pn(mean_hglm_ecos, sd_hglm_ecos),
                    hglm_gurobi = pn(mean_hglm_gurobi, sd_hglm_gurobi),
                    abess = pn(mean_abess, sd_abess),
                    bestglm = pn(mean_bestglm, sd_bestglm))]

binomial_time_wide <- dcast(binomial_time[, .(mean = mean(time), sd = sd(time)), .(p,k,method)], p + k ~ method, value.var = c("mean", "sd"))[, .(p, k, 
                    hglm_ecos = pn(mean_hglm_ecos, sd_hglm_ecos),
                    hglm_mosek = pn(mean_hglm_mosek, sd_hglm_mosek),
                    abess = pn(mean_abess, sd_abess),
                    bestglm = pn(mean_bestglm, sd_bestglm))]

poisson_time_wide <- dcast(poisson_time[, .(mean = mean(time), sd = sd(time)), .(p,k,method)], p + k ~ method, value.var = c("mean", "sd"))[, .(p, k, 
                    hglm_ecos = pn(mean_hglm_ecos, sd_hglm_ecos),
                    hglm_mosek = pn(mean_hglm_mosek, sd_hglm_mosek),
                    abess = pn(mean_abess, sd_abess),
                    bestglm = pn(mean_bestglm, sd_bestglm))]


min_gauss <- apply(gauss_time_wide[,-1:-2][, lapply(.SD, function(x) as.numeric(gsub("(\\s*) .*", "\\1", x)))], 1, which.min)
min_binomial <- apply(binomial_time_wide[,-1:-2][, lapply(.SD, function(x) as.numeric(gsub("(\\s*) .*", "\\1", x)))], 1, which.min)
min_poisson <- apply(poisson_time_wide[,-1:-2][, lapply(.SD, function(x) as.numeric(gsub("(\\s*) .*", "\\1", x)))], 1, which.min)

bss_time <- Reduce(function(x,y) merge(x, y, by=c("p", "k"), all.x=TRUE), list(gauss_time_wide, binomial_time_wide, poisson_time_wide))
bss_time

for (i in seq_along(min_gauss)) bss_time[[i, min_gauss[i]+2]] <- latex_bf(bss_time[[i, min_gauss[i]+2]])
for (i in seq_along(min_binomial)) bss_time[[i, min_binomial[i]+6]] <- latex_bf(bss_time[[i, min_binomial[i]+6]])
for (i in seq_along(min_poisson)) bss_time[[i, min_poisson[i]+10]] <- latex_bf(bss_time[[i, min_poisson[i]+10]])

bss_time_latex = setorder(rbind(bss_time, bss_time), p)
bss_time_latex[, c("p", "k") := list(as.character(p), as.character(k))]
for (i in 1:nrow(bss_time_latex)) {
    for (j in 3:ncol(bss_time_latex)) {
        bss_time_latex[[i,j]] <- if (i %% 2 == 1) split1(bss_time_latex[[i,j]]) else split2(bss_time_latex[[i,j]])
    }
}
bss_time_latex[duplicated(p), c("p", "k") := list("", "")]
bss_time_latex

kbl(bss_time_latex, format="latex", booktabs=TRUE,
    toprule = "\\hline", bottomrule = "\\hline",
    label = "bss", align="rrrrrrrrrrrrrr", escape=FALSE, linesep = "",
    col.names = c("$p$", "$q$", "hglm(ECOS)", "hglm(GUROBI)", "\\pgk{abess}", "\\pkg{bestglm}", "hglm(ECOS)", "hglm(MOSEK)", "\\pgk{abess}", "\\pkg{bestglm}", "hglm(ECOS)", "hglm(MOSEK)", "\\pgk{abess}", "\\pkg{bestglm}"),
    caption = "Comparison of \\proglang{R} packages on best subset selection with respect to runtime. The average runtimes are given in seconds together with the standard deviation (in parentheses) across 10 datasets.") %>%
    kable_classic() %>%
    add_header_above(c(" " =2 , "Linear" = 4, "Logistic" = 4, "Poisson" = 4)) %>%
    kable_styling(latex_options="scale_down")

## linear constraints
gauss_linear <- readRDS(file.path(results_dir, "gauss_linear.rds"))
binomial_linear <- readRDS(file.path(results_dir, "binomial_linear.rds"))
poisson_linear <- readRDS(file.path(results_dir, "poisson_linear.rds"))

# change result data if gurobi not installed
if (!gurobi_installed) {
    gauss_linear[, model_hglm_gurobi := model_hglm_ecos]
    gauss_linear[, time_hglm_gurobi := time_hglm_ecos]
}
# change result data if mosek not installed
if (!mosek_installed) {
    binomial_linear[, model_hglm_mosek := model_hglm_ecos]
    binomial_linear[, time_hglm_mosek := time_hglm_ecos]
    poisson_linear[, model_hglm_mosek := model_hglm_ecos]
    poisson_linear[, time_hglm_mosek := time_hglm_ecos]
}

### validation

#### gauss
ecos_models <- gauss_linear[["model_hglm_ecos"]]
gurobi_models <- gauss_linear[["model_hglm_gurobi"]]
consreg_models <- gauss_linear[["model_consreg"]]
restriktor_models <- gauss_linear[["model_restriktor"]]

ecos_val <- sapply(ecos_models, nonneg)
gurobi_val <- sapply(gurobi_models, nonneg)
consreg_val <- sapply(consreg_models, nonneg)
restriktor_val <- sapply(restriktor_models, nonneg)
vals <- cbind(ecos_val, gurobi_val, consreg_val, restriktor_val)

result <- cbind(gauss_linear[, .(i,p,k)], vals)

gauss_lin_val <- result[, lapply(.SD, sum), .(p, k), .SDcols=c("ecos_val", "gurobi_val", "consreg_val", "restriktor_val")]
gauss_lin_val

#### binomial
ecos_models <- binomial_linear[["model_hglm_ecos"]]
mosek_models <- binomial_linear[["model_hglm_mosek"]]
consreg_models <- binomial_linear[["model_consreg"]]
restriktor_models <- binomial_linear[["model_restriktor"]]

ecos_val <- sapply(ecos_models, nonneg)
mosek_val <- sapply(mosek_models, nonneg)
consreg_val <- sapply(consreg_models, nonneg)
restriktor_val <- sapply(restriktor_models, nonneg)
vals <- cbind(ecos_val, mosek_val, consreg_val, restriktor_val)

result <- cbind(gauss_linear[, .(i,p,k)], vals)

binomial_lin_val <- result[, lapply(.SD, sum), .(p, k), .SDcols=c("ecos_val", "mosek_val", "consreg_val", "restriktor_val")]
binomial_lin_val

#### poisson
ecos_models <- poisson_linear[["model_hglm_ecos"]]
mosek_models <- poisson_linear[["model_hglm_mosek"]]
consreg_models <- poisson_linear[["model_consreg"]]
restriktor_models <- poisson_linear[["model_restriktor"]]

ecos_val <- sapply(ecos_models, nonneg)
mosek_val <- sapply(mosek_models, nonneg)
consreg_val <- sapply(consreg_models, nonneg)
restriktor_val <- sapply(restriktor_models, nonneg)
vals <- cbind(ecos_val, mosek_val, consreg_val, restriktor_val)

result <- cbind(gauss_linear[, .(i,p,k)], vals)

poisson_lin_val <- result[, lapply(.SD, sum), .(p, k), .SDcols=c("ecos_val", "mosek_val", "consreg_val", "restriktor_val")]
poisson_lin_val

lin_val <- Reduce(function(x,y) merge(x, y, by=c("p", "k"), all.x=TRUE), list(gauss_lin_val, binomial_lin_val, poisson_lin_val))
lin_val

kbl(lin_val, format="latex", booktabs=TRUE,
    label = "lin:val", align="r", escape=FALSE, linesep = "",
    col.names = c("$p$", "$q$", "hglm(ECOS)", "hglm(GUROBI)", "\\pkg{ConsReg}", "\\pkg{restriktor}", "hglm(ECOS)", "hglm(MOSEK)", "\\pkg{ConsReg}", "\\pkg{restriktor}", "hglm(ECOS)", "hglm(MOSEK)", "\\pkg{ConsReg}", "\\pkg{restriktor}"),
    caption = "Comparison of \\proglang{R} packages for fitting GLMs with linear constraints with respect to the number of times the method found a valid solution for 10 datasets.") %>%
    kable_classic() %>%
    add_header_above(c(" " = 2 , "Linear" = 4, "Logistic" = 4, "Poisson" = 4)) %>%
    kable_styling(latex_options="scale_down")

### timings

gauss_time <- melt(gauss_linear, id.vars=c("p", "k"), measure.vars=c("time_hglm_ecos", "time_hglm_gurobi", "time_consreg", "time_restriktor"), variable.name="method", value.name="time")
gauss_time[, method := gsub("^[^_]*_", "", method)]

binomial_time <- melt(binomial_linear, id.vars=c("p", "k"), measure.vars=c("time_hglm_ecos", "time_hglm_mosek", "time_consreg", "time_restriktor"), variable.name="method", value.name="time")
binomial_time[, method := gsub("^[^_]*_", "", method)]

poisson_time <- melt(poisson_linear, id.vars=c("p", "k"), measure.vars=c("time_hglm_ecos", "time_hglm_mosek", "time_consreg", "time_restriktor"), variable.name="method", value.name="time")
poisson_time[, method := gsub("^[^_]*_", "", method)]

linear_times <- list(gauss_time, binomial_time, poisson_time)
ofi <- file.path(supp_dir, "linear.rds")
saveRDS(linear_times, ofi)

# tables
gauss_time_wide <- dcast(gauss_time[, .(mean = mean(time), sd = sd(time)), .(p,k,method)], p + k ~ method, value.var = c("mean", "sd"))[, .(p, k, 
                    hglm_ecos = pn(mean_hglm_ecos, sd_hglm_ecos),
                    hglm_gurobi = pn(mean_hglm_gurobi, sd_hglm_gurobi),
                    consreg = pn(mean_consreg, sd_consreg),
                    restriktor = pn(mean_restriktor, sd_restriktor))]

binomial_time_wide <- dcast(binomial_time[, .(mean = mean(time), sd = sd(time)), .(p,k,method)], p + k ~ method, value.var = c("mean", "sd"))[, .(p, k, 
                    hglm_ecos = pn(mean_hglm_ecos, sd_hglm_ecos),
                    hglm_mosek = pn(mean_hglm_mosek, sd_hglm_mosek),
                    consreg = pn(mean_consreg, sd_consreg),
                    restriktor = pn(mean_restriktor, sd_restriktor))]

poisson_time_wide <- dcast(poisson_time[, .(mean = mean(time), sd = sd(time)), .(p,k,method)], p + k ~ method, value.var = c("mean", "sd"))[, .(p, k, 
                    hglm_ecos = pn(mean_hglm_ecos, sd_hglm_ecos),
                    hglm_mosek = pn(mean_hglm_mosek, sd_hglm_mosek),
                    consreg = pn(mean_consreg, sd_consreg),
                    restriktor = pn(mean_restriktor, sd_restriktor))]

linear_time <- Reduce(function(x,y) merge(x, y, by=c("p", "k"), all.x=TRUE), list(gauss_time_wide, binomial_time_wide, poisson_time_wide))
linear_time

min_gauss <- apply(gauss_time_wide[,-1:-2][, lapply(.SD, function(x) as.numeric(gsub("(\\s*) .*", "\\1", x)))], 1, which.min)
min_binomial <- apply(binomial_time_wide[,-1:-2][, lapply(.SD, function(x) as.numeric(gsub("(\\s*) .*", "\\1", x)))], 1, which.min)
min_poisson <- apply(poisson_time_wide[,-1:-2][, lapply(.SD, function(x) as.numeric(gsub("(\\s*) .*", "\\1", x)))], 1, which.min)

for (i in seq(nrow(linear_time))) {
    linear_time[[i, min_gauss[i]+2]] <- latex_bf(linear_time[[i, min_gauss[i]+2]])
}

linear_time
linear_time_latex = setorder(rbind(linear_time, linear_time), p)
linear_time_latex[, c("p", "k") := list(as.character(p), as.character(k))]
for (i in 1:nrow(linear_time_latex)) {
    for (j in 3:ncol(linear_time_latex)) {
        linear_time_latex[[i,j]] <- if (i %% 2 == 1) split1(linear_time_latex[[i,j]]) else split2(linear_time_latex[[i,j]])
    }
}
linear_time_latex[duplicated(p), c("p", "k") := list("", "")]
linear_time_latex

kbl(linear_time_latex, format="latex", booktabs=TRUE,
    toprule = "\\hline", bottomrule = "\\hline",
    label = "linear", align="rrrrrrrrrrrrrr", escape=FALSE, linesep = "",
    col.names = c("$p$", "$q$", "hglm(ECOS)", "hglm(GUROBI)", "\\pkg{ConsReg}", "\\pkg{restriktor}", "hglm(ECOS)", "hglm(MOSEK)", "\\pkg{ConsReg}", "\\pkg{restriktor}", "hglm(ECOS)", "hglm(MOSEK)", "\\pkg{ConsReg}", "\\pkg{restriktor}"),
    caption = "Comparison of \\proglang{R} packages on fitting linear constraints with respect to runtime. The average runtimes are given in seconds together with the standard deviation (in parentheses) across 10 datasets.") %>%
    kable_classic() %>%
    add_header_above(c(" " =2 , "Linear" = 4, "Logistic" = 4, "Poisson" = 4)) %>%
    kable_styling(latex_options="scale_down")
