# run generate.r before to create data

library(holiglm)
library(restriktor)
library(ConsReg)
library(data.table)

result_dir <- normalizePath("results")
dir.create(result_dir, showWarnings = FALSE, recursive = TRUE)

gurobi_installed <- ("ROI.plugin.gurobi" %in% ROI::ROI_installed_solvers())
mosek_installed <- ("ROI.plugin.mosek" %in% ROI::ROI_installed_solvers())

# linear_constraints 

## gaussian data
gauss <- readRDS('data/gauss.rds')
result <- data.table(i=seq_along(gauss))

# warmup solver
hglm(mpg~., data=mtcars, solver="ecos")
if (gurobi_installed) hglm(mpg~., data=mtcars, solver="gurobi")

for (i in seq_along(gauss)) {
    message(i/length(gauss)*100, "%")
    data <- gauss[[i]]
    x <- model.matrix(y~.-1, data=data)
    y <- data[["y"]]
    m <- ncol(data)

    const_hglm <- lower(setNames(double(m-1L), colnames(x)))
    const_consreg <- paste(sprintf("x%d >= 0", seq(m-1)), collapse=", ")

    time_hglm       <- system.time(model_hglm <- hglm(y~., data=data, constraints=const_hglm, family=gaussian(), solver="ecos"))[[3]]
    time_consreg    <- system.time({ini <- coef(lm(y~., data=data)); ini[ini<0] <- 0; model_consreg <- ConsReg(y~., data=data, family="gaussian", constraints=const_consreg, ini.pars.coef=ini)})[[3]]
    time_restriktor <- system.time(model_restriktor <- restriktor(glm(y~., data=data, family=gaussian()), constraints=diag(c(0, rep(1, m-1))), rhs=double(m)))[[3]]
    
    set(result, i=i, j="p", value=m-1L)
    set(result, i=i, j="k", value=sum(attr(data, "true_beta"))-1L)

    set(result, i=i, j="time_hglm_ecos",  value=time_hglm)
    set(result, i=i, j="time_consreg",    value=time_consreg)
    set(result, i=i, j="time_restriktor", value=time_restriktor)

    set(result, i=i, j="model_hglm_ecos",  value=list(model_hglm))
    set(result, i=i, j="model_consreg",    value=list(model_consreg))
    set(result, i=i, j="model_restriktor", value=list(model_restriktor))

    if (gurobi_installed) {
        time_hglm <- system.time(model_hglm <- hglm(y~., data=data, constraints=const_hglm, family=gaussian(), solver="gurobi"))[[3]]
        set(result, i=i, j="time_hglm_gurobi", value=time_hglm)
        set(result, i=i, j="model_hglm_gurobi", value=list(model_hglm))
    }
}

ofi <- file.path(result_dir, "gauss_linear.rds")
saveRDS(result, ofi)

## binomial data
binomial <- readRDS('data/binomial.rds')
result <- data.table(i=seq_along(binomial))

# warmup solver
hglm(group ~ ., data=PlantGrowth, family=binomial(), solver="ecos")
if (mosek_installed) hglm(group ~ ., data=PlantGrowth, family=binomial(), solver="mosek")

for (i in seq_along(binomial)) {
    message(i/length(binomial)*100, "%")
    data <- binomial[[i]]
    x <- model.matrix(y~.-1, data=data)
    y <- data[["y"]]
    m <- ncol(data)

    const_hglm <- lower(setNames(double(m-1L), colnames(x)))
    const_consreg <- paste(sprintf("x%d > 0", seq(m-1)), collapse=", ")

    time_hglm       <- system.time(model_hglm <- hglm(y~., data=data, constraints=const_hglm, family=binomial(), solver="ecos"))[[3]]
    time_consreg    <- system.time({ini <- coef(glm(y~., data=data, family=binomial())); ini[ini<0] <- 0; model_consreg <- ConsReg(y~., data=data, family="binomial", constraints=const_consreg, ini.pars.coef=ini)})[[3]]
    time_restriktor <- system.time(model_restriktor <- restriktor(glm(y~., data=data, family=binomial()), constraints=diag(c(0, rep(1, m-1))), rhs=double(m)))[[3]]
    
    set(result, i=i, j="p", value=m-1L)
    set(result, i=i, j="k", value=sum(attr(data, "true_beta"))-1L)

    set(result, i=i, j="time_hglm_ecos",  value=time_hglm)
    set(result, i=i, j="time_consreg",    value=time_consreg)
    set(result, i=i, j="time_restriktor", value=time_restriktor)

    set(result, i=i, j="model_hglm_ecos",  value=list(model_hglm))
    set(result, i=i, j="model_consreg",    value=list(model_consreg))
    set(result, i=i, j="model_restriktor", value=list(model_restriktor))

    if (mosek_installed) {
        time_hglm  <- system.time(model_hglm <- hglm(y~., data=data, constraints=const_hglm, family=binomial(), solver="mosek"))[[3]]
        set(result, i=i, j="time_hglm_mosek", value=time_hglm)
        set(result, i=i, j="model_hglm_mosek", value=list(model_hglm))
    }
}

ofi <- file.path(result_dir, "binomial_linear.rds")
saveRDS(result, ofi)

## poisson data
poisson <- readRDS('data/poisson.rds')
result <- data.table(i=seq_along(poisson))

# warmup solver
hglm(Assault ~ ., data=USArrests, family=poisson(), solver="ecos")
if (mosek_installed) hglm(Assault ~ ., data=USArrests, family=poisson(), solver="mosek")

for (i in seq_along(poisson)) {
    message(i/length(poisson)*100, "%")
    data <- poisson[[i]]
    x <- model.matrix(y~.-1, data=data)
    y <- data[["y"]]
    m <- ncol(data)

    const_hglm <- lower(setNames(double(m-1L), colnames(x)))
    const_consreg <- paste(sprintf("x%d > 0", seq(m-1)), collapse=", ")

    time_hglm       <- system.time(model_hglm <- hglm(y~., data=data, constraints=const_hglm, family=poisson(), solver="ecos", control=list(maxit=1e3L)))[[3]]
    time_consreg    <- system.time({ini <- coef(glm(y~., data=data, family=poisson())); ini[ini<0] <- 0; model_consreg <- ConsReg(y~., data=data, family="poisson", constraints=const_consreg, ini.pars.coef=ini)})[[3]]
    time_restriktor <- system.time(model_restriktor <- restriktor(glm(y~., data=data, family=poisson()), constraints=diag(c(0, rep(1, m-1))), rhs=double(m)))[[3]]
    
    set(result, i=i, j="p", value=m-1L)
    set(result, i=i, j="k", value=sum(attr(data, "true_beta"))-1L)

    set(result, i=i, j="time_hglm_ecos",  value=time_hglm)
    set(result, i=i, j="time_consreg",    value=time_consreg)
    set(result, i=i, j="time_restriktor", value=time_restriktor)

    set(result, i=i, j="model_hglm_ecos",  value=list(model_hglm))
    set(result, i=i, j="model_consreg",    value=list(model_consreg))
    set(result, i=i, j="model_restriktor", value=list(model_restriktor))

    if (mosek_installed) {
        time_hglm  <- system.time(model_hglm <- hglm(y~., data=data, constraints=const_hglm, family=poisson(), solver="mosek"))[[3]]
        set(result, i=i, j="time_hglm_mosek", value=time_hglm)
        set(result, i=i, j="model_hglm_mosek", value=list(model_hglm))
    }
}

ofi <- file.path(result_dir, "poisson_linear.rds")
saveRDS(result, ofi)
