#! /usr/bin/Rscript
install.packages(c("palmerpenguins", "BNPmix", "mclust",
                   "dirichletprocess", "bayesm", "PReMiuM"),
                 repos = "https://cran.r-project.org")

# Specify pointer to the "temp" folder in the replication material
setwd('temp')
set.seed(1)

library(palmerpenguins)
df <- penguins[complete.cases(penguins[3:6]), 3:6]

library(BNPmix)
bnpmix_time <- system.time({
  est_model <- PYdensity(
    y = as.matrix(sweep(df, 2, colMeans(df))),
    mcmc = list(niter = 1000, nburn = 100, nupd = 1000))
  out <- partition(est_model)})
bnpmix_clustering <- out$partitions[3,]

library(mclust)
mclust_time <- system.time(
  out <- Mclust(df, modelNames="VVV"))
mclust_clustering <- out$classification

library(dirichletprocess)
dirichletprocess_time <- system.time({
  out <- DirichletProcessMvnormal(scale(df))
  dpCluster <- Fit(out, 100, progressBar = F)})
dirichletprocess_clustering <- dpCluster$clusterLabels

library(bayesm)
bayesm_time <- system.time({
out <- rnmixGibbs(Data=list(y=data.matrix(df)),
                  Prior=list(ncomp=3, a=c(rep(1,3))),
                  Mcmc=list(R=1000, keep=1))
## find clusters
outclusterMix <- clusterMix(out$nmix$zdraw[100:1000,])})
bayesm_clustering <- outclusterMix$clustera

library(PReMiuM)
premium_time <- system.time({
  runInfoObj <- profRegr(covNames=colnames(df), data = df,
                         excludeY = TRUE, xModel = "Normal",
                         seed = 1)
  disSimObj <- calcDissimilarityMatrix(runInfoObj)
  out <- calcOptimalClustering(disSimObj, maxNClusters=3)})
premium_clustering <- out$clustering

df_clusters <- data.frame(
  BNPmix=bnpmix_clustering,
  mclust=mclust_clustering,
  dirichletprocess=dirichletprocess_clustering,
  bayesm=bayesm_clustering,
  PReMiuM=premium_clustering)
df_times <- data.frame(
  BNPmix=bnpmix_time,
  mclust=mclust_time,
  dirichletprocess=dirichletprocess_time,
  bayesm=bayesm_time,
  PReMiuM=premium_time)
write.csv(df_clusters, "R_clusters.csv")
write.csv(df_times, "R_times.csv")
