simul <- function(aux, sims = 100, cores = 1) {
  
  cl <- parallel::makeCluster(cores)
  doSNOW::registerDoSNOW(cl)
  
  stor <- foreach::foreach(i = 1 : sims,
    .packages = c("ECOSolveR", "CVXR", "Matrix", "nloptr"),
    .export = c("solve_ECOS", "solve_CVXR", "solve_CVXR_nodcp", "solve_nloptr",
      "obj.fun.min","obj.fun.max", "single.ineq","double.ineq", "norm.equal",
      "prepareOptions", "obj.fun.min.sr","obj.fun.max.sr","double.ineq.sr",
      "norm.equal.sr"),
    .combine = rbind) %dorng% {
      
      ##############################################################################
      # Call ECOS directly
      ##############################################################################
      init <- Sys.time()
      tmp <- solve_ECOS(aux$xt, aux$beta, aux$dataEcos, aux$ns, aux$Jtot, aux$KMI)
      fin <- Sys.time()
      delta_ECOS <- fin - init
      
      if (tmp$flag == TRUE) {
        delta_ECOS <- NA
      }
      
      ##############################################################################
      # Call CVXR 
      ##############################################################################
      init <- Sys.time()
      tmp <- solve_CVXR(aux$xt, aux$beta, aux$Jtot, aux$KMI, aux$I, aux$Q, aux$a, aux$d, aux$Q1, aux$Q2, aux$J, aux$lb, aux$p.str, aux$dire)
      fin <- Sys.time()
      delta_CVXR <- fin - init
      
      if (tmp$flag == TRUE) {
        delta_CVXR <- NA
      }
      
      ##############################################################################
      # Call CVXR with no DCP
      ##############################################################################
      tmp <- solve_CVXR(aux$xt, aux$beta, aux$Jtot, aux$KMI, aux$I, aux$Q, aux$a, aux$d,
        aux$Q1, aux$Q2, aux$J, aux$lb, aux$p.str, aux$dire,
        get_prob_data = TRUE)
      init <- Sys.time()
      tmp <- solve_CVXR_nodcp(tmp$prob, tmp$prob_data)
      fin <- Sys.time()
      delta_CVXR_nodcp <- fin - init
      
      if (tmp$flag == TRUE) {
        delta_CVXR <- NA
      }       
      
      ##############################################################################
      # Call nloptr 
      ##############################################################################
      init <- Sys.time()
      tmp <- solve_nloptr(aux$xt, aux$x0, aux$G, aux$beta, aux$Q, aux$Jtot, aux$S, aux$lb, aux$dire, aux$Q1, aux$Q2, aux$p, aux$KMI, FALSE, aux$opt.list)
      fin <- Sys.time()
      delta_NLOPT <- fin - init    
      
      if (tmp$flag == TRUE) {
        delta_NLOPT <- NA
      }
      
      c(delta_ECOS,delta_CVXR,delta_CVXR_nodcp,delta_NLOPT)
      
    }
  
  parallel::stopCluster(cl)
  
  colnames(stor) <- c("scpi", "CVXR - dcp on", "CVXR - dcp off", "nloptr")
  return(stor)
}


solve_ECOS <- function(xt, beta, data, ns, Jtot, KMI) {
  
  solver_output <- ECOSolveR::ECOS_csolve(c = data[["c"]],
    G = data[["G"]],
    h = data[["h"]],
    dims = data[["dims"]],
    A = data[["A"]],
    b = data[["b"]])
  
  if (!(solver_output$infostring %in% c("Optimal solution found", "Close to optimal solution found"))) {
    lb.f <- NA
    alert <- TRUE
    xx <- NA
  } else {
    xx <- solver_output$x[1:(Jtot+KMI)]
    lb.f <- -sum(xt*(xx - beta))
    alert <- FALSE
  } 
  return(list(obj=lb.f, x=xx, flag=alert))
}


solve_CVXR <- function(xt, beta, Jtot, KMI, I, Q, a, d, QQ, QQ2, J, lb, p, dire, get_prob_data = FALSE) {
  
  x <- CVXR::Variable(Jtot+KMI)
  constraints <- list(CVXR::quad_form(x, Q) + sum(CVXR::multiply(a, x)) + d <= 0)
  
  if (lb[1] > - Inf) {
    constraints <- append(constraints, list(x[1:Jtot] >= lb))
  }
  
  j.lb <- 1
  
  for (i in seq_len(I)) {
    j.ub <- j.lb + J[[i]] - 1
    
    if (p == "L1") {
      if (dire == "==") { # simplex
        constraints <- append(constraints, list(CVXR::sum_entries(x[j.lb:j.ub]) == QQ[i]))
      } else if (dire == "<=") { # lasso
        constraints <- append(constraints, list(CVXR::norm1(x[j.lb:j.ub]) <= QQ[i]))
      }
      
    } else if (p == "L2") {  # ridge
      constraints <- append(constraints, list(CVXR::sum_squares(x[j.lb:j.ub]) <= CVXR::power(QQ[i], 2)))
      
    } else if (p == "L1-L2") {
      constraints <- append(constraints, list(CVXR::sum_entries(x[j.lb:j.ub]) == QQ[i], 
        CVXR::sum_squares(x[j.lb:j.ub]) <= CVXR::power(QQ2[i], 2)))
    }    
    
    j.lb <- j.ub + 1    
  }
  
  objective  <- CVXR::Minimize(-sum(CVXR::multiply(xt,x - beta)))
  prob <- CVXR::Problem(objective, constraints)
  sol <- CVXR::solve(prob)
  alert   <- !(sol$status %in% c("optimal","optimal_inaccurate"))
  
  if (get_prob_data == FALSE) {
    return(list(obj = sol$value, x = sol$getValue(x), flag = alert))
  } else {
    prob_data <- get_problem_data(prob, solver = "ECOS")
    return(list(prob = prob, prob_data = prob_data))
  }
}

solve_CVXR_nodcp <- function(prob, prob_data) {
  ECOS_dims <- ECOS.dims_to_solver_dict(prob_data$data[["dims"]])
  solver_output <- ECOSolveR::ECOS_csolve(c = prob_data$data[["c"]],
    G = prob_data$data[["G"]],
    h = prob_data$data[["h"]],
    dims = ECOS_dims,
    A = prob_data$data[["A"]],
    b = prob_data$data[["b"]])
  
  direct_soln <- unpack_results(prob, solver_output, prob_data$chain, prob_data$inverse_data)
  alert   <- !(direct_soln$status %in% c("optimal","optimal_inaccurate"))
  
  return(list(obj = direct_soln$value, x = NULL, flag = alert))
}



solve_nloptr <- function(xt, x0, G, beta, Q, Jtot, S, lb, dire, QQ, QQ2, p.int, KMI, lasso, opt.list) {
  
  if (lasso == TRUE) { # handle L1 norm + inequality constraint
    
    x <- CVXR::Variable(Jtot+KMI)
    
    objective  <- CVXR::Minimize(-sum(CVXR::multiply(xt,x - beta)))
    
    constraints <- list(CVXR::quad_form(x, Q) + sum(CVXR::multiply(a, x)) + d <= 0)
    
    if (lb[1] > - Inf) {
      constraints <- append(constraints, list(x[1:Jtot] >= lb))
    }
    
    j.lb <- 1
    for (i in seq_len(I)) {
      j.ub <- j.lb + J[[i]] - 1 
      constraints <- append(constraints, list(CVXR::norm1(x[j.lb:j.ub]) <= QQ[i]))
      j.lb <- j.ub + 1
    }  
    
    prob <- CVXR::Problem(objective, constraints)
    sol <- CVXR::solve(prob)
    alert   <- !(sol$status %in% c("optimal","optimal_inaccurate"))
    
    if (alert == TRUE) {
      lb.est <- NA
    } else {
      lb.est <- sol$value
    }
    
  } else {
    
    if (dire == "<=") {
      res.lb <-   nloptr(x0 = c(x0, rep(0,KMI)),
        eval_f = obj.fun.min,
        lb = c(lb, rep(-Inf,KMI)),
        ub = c(rep(Inf,Jtot), rep(Inf,KMI)),
        eval_g_ineq = double.ineq,
        opts = opt.list,
        xt = xt, beta = beta, Q = Q, G = G, J = Jtot, KMI = KMI, 
        QQ = QQ, p.int = p.int, S = S)
      
    } else if (dire == "==") {
      res.lb <-   nloptr(x0 = c(x0, rep(0,KMI)),
        eval_f = obj.fun.min,
        lb = c(lb, rep(-Inf,KMI)),
        ub = c(rep(Inf,Jtot), rep(Inf,KMI)),
        eval_g_eq = norm.equal,
        eval_g_ineq = single.ineq,
        opts = opt.list,
        xt = xt, beta = beta, Q = Q, G = G, J = Jtot, KMI = KMI, 
        QQ = QQ, p.int = p.int, S = S)
      
    } else if (dire == "==/<=") {
      res.lb <-   nloptr(x0 = c(x0, rep(0,KMI)),
        eval_f = obj.fun.min.sr,
        lb = c(lb, rep(-Inf,KMI)),
        ub = c(rep(Inf,Jtot), rep(Inf,KMI)),
        eval_g_eq = norm.equal.sr,
        eval_g_ineq = double.ineq.sr,
        opts = opt.list,
        xt = xt, beta = beta, Q = Q, G = G, J = Jtot, KMI = KMI, 
        Q1 = QQ, Q2 = QQ2, S = S)
      
    } else if (dire == "NULL") {
      res.lb <-   nloptr(x0 = c(x0, rep(0,KMI)),
        eval_f = obj.fun.min,
        lb = c(lb, rep(-Inf,KMI)),
        ub = c(rep(Inf,Jtot), rep(Inf,KMI)),
        eval_g_ineq = single.ineq,
        opts = opt.list,
        xt = xt, beta = beta, Q = Q, G = G, J = Jtot, KMI = KMI, 
        QQ = QQ, p.int = p.int, S = S)  
    }
    
    alert <- res.lb$status < 0 | res.lb$status >= 5
  }
  return(list(obj = res.lb$objective, x = res.lb$solution, flag = alert))    
}


solverGetData <- function(data, w.constr) {
  
  if (methods::is(data, "scdata") == TRUE) {
    class.type <- "scpi_data"
  } else if (methods::is(data, "scdataMulti") == TRUE) {
    class.type <- "scpi_data_multi"
  }
  
  V.type <- V <- "separate"
  
  sc.pred <- scest(data = data, w.constr = w.constr, V = V)
  
  A  <- sc.pred$data$A       # Features of treated unit
  B  <- sc.pred$data$B       # Features of control units
  C  <- sc.pred$data$C       # Covariates for adjustment
  Z  <- sc.pred$data$Z       # B and C column-bind
  Y.donors   <- data$Y.donors        # Outcome variable of control units
  K  <- sc.pred$data$specs$K     # Number of covs for adjustment per feature
  KM <- sc.pred$data$specs$KM    # Dimension of r (total number of covs for adj)
  J  <- sc.pred$data$specs$J     # Number of donors
  M  <- sc.pred$data$specs$M     # Number of features
  T0 <- sc.pred$data$specs$T0.features   # Time periods used per feature
  T1 <- sc.pred$data$specs$T1.outcome    # Number of out-of-sample periods
  features   <- sc.pred$data$specs$features      # Name of features
  constant   <- sc.pred$data$specs$constant      # Logical indicating whether a constant is included
  out.feat   <- sc.pred$data$specs$out.in.features   # Logical indicating whether the outcome variable is among features
  coig.data  <- sc.pred$data$specs$cointegrated.data # Logical indicating whether B is cointegrated
  w.constr   <- sc.pred$est.results$w.constr     # Constraints on w
  V  <- sc.pred$est.results$V    # Weighting matrix
  w  <- sc.pred$est.results$w    # Estimated vector of weights
  r  <- sc.pred$est.results$r    # Estimated coefficients of covariates
  b  <- sc.pred$est.results$b    # w and r column-bind
  Y.post.fit <- sc.pred$est.results$Y.post.fit   # Estimated post-treatment outcome for SC unit
  res    <- sc.pred$est.results$res      # Residuals from estimation
  outcome.var <- sc.pred$data$specs$outcome.var   # name of outcome variable
  sc.effect  <- sc.pred$data$specs$effect    # Causal quantity of interest
  P  <- sc.pred$data$P       # Matrix for out-of-sample prediction
  
  if (class.type == "scpi_data") {
    Jtot   <- J
    KMI    <- KM
    I  <- 1
    T0.tot <- sum(T0)        # Total number of observations used in estimation
    T0.M   <- T0.tot
    T1.tot <- T1
    features   <- list(features)
    out.feat   <- list(out.feat)
    T0     <- list(T0)
    names(T0)  <- sc.pred$data$specs$treated.units
    
  } else if (class.type == "scpi_data_multi") {
    J  <- unlist(J)
    Jtot   <- sum(J)
    KMI    <- data$specs$KMI      # total number of covariates used for adjustment
    I  <- data$specs$I        # number of treated units
    T0.M   <- unlist(lapply(data$specs$T0.features, sum)) # observations per treated unit
    T0.tot <- sum(T0.M)       # Total number of observations used in estimation
    T1.tot <- sum(unlist(T1))         # Total number of observations post-treatment
  }
  
  rho <- "type-1"
  rho.max <- 0.2
  u.missp <- TRUE
  u.sigma <- "HC1"
  u.order <- 1
  u.lags <- 0
  u.design <- NULL
  u.alpha <- 0.05
  e.method = "all"
  e.order = 1
  e.lags = 0
  e.design = NULL
  e.alpha = 0.05  
  lgapp  <- "generalized"
  cores <- 1
  verbose <- FALSE
  
  A.list  <- mat2list(A)
  B.list  <- mat2list(B)
  
  if (is.null(C) == FALSE) {
    if (ncol(C) > 0) {
      C.list  <- mat2list(C)
    } else {
      C.list <- rep(list(NULL), I)
      names(C.list) <- sc.pred$data$specs$treated.units
    }
  } else {
    C.list <- rep(list(NULL), I)
    names(C.list) <- sc.pred$data$specs$treated.units
  }
  
  P.list <- mat2list(P)
  
  if (!is.null(sc.pred$data$P.diff)) {
    Pd.list <- mat2list(sc.pred$data$P.diff)
  } else {
    Pd.list <- rep(list(NULL), I)
  }
  
  V.list  <- mat2list(V)
  w.list  <- mat2list(as.matrix(w))
  res.list <- mat2list(res)
  Y.d.list <- mat2list(Y.donors)
  
  #############################################################################
  #############################################################################
  ### Estimate In-Sample Uncertainty
  #############################################################################
  #############################################################################
  
  if (class.type == "scpi_data") {
    w.constr.list <- list(w.constr)
    names(w.constr.list) <- sc.pred$data$specs$treated.units
  } else if (class.type == "scpi_data_multi") {
    w.constr.list <- w.constr
  }
  
  w.star <- index.w <- rho.vec <- Q.star <- Q2.star <- f.id <- e.res <- u.names <- e.rownames <- e.colnames <- e1.rownames <- c()
  u.des.0 <- e.des.0 <- e.des.1 <- matrix(NA, 0, 0)
  w.constr.inf <- list()
  
  for (i in seq_len(I)) {
    ## Regularize W and local geometry (treated unit by treated unit)
    loc.geom <- local.geom(w.constr.list[[i]], rho, rho.max, res.list[[i]], B.list[[i]], 
      C.list[[i]], coig.data[[i]], T0.M[[i]], J[[i]], w.list[[i]], verbose)
    
    w.star  <- c(w.star, loc.geom$w.star)
    index.w <- c(index.w, loc.geom$index.w)
    w.constr.inf <- append(w.constr.inf, list(loc.geom$w.constr))
    rho.vec <- c(rho.vec, loc.geom$rho)
    Q.star  <- c(Q.star, loc.geom$Q.star)
    Q2.star <- c(Q2.star, loc.geom$Q2.star)
    index.i <- c(loc.geom$index.w, rep(TRUE, KM[[i]]))
    
    # Extract feature id from rownames of B
    feature.id <- unlist(purrr::map(stringr::str_split(rownames(B.list[[i]]), "\\."), 2))
    
    for (f in features[[i]]) {
      len.feat <- sum(feature.id == f)
      if (len.feat <= u.lags && u.lags > 0) {
        if (verbose) {
          warning("At least one of your features is observed for less periods than the number of lags, u.lags reverted to 0.", immediate. = TRUE, call. = FALSE)
        }
        u.lags <- 0
      }
    }
    
    ## Prepare design matrix for in-sample uncertainty
    obj <- u.des.prep(B.list[[i]], C.list[[i]], u.order, u.lags, coig.data[[i]],
      T0.M[i], constant[[i]], index.i, loc.geom$index.w,
      features[[i]], feature.id, u.design, res.list[[i]])
    u.names <- c(u.names, colnames(obj$u.des.0))
    
    u.des.0 <- Matrix::bdiag(u.des.0, obj$u.des.0)
    f.id <- c(f.id, as.factor(feature.id))
    
    ## Prepare design matrices for out-of-sample uncertainty
    e.des <- e.des.prep(B.list[[i]], C.list[[i]], P.list[[i]], e.order, e.lags,
      res.list[[i]], sc.pred, Y.d.list[[i]], out.feat[[i]],
      features[[i]], J[[i]], index.i, loc.geom$index.w,
      coig.data[[i]], T0[[i]][outcome.var], T1[[i]], constant[[i]], 
      e.design, Pd.list[[i]], sc.pred$data$specs$effect, I)
    e.res  <- c(e.res, e.des$e.res)
    e.rownames <- c(e.rownames, rownames(e.des$e.res))
    cnames <- rep(paste0(names(w.constr.list)[[i]], "."), ncol(e.des$e.des.0))
    e.colnames <- c(e.colnames, cnames)
    
    if (sc.pred$data$specs$effect == "time") {
      trname <- unlist(purrr::map(stringr::str_split(rownames(e.des$e.des.0)[1], "\\."), 1))
      rnames <- paste(trname, as.character(c(1:nrow(e.des$e.des.1))), sep=".")
      e1.rownames <- c(e1.rownames, rnames)
    }
    
    e.des.0 <- Matrix::bdiag(e.des.0, e.des$e.des.0)
    e.des.1 <- Matrix::bdiag(e.des.1, e.des$e.des.1)
  }
  
  # Create an index that selects all non-zero weights and additional covariates
  index <- c(index.w, rep(TRUE, KMI))
  
  if (lgapp == "generalized") {
    beta <- b # we use rho only to impose sparsity on B when predicting moments
    Q <- c()
    for (i in seq_len(I)) {
      Q <- c(Q, w.constr.list[[i]]$Q)
    }
    
    reg.geom <- local.geom.2step(w, r, rho.vec, w.constr.list, Q, I)
    Q.star <- reg.geom$Q
    lb <- reg.geom$lb
    
  } else if (lgapp == "linear") { # we use rho to regularize w too
    beta <- c(w.star, r)
    lb <- c()
    for (i in seq_len(I)) {
      lb <- c(lb, rep(w.constr.inf[[i]]$lb, J[i]))
    }
  }
  
  names(beta) <- names(sc.pred$est.results$b)
  
  # Transform sparse matrices to matrices
  e.des.0 <- as.matrix(e.des.0)
  e.des.1 <- as.matrix(e.des.1)
  u.des.0 <- as.matrix(u.des.0)
  e.res  <- as.matrix(e.res)
  colnames(u.des.0) <- u.names
  rownames(e.res) <- e.rownames
  rownames(e.des.0) <- e.rownames
  colnames(e.des.0) <- e.colnames
  if (sc.pred$data$specs$effect == "time")  {
    rownames(e.des.1) <- e1.rownames
  } else {
    rownames(e.des.1) <- rownames(P)
  }
  colnames(e.des.1) <- e.colnames
  
  #############################################################################
  ###########################################################################
  # Remove NA - In Sample Uncertainty
  X <- cbind(A, res, u.des.0, Z, f.id)
  XX <- na.omit(X)
  j1 <- 1
  j2 <- 2
  j3 <- j2 + 1
  j4 <- j2 + ncol(u.des.0)
  j5 <- j4 + 1
  j6 <- ncol(XX) - 1
  
  A.na  <- XX[, j1, drop = FALSE]
  res.na <- XX[, j2, drop = FALSE]
  u.des.0.na <- XX[, j3:j4, drop = FALSE]
  Z.na  <- XX[, j5:j6, drop = FALSE]
  f.id.na   <- XX[, ncol(XX), drop = FALSE]
  
  active.features <- rowSums(is.na(X)) == 0
  V.na <- V[active.features, active.features]
  
  # Effective number of observation used for inference (not yet adjusted for df used)
  TT <- nrow(Z.na)
  
  # Remove NA - Out of Sample Uncertainty
  X <- cbind(e.res, e.des.0)
  XX <- na.omit(X)
  e.res.na  <- XX[, 1, drop = FALSE]
  e.des.0.na <- XX[, -1, drop = FALSE]
  
  # Proceed cleaning missing data in the post-treatment period
  P.na <- na.omit(P)
  
  #############################################################################
  ########################################################################
  ## Estimate E[u|H], V[u|H], and Sigma
  # If the model is thought to be misspecified then E[u|H] is estimated
  if (u.missp == TRUE) {
    T.u <- nrow(u.des.0.na)
    u.des.list <- mat2list(u.des.0.na)
    f.id.list <- mat2list(f.id.na)
    u.des.0.flex <- matrix(NA, 0, 0)
    u.des.0.noflex <- matrix(NA, 0, 0)
    for (i in seq_len(I)) {
      u.des.0.flex <- Matrix::bdiag(u.des.0.flex, DUflexGet(u.des.list[[i]], C.list[[i]],
        f.id.list[[i]], M[[i]]))
      u.des.0.noflex <- Matrix::bdiag(u.des.0.noflex, u.des.list[[i]])
    }
    
    df.U <- T.u - 10
    
    u.simple <- df.U <= ncol(u.des.0.noflex)
    u.noflex <- (df.U > ncol(u.des.0.noflex)) & (df.U <= ncol(u.des.0.flex))
    u.flex <- df.U > ncol(u.des.0.flex)
    
    if (u.simple) {
      if (verbose && (u.order > 0 || u.lags > 0)) {
        warning(paste0("One of u.order > 0 and u.lags > 0 was specified, however the current number of observations (",
          T.u, ") used to estimate conditional moments of the pseudo-residuals ",
          "is not larger than the number of parameters used in estimation (",ncol(u.des.0.flex),") plus 10. ",
          "To avoid over-fitting issues u.order and u.lags were set to 0."), immediate. = TRUE, call. = FALSE)
      }
      u.des.0.na <- matrix(1, nrow = T.u, 1)
      u.order <- 0
      u.lags <- 0
    } else if (u.noflex) {
      if (verbose && (u.order > 0 || u.lags > 0)) {
        warning(paste0("The current number of observations (",T.u,") used to estimate conditional moments of the pseudo-residuals ",
          "is not larger than the number of parameters used in estimation (",ncol(u.des.0.flex),") plus 10 when allowing for a ",
          "feature specific model. To avoid over-fitting issues, the conditional moments of the pseudo-residuals are predicted with ",
          "the same model across features."), immediate. = TRUE, call. = FALSE)
      }
      u.des.0.na <- as.matrix(u.des.0.noflex)
      
    } else if (u.flex){
      u.des.0.na <- as.matrix(u.des.0.flex)
      
    }
    
    u.mean <- lm(res.na ~ u.des.0.na - 1)$fitted.values
    params.u <- ncol(u.des.0.na)
    
  } else if (u.missp == FALSE) {
    u.mean <- 0
    params.u <- 0
    T.u <- 0
  }
  
  # Estimate degrees of freedom to be used for V[u|H] (note that w is pre-regularization)
  df <- df.EST(w.constr = w.constr.list[[1]], w = w, B = B, J = Jtot, KM = KMI)
  
  # Use HC inference to estimate V[u|H]
  result <- u.sigma.est(u.mean = u.mean, u.sigma = u.sigma, res = res.na,
    Z = Z.na, V = V.na, index = index, TT = TT, df = df)
  Sigma <- result$Sigma
  Omega <- result$Omega
  Sigma.root <- sqrtm(Sigma)
  
  Q <- t(Z.na) %*% V.na %*% Z.na / TT
  
  Jtot <- sum(unlist(J))
  
  ns <- ECOS_get_n_slacks(w.constr.inf[[1]], Jtot, I)
  
  Qreg <- matRegularize(Q)
  red <- nrow(Q) - nrow(Qreg)
  regul <- TRUE
  
  if (red==0) {
    regul <- FALSE
    Qreg <- sqrtm(Q)
  }  
  
  data <- list()
  data[["dims"]] <- ECOS_get_dims(Jtot, J, KMI, w.constr.inf[[1]], I, red)
  data[["A"]] <- ECOS_get_A(J, Jtot, KMI, I, w.constr.inf[[1]], ns)
  data[["b"]] <- ECOS_get_b(Q.star, Q2.star, w.constr.inf[[1]])
  
  zeta   <- rnorm(length(beta))
  G  <- Sigma.root %*% zeta
  
  a <- -2 * G - 2 * c(t(beta) %*% Q)
  d <- 2 * sum(G * beta) + sum(beta * (Q %*% beta))
  
  data[["G"]] <- ECOS_get_G(Jtot, KMI, J, I, a, Qreg, w.constr.inf[[1]], ns, red)
  data[["h"]] <- ECOS_get_h(d, lb, J, Jtot, KMI, I, w.constr.inf[[1]], Q.star, Q2.star, red)
  
  xt <- P.na[1, ]
  data[["c"]] <- ECOS_get_c(-xt, ns)  
  
  S <- matrix(0, nrow = I, ncol = Jtot)
  
  w.store <- c()
  r.store <- c()
  j.lb <- 1
  j.ub <- J[[1]]
  
  for (i in seq_len(I)) {
    if (i > 1){
      j.lb <- j.ub + 1
      j.ub <- j.lb + J[[i]] - 1
    }
    S[i, j.lb:j.ub] <- 1
  }
  
  if (w.constr.inf[[1]][["p"]] == "no norm") p.int <- 0
  if (w.constr.inf[[1]][["p"]] == "L1") p.int <- 1
  if (w.constr.inf[[1]][["p"]] == "L2") p.int <- 2
  if (w.constr.inf[[1]][["p"]] == "L1-L2") p.int <- NULL  
  
  # Algorithm initial value is lower bound unless -Inf
  x0 <- lb
  x0[is.infinite(lb)] <- 0
  
  opt.list <- prepareOptions(NULL, w.constr.infinf[[1]][["p"]], w.constr.inf[[1]][["dir"]], lb, "scpi", I)
  
  return(list(xt=xt, beta=beta, J=J, Jtot=Jtot, KMI=KMI, I=I, lb=lb, Q1=Q.star, Q2=Q2.star, x0=x0, opt.list=opt.list,
    G=G, Q=Q, a=a, d=d, S=S, Sigma.root=Sigma.root, p=p.int, p.str = w.constr.inf[[1]][["p"]], 
    dire=w.constr.inf[[1]][["dir"]], dataEcos=data, ns=ns, P=P.na, w.constr.inf = w.constr.inf))  
}


blockdiag <- function(I, Jtot, J, KMI, ns, slack = FALSE) {
  mat <- matrix(0, nrow = I, ncol = Jtot + KMI + ns)
  j.lb <- 1
  j.ub <- J[[1]]
  
  if (slack == TRUE) {
    j.lb <- j.lb + Jtot + KMI
    j.ub <- j.ub + Jtot + KMI
  }
  
  for (i in seq_len(I)) {
    if (i > 1){
      j.lb <- j.ub + 1
      j.ub <- j.lb + J[[i]] - 1
    }
    
    mat[i, j.lb:j.ub] <- 1
  }
  
  return(mat)
}

blockdiagRidge <- function(Jtot, J, KMI, I) {
  
  mat <- matrix(0, Jtot + 2 * I, Jtot + KMI + I + 1)
  
  i.lb <- 1 + 2
  i.ub <- J[[1]] + 2
  j.lb <- 1
  j.ub <- J[[1]]
  
  for (i in seq_len(I)) {
    if (i > 1){
      j.lb <- j.ub + 1
      j.ub <- j.lb + J[[i]] - 1
      i.lb <- i.ub + 1 + 2
      i.ub <- i.lb + J[[i]] - 1
    }
    
    mat[(i.lb - 2):(i.lb - 1), Jtot+KMI+i] <- c(-1, 1)
    mat[i.lb:i.ub, j.lb:j.ub] <- -diag(2, i.ub - i.lb + 1, j.ub - j.lb + 1)
  }
  
  return(mat)
}

ECOS_get_n_slacks <- function(w.constr, Jtot, I) {
  
  n_slacks <- 1
  
  # in lasso we add one slack per component of w to handle the abs value
  if (w.constr[["p"]] == "L1" & w.constr[["dir"]] == "<=") { # lasso
    n_slacks <- Jtot + n_slacks
  }
  
  # in ridge we have two hyperbolic constraints (norm and loss function)
  if (w.constr[["p"]] == "L2" & w.constr[["dir"]] == "<=") { # ridge
    n_slacks <- I + n_slacks
  }
  
  # L1-L2 combines ridge and simplex slack variables
  if (w.constr[["p"]] == "L1-L2") { # L1-L2
    n_slacks <- I + n_slacks
  }
  
  return(n_slacks)
}

ECOS_get_dims <- function(Jtot, J, KMI, w.constr, I, red) {
  
  if (w.constr[["p"]] == "L1" & w.constr[["dir"]] == "==") { # simplex
    dims <- list("l" = Jtot + 1, "q" = list(Jtot + KMI + 2 - red), "e" = 0)
    
  } else if (w.constr[["p"]] == "L1" & w.constr[["dir"]] == "<=") { # lasso
    dims <- list("l" = 1 + 2*Jtot + I, "q" = list(Jtot + KMI + 2 - red), "e" = 0)
    
  } else if (w.constr[["p"]] == "L2" & w.constr[["dir"]] == "<=") { # ridge
    dims <- list("l" = 1 + I, "q" = append(lapply(J, function(i) i+2), Jtot + KMI + 2 - red), "e" = 0)
    
  } else if (w.constr[["p"]] == "L1-L2") { # L1-L2
    dims <- list("l" = 1 + I + Jtot, "q" = append(lapply(J, function(i) i+2), Jtot + KMI + 2 - red), "e" = 0)
    
  } else if (w.constr[["p"]] == "no norm") { # ols
    dims <- list("l" = 1, "q" = list(Jtot + KMI + 2 - red), "e" = 0)
    
  }
  
  return(dims)
}

ECOS_get_c <- function(xt, ns) {
  C <- c(xt, rep(0, ns))
  return(C)
}

ECOS_get_A <- function(J, Jtot, KMI, I, w.constr, ns) {
  
  if ((w.constr[["p"]] == "L1" & w.constr[["dir"]] == "==") || w.constr[["p"]] == "L1-L2") { # simplex, L1-L2
    
    A <- blockdiag(I, Jtot, J, KMI, ns)
    
  } else  { # ols, lasso, ridge
    
    A <- matrix(NA, 0, 0)
    
  }
  
  return(methods::as(A, "sparseMatrix"))
}

ECOS_get_b <- function(Q1, Q2, w.constr) {
  
  if ((w.constr[["p"]] == "L1" & w.constr[["dir"]] == "==") || w.constr[["p"]] == "L1-L2") { # simplex, L1-L2
    b <- Q1
    
  } else { # ols, lasso, ridge
    b <- NULL
  }
  
  return(b)
}

ECOS_get_G <- function(Jtot, KMI, J, I, a, Q, w.constr, ns, red) {
  
  if (w.constr[["p"]] == "L1" & w.constr[["dir"]] == "==") { # simplex
    
    G <- rbind(c(a, 1),              # linear part of QF
      cbind(-diag(1,Jtot), matrix(0, Jtot, KMI), matrix(0, Jtot, ns)),  # lower bounds on w
      c(rep(0, Jtot+KMI),-1),           # SOC definition (||sqrt(Q)beta|| <= t)
      c(rep(0, Jtot+KMI), 1),           
      cbind(-2*Q, rep(0, Jtot+KMI-red)))
    
  } else if (w.constr[["p"]] == "L1" & w.constr[["dir"]] == "<=") { # lasso, x = (beta, z, t)
    
    G <- rbind(c(a, rep(0, ns-1), 1),            # linear part of QF
      cbind(diag(1, Jtot), matrix(0, Jtot, KMI), diag(1, Jtot), rep(0, Jtot)),  # z >= -w
      cbind(-diag(1, Jtot), matrix(0, Jtot, KMI), diag(1, Jtot), rep(0, Jtot)), # z >= w
      -blockdiag(I, Jtot, J, KMI, ns, TRUE),        # norm-inequality constraint 
      c(rep(0, Jtot+KMI+Jtot),-1),          # SOC definition (||sqrt(Q)beta|| <= t)
      c(rep(0, Jtot+KMI+Jtot), 1),          
      cbind(-2*Q, matrix(0, Jtot+KMI-red, Jtot + 1)))
    
  } else if (w.constr[["p"]] == "L2" & w.constr[["dir"]] == "<=") { # ridge, x = (beta, s, t)
    
    G <- rbind(c(a, rep(0, I), 1),           # linear part of QF
      cbind(matrix(0, I, Jtot+KMI), diag(1, I, I), rep(0, I)),      # s <= Q1^2
      blockdiagRidge(Jtot, J, KMI, I),          # SOC definition (||w|| <= s)
      c(rep(0, Jtot+KMI), rep(0, I), -1),           # SOC definition (||sqrt(Q)beta|| <= t)
      c(rep(0, Jtot+KMI), rep(0, I), 1),          
      cbind(-2*Q, matrix(0, Jtot+KMI-red, I + 1)))
    
  } else if (w.constr[["p"]] == "L1-L2") { # L1-L2, x = (beta, s, t)
    
    G <- rbind(c(a, rep(0, ns-1), 1),            # linear part of QF
      cbind(-diag(1,Jtot), matrix(0, Jtot, KMI), matrix(0, Jtot, ns)),  # lower bounds on w
      cbind(matrix(0, I, Jtot+KMI), diag(1, I, I), rep(0, I)),      # s <= Q2^2
      blockdiagRidge(Jtot, J, KMI, I),          # SOC definition (||w||_2 <= s)
      c(rep(0, Jtot+KMI), rep(0, I), -1),           # SOC definition (||sqrt(Q)beta||_2 <= t)
      c(rep(0, Jtot+KMI), rep(0, I), 1),          
      cbind(-2*Q, matrix(0, Jtot+KMI-red, I + 1)))
    
  } else if (w.constr[["p"]] == "no norm") { # ols
    
    G <- rbind(c(a, 1),              # linear part of QF
      c(rep(0, Jtot+KMI),-1),           # SOC definition (||sqrt(Q)beta|| <= t) 
      c(rep(0, Jtot+KMI), 1),          
      cbind(-2*Q, rep(0, Jtot+KMI-red)))
    
  }
  
  return(methods::as(G, "sparseMatrix"))
}


ECOS_get_h <- function(d, lb, J, Jtot, KMI, I, w.constr, Q1, Q2, red) {
  
  if (w.constr[["p"]] == "L1" & w.constr[["dir"]] == "==") { # simplex
    
    h <- c(-d,     # linear part of QF
      -lb,        # lower bounds of w
      1, 1, rep(0,Jtot+KMI-red))  # SOC definition
    
  } else if (w.constr[["p"]] == "L1" & w.constr[["dir"]] == "<=") { # lasso
    
    h <- c(-d,     # linear part of QF 
      rep(0, 2*Jtot),     # abs(w) <= z
      Q1,     # norm-inequality constraints
      1, 1, rep(0,Jtot+KMI-red))  # SOC definition
    
  } else if (w.constr[["p"]] == "L2" & w.constr[["dir"]] == "<=") { # ridge
    
    aux <- unlist(lapply(1:I, function(x) c(1,1,rep(0,J[[x]]))))
    
    h <- c(-d,     # linear part of QF
      Q1^2,       # s <= Q1^2
      aux,        # SOC definition (||w|| <= s)
      1, 1, rep(0,Jtot+KMI-red))  # SOC definition (||sqrt(Q)beta|| <= t)
    
  } else if (w.constr[["p"]] == "L1-L2") { # L1-L2
    
    aux <- unlist(lapply(1:I, function(x) c(1,1,rep(0,J[[x]]))))
    
    h <- c(-d,     # linear part of QF
      -lb,        # lower bounds of w
      Q2^2,       # s <= Q2^2
      aux,        # SOC definition (||w||_2 <= s)
      1, 1, rep(0,Jtot+KMI-red))  # SOC definition (||sqrt(Q)beta|| <= t)
    
  } else if (w.constr[["p"]] == "no norm") { # ols
    
    h <- c(-d,     # linear part of QF
      1, 1, rep(0,Jtot+KMI-red))  # SOC definition
    
  }
  
  return(h)
}



# prepare algorithm options
prepareOptions <- function(opt.list, p, dire, lb, input, I = 1) {
  
  if (is.null(opt.list$algorithm)) {
    opt.list$algorithm <- "NLOPT_LD_SLSQP"
  }
  
  if (is.null(opt.list$xtol_rel)) opt.list$xtol_rel <- 1.0e-8
  if (is.null(opt.list$xtol_abs)) opt.list$xtol_abs <- 1.0e-8
  if (is.null(opt.list$ftol_rel)) opt.list$ftol_rel <- 1.0e-8
  if (is.null(opt.list$ftol_abs)) opt.list$ftol_abs <- 1.0e-8
  if (is.null(opt.list$maxeval))  opt.list$maxeval <- 5000
  
  if (dire == "NULL") {
    if (is.null(opt.list$tol_constraints_ineq)) {
      opt.list$tol_constraints_ineq <- 1.0e-8
    } else {
      opt.list$tol_constraints_ineq <- opt.list$tol_constraints_ineq[1]
    }
    
  } else if (dire == "==") {
    if (is.null(opt.list$tol_constraints_ineq)) {
      opt.list$tol_constraints_ineq <- 1.0e-8
    } else {
      opt.list$tol_constraints_ineq <- opt.list$tol_constraints_ineq[1]
    }
    if (is.null(opt.list$tol_constraints_eq)) {
      opt.list$tol_constraints_eq <- rep(1.0e-8, I)
    } else {
      opt.list$tol_constraints_eq <- rep(opt.list$tol_constraints_eq[1], I) 
    }
    
  } else if (dire == "<=") {
    opt.list$ftol_rel <- 1.0e-8
    opt.list$ftol_abs <- 1.0e-8
    if (is.null(opt.list$tol_constraints_ineq)) {
      opt.list$tol_constraints_ineq <- rep(1.0e-8, I + 1)
    } else {
      opt.list$tol_constraints_ineq <- rep(opt.list$tol_constraints_ineq[1], I + 1) 
    }
    
  } else if (dire == "==/<=") {
    opt.list$ftol_rel <- 1.0e-8
    opt.list$ftol_abs <- 1.0e-8
    if (is.null(opt.list$tol_constraints_ineq)) {
      opt.list$tol_constraints_ineq <- rep(1.0e-8, I + 1)
    } else {
      opt.list$tol_constraints_ineq <- rep(opt.list$tol_constraints_ineq[1], I + 1)
    }
    if (is.null(opt.list$tol_constraints_eq)) {
      opt.list$tol_constraints_eq <- rep(1.0e-8, I)
    } else {
      opt.list$tol_constraints_eq <- rep(opt.list$tol_constraints_eq[1], I) 
    }
  }  
  
  return(opt.list) 
}

###############################################################################
### Auxiliary functions for estimation

# Quadratic loss function
obj.fun.est <- function(x, Z, V, A, J, QQ, KM, p) {
  f <- x %*% t(Z) %*% V %*% Z %*% x - 2 * t(A) %*% V %*% Z %*% x
  g <- 2 * t(Z) %*% V %*% Z %*% x - 2 * t(t(A) %*% V %*% Z)
  
  return(list("objective" = f,
    "gradient" = g))
}

obj.fun.est.sr <- function(x, Z, V, A, J, Q1, Q2, KMI, S) {
  f <- x %*% t(Z) %*% V %*% Z %*% x - 2 * t(A) %*% V %*% Z %*% x
  g <- 2 * t(Z) %*% V %*% Z %*% x - 2 * t(t(A) %*% V %*% Z)
  
  return(list("objective" = f,
    "gradient" = g))
}

obj.fun.est.multi <- function(x, Z, V, A, J, QQ, KMI, p, S) {
  f <- x %*% t(Z) %*% V %*% Z %*% x - 2 * t(A) %*% V %*% Z %*% x
  g <- 2 * t(Z) %*% V %*% Z %*% x - 2 * t(t(A) %*% V %*% Z)
  
  return(list("objective" = f,
    "gradient" = g))
}


## Constraint on the norm

# Single treated unit
norm.co.est <- function(x, Z, V, A, J, QQ, KM, p) {
  
  if (p == 1) {
    av <- rep(1, J)
    av[x[1:J] < 0] <- -1
    ja <- c(av, rep(0, KM))
    co <- sum(abs(x[1:J])) - QQ
  } else {
    ja <- c(2 * x[1:J], rep(0, KM))
    co <- sum(x[1:J]^2) - QQ^2
  }
  
  return(list("constraints" = co,
    "jacobian" = ja))
}

# Functions for L1-L2

norm.L1 <- function(x, Z, V, A, J, Q1, Q2, KMI, S) {
  
  av <- rep(1, J)
  av[x[1:J] < 0] <- -1
  ja <- c(av, rep(0, KMI))
  co <- S %*% abs(x[1:J]) - Q1
  
  ja.vec <- matrix(ja, nrow = nrow(S), ncol = length(ja), byrow = TRUE)
  ja.vec[, 1:J] <- ja.vec[, 1:J] * S
  
  return(list("constraints" = co,
    "jacobian" = ja.vec))  
}

norm.L2 <- function(x, Z, V, A, J, Q1, Q2, KMI, S) {
  
  ja <- c(2 * x[1:J], rep(0, KMI))
  co <- S %*% x[1:J]^2 - Q2^2
  
  ja.vec <- matrix(ja, nrow = nrow(S), ncol = length(ja), byrow = TRUE)
  ja.vec[, 1:J] <- ja.vec[, 1:J] * S
  
  return(list("constraints" = co,
    "jacobian" = ja.vec))
}

# Multiple treated units
norm.co.est.multi <- function(x, Z, V, A, J, QQ, KMI, p, S) {
  
  if (p == 1) {
    av <- rep(1, J)
    av[x[1:J] < 0] <- -1
    ja <- c(av, rep(0, KMI))
    co <- S %*% abs(x[1:J]) - QQ
  } else {
    ja <- c(2 * x[1:J], rep(0, KMI))
    co <- S %*% x[1:J]^2 - QQ^2
  }
  
  ja.vec <- matrix(ja, nrow = nrow(S), ncol = length(ja), byrow = TRUE)
  ja.vec[, 1:J] <- ja.vec[, 1:J] * S
  
  return(list("constraints" = co,
    "jacobian" = ja.vec))
}


### Auxiliary functions for inference

# Prepare objective functions
obj.fun.min <- function(x, xt, beta, Q, G, J, KMI, QQ, p.int, S) {
  f <- -sum(xt * (x - beta))
  g <- -xt
  
  return(list("objective" = f,
    "gradient" = g))
}

obj.fun.max <- function(x, xt, beta, Q, G, J, KMI, QQ, p.int, S) {
  f <- sum(xt * (x - beta))
  g <- xt
  
  return(list("objective" = f,
    "gradient" = g))
}

obj.fun.min.sr <- function(x, xt, beta, Q, G, J, KMI, Q1, Q2, S) {
  f <- -sum(xt * (x - beta))
  g <- -xt
  
  return(list("objective" = f,
    "gradient" = g))
}

obj.fun.max.sr <- function(x, xt, beta, Q, G, J, KMI, Q1, Q2, S) {
  f <- sum(xt * (x - beta))
  g <- xt
  
  return(list("objective" = f,
    "gradient" = g))
}


# Prepare inequality constraint(s): loss functions constraint + (inequality norm constraint)

# Unique inequality constraint
single.ineq <- function(x, xt, beta, Q, G, J, KMI, QQ, p.int, S) {
  a <- -2 * G - 2 * c(t(beta) %*% Q)
  d <- 2 * sum(G * beta) + sum(beta * (Q %*% beta))
  
  co <- x %*% Q %*% x + sum(a * x) + d
  ja <- 2 * Q %*% x + a
  
  return(list("constraints" = co,
    "jacobian" = ja))
}

# Inequality constraints: loss function + L1-L2 norm
double.ineq <- function(x, xt, beta, Q, G, J, KMI, QQ, p.int, S) {
  # Loss function constraint
  a <- -2 * G - 2 * c(t(beta) %*% Q)
  d <- 2 * sum(G * beta) + sum(beta * (Q %*% beta))
  
  co1 <- x %*% Q %*% x + sum(a * x) + d
  ja1 <- 2 * Q %*% x + a
  
  # Norm constraint
  if (p.int == 1) {
    co2 <- S %*% abs(x[1:J]) - QQ
    av <- rep(1, J)
    av[x[1:J] < 0] <- -1
    ja <- c(av, rep(0, KMI))
  } else {
    co2 <- S %*% (x[1:J]^2) - QQ^2
    ja <- c(2 * x[1:J], rep(0, KMI))
  }
  
  ja.vec <- matrix(ja, nrow = nrow(S), ncol = length(ja), byrow = TRUE)
  ja.vec[, 1:J] <- ja.vec[, 1:J] * S
  ja <- c(rbind(t(ja1), ja.vec))    # vectorize matrix row-by-row
  
  return(list("constraints" = c(co1,co2),
    "jacobian" = ja))
}

# Eventual equality constraint on norm
norm.equal <- function(x, xt, beta, Q, G, J, KMI, QQ, p.int, S) {
  
  if (p.int == 1) {
    co <- S %*% abs(x[1:J]) - QQ
    av <- rep(1, J)
    av[x[1:J] < 0] <- -1
    ja <- c(av, rep(0, KMI))
  } else {
    co <- S %*% (x[1:J]^2) - QQ^2
    ja <- c(2 * x[1:J], rep(0, KMI))
  }
  
  ja.vec <- matrix(ja, nrow = nrow(S), ncol = length(ja), byrow = TRUE)
  ja.vec[, 1:J] <- ja.vec[, 1:J] * S
  
  return(list("constraints" = co,
    "jacobian" = ja.vec))
}

norm.equal.sr <- function(x, xt, beta, Q, G, J, KMI, Q1, Q2, S) {
  
  co <- S %*% abs(x[1:J]) - Q1
  av <- rep(1, J)
  av[x[1:J] < 0] <- -1
  ja <- c(av, rep(0, KMI))
  
  ja.vec <- matrix(ja, nrow = nrow(S), ncol = length(ja), byrow = TRUE)
  ja.vec[, 1:J] <- ja.vec[, 1:J] * S
  
  return(list("constraints" = co,
    "jacobian" = ja.vec))
}

double.ineq.sr <- function(x, xt, beta, Q, G, J, KMI, Q1, Q2, S) {
  # Loss function constraint
  a <- -2 * G - 2 * c(t(beta) %*% Q)
  d <- 2 * sum(G * beta) + sum(beta * (Q %*% beta))
  
  co1 <- x %*% Q %*% x + sum(a * x) + d
  ja1 <- 2 * Q %*% x + a
  
  co2 <- S %*% (x[1:J]^2) - Q2^2
  ja <- c(2 * x[1:J], rep(0, KMI))
  
  ja.vec <- matrix(ja, nrow = nrow(S), ncol = length(ja), byrow = TRUE)
  ja.vec[, 1:J] <- ja.vec[, 1:J] * S
  ja <- c(rbind(t(ja1), ja.vec))    # vectorize matrix row-by-row
  
  return(list("constraints" = c(co1,co2),
    "jacobian" = ja))
}


mat2list <- function(mat, cols = TRUE){
  # select rows
  names <- strsplit(rownames(mat), "\\.")
  rnames <- unlist(lapply(names, "[[", 1))
  tr.units <- unique(rnames)
  
  # select columns
  matlist <- list()
  if (cols == TRUE) {
    if (ncol(mat) > 1) {
      names <- strsplit(colnames(mat), "\\.")
      cnames <- unlist(lapply(names, "[[", 1))
      for (tr in tr.units) {
        matlist[[tr]] <- mat[rnames == tr, cnames == tr, drop = FALSE]
      }
    } else if (ncol(mat) == 1) {
      for (tr in tr.units) {
        matlist[[tr]] <- mat[rnames == tr, 1, drop = FALSE]
      }
    } else {
      for (tr in tr.units) {
        matlist[[tr]] <- mat[rnames == tr, 0, drop = FALSE]
      }
    }
  } else if (cols == FALSE) {
    for (tr in tr.units) {
      matlist[[tr]] <- mat[rnames == tr, , drop = FALSE]
    }
  }
  
  return(matlist)
}

local.geom <- function(w.constr, rho, rho.max, res, B, C, coig.data, T0.tot, J, w, verbose) {
  
  Q  <- w.constr[["Q"]]
  Q2.star <- NULL
  
  if (is.character(rho)) rho <- regularize.w(rho, rho.max, res, B, C, coig.data, T0.tot)
  
  if ((w.constr[["name"]] == "simplex") || ((w.constr[["p"]] == "L1") && (w.constr[["dir"]] == "=="))) {
    index.w <- abs(w) > rho
    index.w <- regularize.check(w, index.w, rho, verbose)
    w.star <- w
    w.star[!index.w] <- 0
    
    Q.star <- sum(w.star)  
    
  } else if ((w.constr[["name"]] == "lasso") || ((w.constr[["p"]] == "L1") && (w.constr[["dir"]] == "<="))) {
    
    if ((sum(abs(w)) >= Q - rho*sqrt(J)) && (sum(abs(w)) <= Q)) {
      Q.star <- sum(abs(w))
    } else {
      Q.star <- Q
    }
    index.w <- abs(w) > rho
    index.w <- regularize.check(w, index.w, rho, verbose)
    
    w.star <- w
    w.star[!index.w] <- 0    
    
  } else if ((w.constr[["name"]] == "ridge") || (w.constr[["p"]] == "L2")) {
    
    if (sqrt((sum(w^2)) >= Q - rho) && (sqrt(sum(w^2)) <= Q)) {
      Q.star <- sqrt(sum(w^2))
    } else {
      Q.star <- Q
    }
    
    index.w <- rep(TRUE, length(w))
    w.star <- w
    
  } else if (w.constr[["name"]] == "L1-L2") {
    
    index.w <- abs(w) > rho
    index.w <- regularize.check(w, index.w, rho, verbose)
    w.star <- w
    w.star[!index.w] <- 0
    
    Q.star <- sum(w.star)  
    
    if (sqrt((sum(w^2)) >= Q - rho) && (sqrt(sum(w^2)) <= Q)) {
      Q2.star <- sqrt(sum(w^2))
    } else {
      Q2.star <- w.constr[["Q2"]] 
    }
    w.constr[["Q2"]] <- Q2.star
    
  } else {
    Q.star <- Q
    w.star <- w
    index.w <- rep(TRUE, length(w))
  }
  
  w.constr[["Q"]] <- Q.star
  
  return(list(w.constr = w.constr, w.star = w.star, index.w = index.w, rho = rho, Q.star = Q.star, Q2.star = Q2.star)) 
}


regularize.w <- function(rho, rho.max, res, B, C, coig.data, T0.tot) {
  if (rho == "type-1") {
    sigma.u <- sqrt(mean((res-mean(res))^2))
    sigma.bj <- min(apply(B, 2, sd))
    CC  <- sigma.u/sigma.bj
    
  } else if (rho == "type-2"){
    sigma.u  <- sqrt(mean((res-mean(res))^2))
    sigma.bj2 <- min(apply(B, 2, var))
    sigma.bj <- max(apply(B, 2, sd))
    CC   <- sigma.bj*sigma.u/sigma.bj2
    
  } else if (rho == "type-3"){
    sigma.bj2 <- min(apply(B, 2, var))
    sigma.bju <- max(apply(B, 2, function(bj) cov(bj, res)))
    CC   <- sigma.bju/sigma.bj2
  }
  
  
  if (coig.data == TRUE) { # cointegration
    c <- 1
  } else {    # iid or ar
    c <- 0.5
  }
  
  rho <- (CC*(log(T0.tot))^c)/(sqrt(T0.tot))
  
  if (is.null(rho.max) == FALSE)  rho <- min(rho, rho.max)     
  
  return(rho)
}

regularize.check <- function(w, index.w, rho, verbose) {
  if (sum(index.w) == 0) {
    index.w <- rank(-w) <= 1
    if (verbose){
      warning(paste0("Regularization paramater was too high (", round(rho, digits = 3), "). ",
        "We set it so that at least one component in w is non-zero."), immediate. = TRUE, call. = FALSE)
    }
  }
  return(index.w)
}

local.geom.2step <- function(w, r, rho.vec, w.constr, Q, I) {
  beta <- c(w, r)
  w.list <- mat2list(as.matrix(w))
  
  ## Constraint on the norm of the weights
  if (w.constr[[1]]$p == "no norm") { # Unconstrained problem
    rhoj.vec <- rho.vec # auxiliary list never used in this case
    
  } else if (w.constr[[1]]$p == "L1") { 
    rhoj.vec <- rho.vec
    w.norm <- unlist(lapply(w.list, function(x) sum(abs(x))))
    
  } else if (w.constr[[1]]$p %in% c("L2","L1-L2")) {
    rhoj.vec <- c()
    for (i in seq_len(I)) {
      rhoj.vec[i] <- 2*sum(abs(w.list[[i]]))*rho.vec[i]
    }
    w.norm <- unlist(lapply(w.list, function(x) sum(x^2)))
  }
  
  # Check if constraint is equality or inequality
  if (w.constr[[1]]$dir %in% c("<=", "==/<=")) {
    active <- 1*((w.norm - Q) > -rhoj.vec)
    Q <- active*(w.norm - Q) + Q 
  }
  
  ## Constraint on lower bound of the weights
  lb <- c()
  for (i in seq_len(I)) {
    if (w.constr[[i]]$lb == 0) {
      active <- 1*(w.list[[i]] < rhoj.vec[[i]])
      lb <- c(lb, rep(0, length(w.list[[i]])) + active*w.list[[i]])
    } else {
      lb <- c(lb, rep(-Inf, length(w.list[[i]])))
    }
  }
  
  return(list(Q = Q, lb = lb))
}


u.des.prep <- function(B, C, u.order, u.lags, coig.data, T0.tot, constant,
  index, index.w, features, feature.id, u.design, res, verbose) {
  
  ## Construct the polynomial terms in B
  if (u.order == 0) {    # Simple mean
    
    u.des.0 <- as.matrix(rep(1, T0.tot))
    
  } else if (u.order > 0) {     # Include covariates (u.order = 1 just covariates)
    
    if (coig.data == TRUE) {    # Take first differences of B and active covariates
      
      B.diff  <- NULL
      
      # Create first differences feature-by-feature of the matrix B (not of C!!)
      for (feature in features) {
        BB  <- B[feature.id == feature,]
        B.diff  <- rbind(B.diff, BB - dplyr::lag(BB))
      }
      u.des.0 <- cbind(B.diff, C)[, index, drop = FALSE]    # combine with C
      
    } else if (coig.data == FALSE) {
      
      u.des.0 <- cbind(B, C)[, index, drop = FALSE]  # take active covariates
      
    }
    
    # Augment H with powers and interactions of B (not of C!!!)
    if (u.order > 1) {
      name.tr <- lapply(strsplit(rownames(u.des.0), "\\."), "[[", 1)[[1]]
      act.B <- sum(index.w)
      u.des.poly <- poly(u.des.0[, (1:act.B), drop = FALSE], degree = u.order, raw = TRUE, simple = TRUE)
      colnames(u.des.poly) <- paste(name.tr, colnames(u.des.poly), sep = ".")
      u.des.0 <- cbind(u.des.poly,
        u.des.0[, -(1:act.B), drop = FALSE])
      
    }
    
    # Include the constant if a global constant is not present
    # In case a constant is already specified lm.fit and qfit will automatically remove
    # the collinear covariates!!
    if (constant == FALSE) {
      u.des.0 <- cbind(u.des.0, rep(1, nrow(u.des.0)))
      name.tr <- lapply(strsplit(rownames(u.des.0), "\\."), "[[", 1)[[1]]
      colnames(u.des.0) <- c(colnames(u.des.0[, -ncol(u.des.0), drop = FALSE]),
        paste0(name.tr, ".0.constant"))
    }
  }
  
  ## Construct lags of B
  if (u.lags > 0) {
    
    B.lag <- NULL
    if (coig.data == TRUE) {
      # Take first differences of B
      B.diff  <- NULL
      
      # Create first differences feature-by-feature of the matrix B (not of C!!)
      for (feature in features) {
        BB  <- B[feature.id == feature,]
        B.diff  <- rbind(B.diff, BB - dplyr::lag(BB))
      }
    }
    
    for (ll in seq_len(u.lags)) {
      B.l <- NULL
      for (feature in features) {
        if (coig.data == FALSE) {
          B.l <- rbind(B.l, dplyr::lag(B[feature.id == feature, , drop = FALSE], n = ll))
        } else {
          B.l <- rbind(B.l, dplyr::lag(B.diff[feature.id == feature, , drop = FALSE], n = ll))
        }
      }
      B.lag <- cbind(B.lag, B.l)
    }
    name.tr <- lapply(strsplit(rownames(u.des.0), "\\."), "[[", 1)[[1]]
    colnames(B.lag) <- rep(paste0(name.tr,".lag"), ncol(B.lag))
    u.des.0 <- cbind(u.des.0, B.lag[, index.w, drop = FALSE])
  }
  
  # If user provided check compatibility of the matrix and overwrite what has been created
  if (is.null(u.design) == FALSE) {
    if (is.matrix(u.design) == FALSE) {
      stop("The object u.design should be a matrix!!")
    }
    
    if (nrow(u.design) != nrow(res)) {
      stop(paste("The matrix u.design has", nrow(u.design),"rows when", nrow(res),
        "where expected!"))
    }
    u.des.0 <- u.design
  }
  
  return(list(u.des.0 = u.des.0))
}


e.des.prep <- function(B, C, P, e.order, e.lags, res, sc.pred, Y.donors, out.feat, features,
  J, index, index.w, coig.data, T0, T1, constant, e.design, P.diff.pre, effect, I) {
  
  # If the outcome variable is not among the features we need to create the
  # proper vector of residuals. Further, we force the predictors to be
  # the outcome variable of the donors
  aux <- trendRemove(P)
  C <- trendRemove(C)$mat
  index <- index[aux$sel]
  P <- aux$mat
  
  if (!is.null(P.diff.pre)) P.diff.pre <- trendRemove(as.matrix(P.diff.pre))$mat
  
  if (out.feat == FALSE) {
    e.res   <- sc.pred$data$Y.pre - sc.pred$est.results$Y.pre.fit
    
    if (coig.data == TRUE) {
      e.des.0 <- apply(Y.donors, 2, function(x) x - dplyr::lag(x))[, index.w]
      
      if (effect == "time") {
        P.first <- (P[1, ]*I - Y.donors[T0[1], ])/I
      } else {
        P.first <- P[1, ] - Y.donors[T0[1], ]
      }
      P.diff <- rbind(P.first, apply(P, 2, diff))[, index, drop = FALSE]
      e.des.1 <- P.diff
      
    } else {
      e.des.0 <- Y.donors[, index.w]
      e.des.1 <- P[, index, drop = FALSE]
    }
    
  } else if (out.feat == TRUE) {    # outcome variable is among features
    e.res <- res[1:T0[1], , drop = FALSE]
    
    ## Construct the polynomial terms in B (e.des.0) and P (e.des.1)
    if (e.order == 0) {
      
      e.des.0 <- as.matrix(rep(1, T0[1]))
      e.des.1 <- as.matrix(rep(1, T1))
      
    } else if (e.order > 0) {
      feature.id <- unlist(purrr::map(stringr::str_split(rownames(B), "\\."), 2))
      
      if (coig.data == TRUE) {
        
        ## Take first differences of B
        B.diff  <- NULL
        
        # Create first differences of the first feature (outcome) of the matrix B (not of C!!)
        BB  <- B[feature.id == features[1], ]
        B.diff  <- rbind(B.diff, BB - dplyr::lag(BB))
        
        e.des.0 <- cbind(B.diff, C[feature.id == features[1], ])[, index, drop = FALSE]
        
        
        ## Take first differences of P
        # Remove last observation of first feature from first period of P
        if (effect == "time") {
          P.first <- c((P[1, (1:J), drop = FALSE]*I - B[feature.id == features[1], , drop = FALSE][T0[1], ]),
            P[1, -(1:J), drop = FALSE]*I)/I
          
        } else {
          P.first <- c((P[1, (1:J), drop = FALSE] - B[feature.id == features[1], , drop = FALSE][T0[1], ]),
            P[1, -(1:J), drop = FALSE])
        }
        
        # Take differences of other periods
        if (nrow(P) > 2) {
          Pdiff   <- apply(P[, (1:J), drop = FALSE], 2, diff)
          P.diff  <- rbind(P.first, cbind(Pdiff, P[-1, -(1:J), drop = FALSE]))[, index, drop = FALSE]
        } else if (nrow(P) == 2) {
          Pdiff   <- t(as.matrix(apply(P[, (1:J), drop = FALSE], 2, diff)))
          P.diff  <- rbind(P.first, cbind(Pdiff, P[-1, -(1:J), drop = FALSE]))[, index, drop = FALSE]
        } else {
          P.diff <- matrix(P.first, 1, length(P.first))[, index, drop = FALSE]
        }
        e.des.1 <- P.diff
        
      } else {
        e.des.0 <- cbind(B, C)[feature.id == features[1], index, drop = FALSE]
        e.des.1 <- P[, index, drop = FALSE]
      }
      
      # Augment H with powers and interactions of B (not of C!!!)
      if (e.order > 1) {
        act.B <- sum(index.w)
        e.des.0 <- cbind(poly(e.des.0[,(1:act.B), drop = FALSE], degree = e.order, raw = TRUE, simple = TRUE),
          e.des.0[, -(1:act.B), drop = FALSE])
        e.des.1 <- cbind(poly(e.des.1[,(1:act.B), drop = FALSE], degree = e.order, raw = TRUE, simple = TRUE),
          e.des.1[, -(1:act.B), drop = FALSE])
      }
      
      # Include the constant if a global constant is not present
      # In case a constant is already specified lm.fit will automatically remove
      # the collinear covariates!!
      if (constant == FALSE) {
        e.des.0 <- cbind(e.des.0, rep(1, nrow(e.des.0)))
        e.des.1 <- cbind(e.des.1, rep(1, nrow(e.des.1)))
      }
    }
    
    nolag <- FALSE
    if (is.null(P.diff.pre) == FALSE) {
      e.des.1 <- P.diff.pre[, index, drop = FALSE]
      nolag <- TRUE
    }
    
    if (e.lags > 0 && nolag == FALSE) {
      # Construct lags of B and P
      B.lag <- NULL
      P.lag <- NULL
      
      # Take first differences of B and P
      B.diff  <- NULL
      feature.id <- unlist(purrr::map(stringr::str_split(rownames(B), "\\."), 2))
      
      # Create first differences of the first feature (outcome) of the matrix B (not of C!!)
      BB  <- B[feature.id == features[1], , drop = FALSE]
      B.diff  <- rbind(B.diff, BB - dplyr::lag(BB))
      
      ## Create first differences of P
      # Attach some pre-treatment value in order to avoid having missing values
      if (coig.data == FALSE) {
        PP <- rbind(B[feature.id == features[1], , drop = FALSE][((T0[1] - e.lags + 1):T0[1]), , drop = FALSE],
          P[, (1:J), drop = FALSE])
      } else {
        PP <- rbind(B[feature.id == features[1], , drop = FALSE][((T0[1] - e.lags):T0[1]), , drop = FALSE],
          P[, (1:J), drop = FALSE])
      }
      PP.diff <- PP - dplyr::lag(PP)
      
      for (ll in seq_len(e.lags)) {
        if (coig.data == FALSE) {
          P.l <- dplyr::lag(PP, n = ll)[, index.w, drop = FALSE][((e.lags+1):nrow(PP)), , drop = FALSE]
        } else {
          P.l <- dplyr::lag(PP.diff, n = ll)[, index.w, drop = FALSE][((e.lags+2):nrow(PP)), , drop = FALSE]
        }
        
        if (coig.data == FALSE) {
          B.l <- dplyr::lag(B[feature.id == features[1], , drop = FALSE], n = ll)[, index.w, drop = FALSE]
        } else {
          B.l <- dplyr::lag(B.diff[, , drop = FALSE], n = ll)[, index.w, drop = FALSE]
        }
        
        B.lag <- cbind(B.lag, B.l)
        P.lag <- cbind(P.lag, P.l)
      }
      e.des.0 <- cbind(e.des.0, B.lag)
      e.des.1 <- cbind(e.des.1, P.lag)
    }
  }
  
  if (is.null(e.design) == FALSE) {
    if (is.matrix(e.design) == FALSE) {
      stop("The object e.design should be a matrix!!")
    }
    
    if (nrow(e.design) != nrow(e.res)) {
      stop(paste("The matrix e.design has", nrow(e.design), "rows when", nrow(e.res),
        "where expected!"))
    }
    e.des.0 <- e.design
  }
  
  return(list(e.res = e.res, e.des.0 = e.des.0, e.des.1 = e.des.1))
}

DUflexGet <- function(u.des.0.na, C, f.id.na, M) {
  sel <- colnames(u.des.0.na) %in% colnames(C)
  D.b <- u.des.0.na[, !sel, drop = FALSE]
  D.c <- u.des.0.na[, sel, drop = FALSE]
  f.df <- data.frame(f.id.na)
  f.D <- fastDummies::dummy_cols(f.df, select_columns = "f.id", remove_selected_columns = TRUE)
  D.b.int <- matrix(NA, nrow = nrow(D.b), ncol = 0)
  for (m in seq_len(ncol(f.D))) {
    D.b.int <- cbind(D.b.int, D.b*f.D[,m])
  }
  
  D <- cbind(D.b.int, D.c)
  
  return(D)
}

trendRemove <- function(mat) {
  sel <- c()
  for (l in stringr::str_split(colnames(mat), "\\.")) {
    if (length(l) < 3) {
      sel <- c(sel, TRUE)
    } else {
      if (l[[3]] == "trend") {
        sel <- c(sel, FALSE)
      } else {
        sel <- c(sel, TRUE)
      }
    }
  }
  
  return(list(mat = mat[, sel, drop = FALSE], sel = sel))
}

aggregateUnits <- function(xx, labels) {
  xx.df <- data.frame(xx = xx, id = labels)
  x.mean <- aggregate(xx.df[, "xx"], by = list(unit = xx.df$id), FUN = mean, na.rm = TRUE)
  x.mean <- x.mean[order(as.numeric(x.mean$unit)),]$x
  return(x.mean)
}

# sqrtm regularizer for symmetric positive definite matrices
matRegularize <- function(mat, threshold = 1e-08) {
  
  matsvd <- eigen(mat)
  sel <- matsvd$values > threshold
  U <- matsvd$vectors[, sel]
  D <- diag(sqrt(matsvd$values[sel]))
  matreg <- D %*% t(U) 
  
  return(matreg)
}


df.EST <- function(w.constr, w, B, J, KM){
  if ((w.constr[["name"]] == "ols") || (w.constr[["p"]] == "no norm")) {
    df <- J 
    
  } else if ((w.constr[["name"]] == "lasso") || ((w.constr[["p"]] == "L1") &&
      (w.constr[["dir"]] == "<="))) {
    df <- sum(abs(w) >= 1e-6) 
    
  } else if ((w.constr[["name"]] == "simplex") || ((w.constr[["p"]] == "L1") &&
      (w.constr[["dir"]] == "=="))) {
    df <- sum(abs(w) >= 1e-6) - 1
    
  } else if ((w.constr[["name"]] == "ridge") || (w.constr[["name"]] == "L1-L2") ||
      (w.constr[["p"]] == "L2")) {
    d <- svd(B)$d
    d[d < 0] <- 0
    df <- sum(d^2/(d^2+w.constr[["lambda"]]))
    
  } 
  
  # add degrees of freedom coming from C block
  df <- df + KM
  
  return(df)
}


u.sigma.est <- function(u.mean, u.sigma, res, Z, V, index, TT, df) {
  
  if  (u.sigma == "HC0") { # White (1980)
    vc <- 1
  }
  
  else if (u.sigma == "HC1") { # MacKinnon and White (1985)
    vc <- TT/(TT-df)
  }
  
  else if (u.sigma == "HC2") { # MacKinnon and White (1985)
    PP <- Z %*% base::solve(t(Z)%*% V %*% Z) %*% t(Z) %*% V
    vc <- 1/(1-diag(PP))
  }
  
  else if (u.sigma == "HC3") { # Davidson and MacKinnon (1993)
    PP <- Z %*% base::solve(t(Z)%*% V %*% Z) %*% t(Z) %*% V
    vc <- 1/(1-diag(PP))^2
  }
  
  else if (u.sigma == "HC4") { # Cribari-Neto (2004)
    PP <- Z %*% base::solve(t(Z)%*% V %*% Z) %*% t(Z) %*% V
    CN <- as.matrix((TT)*diag(PP)/df)
    dd <- apply(CN, 1, function(x) min(4, x))
    vc <- as.matrix(NA, length(res), 1)
    for (ii in seq_len(length(res))) {
      vc[ii] <- 1/(1 - diag(PP)[ii])^dd[ii]
    }
  }
  
  Omega <- diag(c((res-u.mean)^2)*vc)
  Sigma <- t(Z) %*% V %*% Omega %*% V %*% Z / (TT^2)
  
  return(list(Omega = Omega, Sigma = Sigma))
}

sqrtm <- function(A) {
  decomp <- svd(A)
  decomp$d[decomp$d < 0] <- 0
  rootA <- decomp$u %*% diag(sqrt(decomp$d)) %*% t(decomp$u)
  return(rootA)
}
