/****************************************************************************/
/*                                                          				*/
/*						SAS Macro surveyhlm.sas               			    */
/*		SAS 9.4 TS Level 1M1 X64_7PROWindows NT Server-Version				*/
/*                                                          				*/
/****************************************************************************/
/*                                                          				*/
/*	Computes three-level regression coefficients, random components and 	*/
/*	their standard errors for a linear mixed model with a dependent 		*/
/*	variable (ROOTPV) and continues independent variable(s) on level 		*/
/*	one (XVAR1), level two (XVAR2) or level three (XVAR3) and/or			*/ 
/*	categorical independent variable(s) on level one (CVAR1), level 		*/
/*	two (CVAR2) or level three (CVAR3). Plausible values are possible as 	*/
/*	dependent variable (NPV). Class mean centering on level 2 (CCENT2) or	*/
/*	level 3 (CCENT3) and group mean centering (GCENT) of the predictor		*/
/*	variable(s) are supported. A random intercept or random slopes			*/ 
/*	on level two (RSLOPE2) or level three (RSLOPE3) can be specified.		*/ 
/*	Weights can be specified on level one (L1WGT), level two (L2WGT) and	*/ 
/*	level three (L3WGT), or as a combined weight (WGT). Scaling of level	*/ 
/*	one (SFW), level two (SFB2) and	level three (SFB3) is supported.		*/ 
/*	Standard errors can be model based, or calculated with jackknife		*/ 
/*	procedure (JKREP, JKZONE) or through user supplied replication weights	*/ 
/*	(REPWP). For jackknife either a full replication procedure or a half	*/ 
/*	replication procedure can be used (JKTYP). If plausible values are used	*/ 
/*	as dependent variable and standard errors are calculated with either	*/ 
/*	jackknife or user supplied replication weights, then the user can		*/ 
/*	decide whether all plausible values are used for the calculation of		*/ 
/*	the standard errors or only the first plausible value (SHRTCUT).		*/
/*                                                          				*/
/****************************************************************************/
/*																			*/	
/* Developed under SAS Base 9.4_M1, SAS/IML 12.3_M1 and SAS/STAT 13.1_M1	*/
/*																			*/
/****************************************************************************/
/*   																		*/
/* Depends on macro NAMEP, NEWDATASETNAME, PVARNAME and RENAMEVARIABLE		*/
/*																			*/
/****************************************************************************/ 



%macro surveyhlm(DATN=, ROOTPV=, NPV=1, NOINT=N, XVAR1=, XVAR2=, XVAR3=, CVAR1=, CVAR2=, CVAR3=, CCENT2=, CCENT3=, 
GCENT=, NORINT2=N, NORINT3=N, RSLOPE2=, RSLOPE3=, NEST1=, NEST2=, NEST3=, L1WGT=, L2WGT=, L3WGT=, WGT=, SFW=1, SFB2=2, 
SFB3=2, JKREP=, JKZONE=, NRWGT=, JKTYP=full, JKFAC=0.5, REPWP=, SHRTCUT=N, SRVYSAM=Y, ODESC=N, GRAPH=N, REFFECT=N, 
LABEL=model, QPOINTS=, TEC=trureg, GCONV=1E-8, MAXFUNC=, MAXITER=, TYPE2=, TYPE3=, LDATA2=, CLIST2=, LDATA3=, CLIST3=, 
DIST=normal, START=y, STARTRW=y, LIBD=, LIBE=) / mindelimiter='|';
ods select none;
ods listing close;
options minoperator nonotes;



/*	DATN	:	The data set																		*/
/*	ROOTPV	:	Dependent Variable (PV is possible)													*/
/*	NPV		:	Number of PV's (else 1)																*/
/*	NOINT	:	No intercept in the fixed effects model (input Y)									*/
/*	XVAR1	:	Level 1 continues predictors														*/
/*	XVAR2	:	Level 2 continues predictors														*/
/*	XVAR3	:	Level 3 continues predictors														*/
/*	CVAR1	:	Level 1 categorical predictors														*/
/*	CVAR2	:	Level 2 categorical predictors														*/
/*	CVAR3	:	Level 3 categorical predictors														*/
/*	CCENT2	:	Variables that should be centered (group mean on level 2)							*/
/*	CCENT3	:	Variables that should be centered (group mean on level 3)							*/
/*	GCENT	:	Variables that should be centered (grand mean)										*/
/*	NORINT2	:	No random intercept on level 2 (input Y)											*/
/*	NORINT3	:	No random intercept on level 3 (input Y)											*/
/*	RSLOPE2	:	Predictors with random slopes on level 2											*/
/*	RSLOPE3	:	Predictors with random slopes on level 3											*/
/*	NEST1	:	Level 1 identifier																	*/
/*	NEST2	:	Level 2 identifier																	*/
/*	NEST3	:	Level 3 identifier																	*/
/*	L1WGT	:	Level 1 sampling weight																*/
/*	L2WGT	:	Level 2 sampling weight																*/
/*	L3WGT	:	Level 3 sampling weight																*/
/*	WGT		:	Combined sampling weight															*/
/*	SFW		:	Scaling the L1 weight (0: unscaled; 1: sample size; 2: effective sample size)		*/ 
/*	SFB2	:	Scaling the L2 weight (0: unscaled; 1: sum of L1WGT*L2WGT is sample size			*/ 
/*				(L1WGT is unscaled); 2: sum of L1WGT*L2WGT is sample size (L1WGT is scaled to		*/ 
/*				sample size); 3: sum of L1WGT*L2WGT is effective sample size (L1WGT is scaled 		*/ 
/*				to effective sample size))															*/ 
/*	SFB3	:	Scaling the L3 weight (0: unscaled; 1: sum of L1WGT*L3WGT is sample size			*/ 
/*				(L1WGT is unscaled); 2: sum of L1WGT*L3WGT is sample size (L1WGT is scaled to		*/ 
/*				sample size); 3: sum of L1WGT*L3WGT is effective sample size (L1WGT is scaled 		*/ 
/*				to effective sample size))															*/ 
/*	JKREP	:	Replicate assignment																*/
/*	JKZONE	:	Zone assignment																		*/
/*	NRWGT	:	Number of replications																*/
/*	JKTYP	:	FULL or HALF (HALF: Only one part of the sample is left out and the other part		*/ 
/*				is weighted twice; FULL: Both part of the sample are left out successively and		*/ 
/*				the other part is weighted twice)													*/ 
/*	JKFAC	:	Variance factor for jackknife (usually JKFAC=0.5 if JKTYPE=FULL and JKFAC=1			*/
/*				if JKTYPE=HALF)																		*/
/*	REPWP	:	Prefix of user supplied replication weights											*/
/*	SHRTCUT	:	Compute jackknife variance using all PV's (if only one PV should be used SHRTCUT=Y)	*/ 
/*	SRVYSAM	:	Should standard errors of the GLMM coefficients be based on replication weights		*/ 
/*	ODESC	:	Only descriptive statistics (input Y)												*/
/*	GRAPH	:	Should residual plots be printed (input Y)											*/
/*	REFFECT	:	Should the random-effect solution vector be saved (input Y)							*/
/*	LABEL	:	Label for the Model																	*/
/*	QPOINTS	:	Quadratur points																	*/
/*	TEC		:	Optimizer																			*/
/*	GCONV	:	Convergence criterion																*/
/*	MAXFUNC	:	Maximum number of function evaluation												*/
/*	MAXITER	:	Maximum number of iteration steps													*/
/*	TYPE2	:	Type of the random component matrix	on level 2										*/
/*	TYPE3	:	Type of the random component matrix	on level 3										*/
/*	LDATA2	:	If TYPE2=LIN(q) then LDATA provides the known matrices with the linear combinations	*/
/*	CLIST2	:	IF TYPE2=SP(EXP)(c-list) then CLIST provides the variable names for c-list			*/
/*	LDATA3	:	If TYPE3=LIN(q) then LDATA provides the known matrices with the linear combinations	*/
/*	CLIST3	:	IF TYPE3=SP(EXP)(c-list) then CLIST provides the variable names for c-list			*/
/*	DIST	:	Probability distribution of ROOTPV (default to normal) 								*/ 
/*	START	:	If PV's are used: Should start values for the random components be used from 		*/
/*				the first PV (y=yes; n=no)  														*/
/*	STARTRW	:	If survey SE are requested: Should start values for the random components be used	*/ 
/*				from the model of the full data set (in case of PV's of the first PV (y=yes; n=no)) */
/*	LIBD	:	Libref for the data (default is work)												*/
/*	LIBE	:	Libref for the results (default is work)											*/



/*** Define local Variables ***/
%local a b c ccentn checkjktyp checktyp checktype cmean cmtemp cntind coln coln1 coln2 constat continuous conv1 convst 
countc cvar cvarf cvarr cvar1n cvar2n cvar3n d d3 datacst 
dataset datav datavm datavm2 datndesc0 datndesc1 datndesc2 datndesc3 datndesc4 datnr datnrn datnrlm ddvnames distdv distds dpath dropra dsnid dsnid2 dvnames e e3 
epath f fitstat frepnames f3 g gcentn gmat gname gname1 gname2 gmean gred gs gv h idtemp iml1 iml2 iml3 iml5 iml7 iml8 
iml9 iml10 iml11 intercept interceptn inter2 inter3 j jkname jkname2 k k3 l l3 ldatan2 ldatan3 level listdv lm1 lw 
l1l3c0 l1l3c1 l1l3c2 l1l3c0s l1l3c1s l1l3c2s l1rwgtb l1rwgte l1wgts 
l2rwgtb l2rwgte l2wgts l3rwgtb l3rwgte l3wgts lmmmethod m m3 macroname maxfn maxiter maxf maxi method mod ncvar ncvar1 
ncvar2 ncvar3 neffect neffect1 neffect2 njk1 nobs nobsglm nointr npvr nrslope nrslope3 nrwgtc ntvar nvpef 
nxvar1 nxvar2 nxvar3 n6 n7 n8 o o3 obsweight obsweightc outods outodspv obsweight p p3 plotsm pnames pvfreq1 pvfreq2 pvfreq3 
pvfreq4 pvfreq5 
q q3 rc rdvnames reason renamehalf renaml1gt renaml2gt renaml3gt renamrwgt renamrwgtsvy res root rootpvn rslopen rslopen3 rweight2 rweight2c 
rweight3 rweight3c rwnames sub2 sub3 sucrep subject tdvnames tdvcnames tempcc tempcf tempcfr tempcm tempcmr tempdesc1 tempjkp tempjkpr tempjkv tempjkvr  u v vardef vpefn w 
wgtlm wgtlmc wgtn wgtnc wgtnrep xvar1n xvar2n xvar3n y yf zero;



/*** Define library ***/
%if %sysevalf(%superq(libd)=,boolean) = 0 %then %let dpath = &libd;
%else %let dpath = %sysfunc(getoption(work));
%if %sysevalf(%superq(libe)=,boolean) = 0 %then %let epath =&libe; 
%else %let epath = %sysfunc(getoption(work));
Libname d "&dpath";
Libname e "&epath";
%if &syserr. > 4 %then %put %str(ER)ROR: Something went wrong with the libname statements LIBD or LIBE. Analysis were interrupted.;
%if &syserr. > 4 %then %return;



/*** Check if necessary information is given ***/
%if %sysevalf(%superq(nest1)=,boolean) = 1 or %sysevalf(%superq(nest2)=,boolean) = 1 %then %put %str(ER)ROR: Necessary arguments NEST1 or NEST2 are not given. Analyzes were interrupted.;
%if %sysevalf(%superq(npv)=,boolean) = 1 %then %put %str(ER)ROR: Necessary argument NPV is not given. Analysis were interrupted.;
%if %sysevalf(%superq(rootpv)=,boolean) = 1 %then %put %str(ER)ROR: Necessary argument ROOTPV is not given. Analysis were interrupted.;
%if %sysevalf(%superq(nest1)=,boolean) = 1 or %sysevalf(%superq(nest2)=,boolean) or %sysevalf(%superq(npv)=,boolean) = 1 or
%sysevalf(%superq(rootpv)=,boolean) = 1 %then %return;
%if %sysevalf(%superq(nest3)=,boolean) = 1 %then %do;
	%if %sysevalf(%superq(xvar3)=,boolean) = 0 or %sysevalf(%superq(cvar3)=,boolean) = 0 or 
	%sysevalf(%superq(ccent3)=,boolean) = 0 or %sysevalf(%superq(rslope3)=,boolean) = 0 or
	%sysevalf(%superq(l3wgt)=,boolean) = 0 %then %do;
		%put %str(WA)RNING: Argument NEST3 is missing. Arguments XVAR3, CVAR3, CCENT3, RSLOP3, L3WGT and SFB3 are ignored.;
		%let XVAR3 = ;
		%let CVAR3 = ;
		%let CCENT3 = ;
		%let RSLOPE3 = ;
		%let L3WGT = ;
		%let SFB3 = ;
	%end;
%end;



/*** write dv into macro variable ***/
%if &npv = 1 %then %let dvnames=&rootpv;
%else %let dvnames=%namep(&rootpv,&npv);



/*** Check which kind of analysis is requested ***/
/** number of levels **/
%if %sysevalf(%superq(nest3)=,boolean) = 0 %then %do;
	%if %upcase(%substr(&norint2, 1, 1)) = Y and %sysevalf(%superq(rslope2)=,boolean) = 1 %then %let level = 1;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N or %sysevalf(%superq(rslope2)=,boolean) = 0 %then %do;
		%if %upcase(%substr(&norint3, 1, 1)) = Y and %sysevalf(%superq(rslope3)=,boolean) = 1 %then %let level = 2;
		%else %if %upcase(%substr(&norint3, 1, 1)) = N or %sysevalf(%superq(rslope3)=,boolean) = 0 %then %let level = 3;
	%end;
%end;
%else %if %sysevalf(%superq(nest3)=,boolean) = 1 %then %do;
	%if %upcase(%substr(&norint2, 1, 1)) = Y and %sysevalf(%superq(rslope2)=,boolean) = 1 %then %let level = 1;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N or %sysevalf(%superq(rslope2)=,boolean) = 0 %then %let level = 2;
%end;
%if &level ~= 3 %then %do;
	%let xvar3= ; 
	%let cvar3= ;
	%let ccent3= ;
	%let norint3= ;
	%let rslope3= ;
	%let nest3= ;
	%let l3wgt= ;
	%let sfb3= ;
	%let type3= ;
	%let ldata3= ;
	%let clist3= ;
%end; 


/** kind of model **/
%if &level~=3 %then %do;
	%if %upcase(%substr(&norint2, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 1 %then %let mod=1;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 0 %then %let mod=2;
	%else %if %upcase(%substr(&norint2, 1, 1)) = Y and %sysevalf(%superq(rslope2)=,boolean) = 1 %then %let mod=3;
	%else %if %upcase(%substr(&norint2, 1, 1)) = Y and %sysevalf(%superq(rslope2)=,boolean) = 0 %then %let mod=4;
	%if &mod=1 %then %put A two level random intercept model is fitted.;
	%else %if &mod=2 %then %put A two level random intercept and random slope model is fitted.;
	%else %if &mod=3 %then %put A one level fixed effect model is fitted.;
	%else %if &mod=4 %then %put A two level random slope model is fitted.;
%end;
%else %if &level=3 %then %do; 
	%if %upcase(%substr(&norint2, 1, 1)) = N and %upcase(%substr(&norint3, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 1 and  %sysevalf(%superq(rslope3)=,boolean) = 1 %then %let mod=5;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N and %upcase(%substr(&norint3, 1, 1)) = Y and %sysevalf(%superq(rslope2)=,boolean) = 1 and  %sysevalf(%superq(rslope3)=,boolean) = 0 %then %let mod=6;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N and %upcase(%substr(&norint3, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 1 and  %sysevalf(%superq(rslope3)=,boolean) = 0 %then %let mod=7;
	%else %if %upcase(%substr(&norint2, 1, 1)) = Y and %upcase(%substr(&norint3, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 0 and  %sysevalf(%superq(rslope3)=,boolean) = 1 %then %let mod=8;
	%else %if %upcase(%substr(&norint2, 1, 1)) = Y and %upcase(%substr(&norint3, 1, 1)) = Y and %sysevalf(%superq(rslope2)=,boolean) = 0 and  %sysevalf(%superq(rslope3)=,boolean) = 0 %then %let mod=9;
	%else %if %upcase(%substr(&norint2, 1, 1)) = Y and %upcase(%substr(&norint3, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 0 and  %sysevalf(%superq(rslope3)=,boolean) = 0 %then %let mod=10;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N and %upcase(%substr(&norint3, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 0 and  %sysevalf(%superq(rslope3)=,boolean) = 1 %then %let mod=11;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N and %upcase(%substr(&norint3, 1, 1)) = Y and %sysevalf(%superq(rslope2)=,boolean) = 0 and  %sysevalf(%superq(rslope3)=,boolean) = 0 %then %let mod=12;
	%else %if %upcase(%substr(&norint2, 1, 1)) = N and %upcase(%substr(&norint3, 1, 1)) = N and %sysevalf(%superq(rslope2)=,boolean) = 0 and  %sysevalf(%superq(rslope3)=,boolean) = 0 %then %let mod=13;
	%if &mod=5 %then %put A three level model with a random intercept on level two and level three is fitted.;
	%else %if &mod=6 %then %put A three level model with a random intercept on level two and random slopes on level three is fitted.;
	%else %if &mod=7 %then %put A three level model with a random intercept on level two, a random intercept and random slopes on level three is fitted.;
	%else %if &mod=8 %then %put A three level model with random slopes on level two and a random intercept on level three is fitted.;
	%else %if &mod=9 %then %put A three level model with random slopes on level two and level three is fitted.;
	%else %if &mod=10 %then %put A three level model with random slopes on level two, a random intercept and random slopes on level three is fitted.;
	%else %if &mod=11 %then %put A three level model with a random intercept and random slopes on level two and a random intercept on level three is fitted.;
	%else %if &mod=12 %then %put A three level model with a random intercept and random slopes on level two and random slopes on level three is fitted.;
	%else %if &mod=13 %then %put A three level model with a random intercept and random slopes on level two and level three is fitted.;
%end;


/** check which weights are given **/
%if %sysevalf(%superq(l1wgt)=,boolean) = 0 and %sysevalf(%superq(l2wgt)=,boolean) = 0 and %sysevalf(%superq(l3wgt)=,boolean) = 0 %then %let lw=1;
%else %if %sysevalf(%superq(l1wgt)=,boolean) = 0 and %sysevalf(%superq(l2wgt)=,boolean) = 0 and %sysevalf(%superq(l3wgt)=,boolean) = 1 %then %let lw=2;
%else %let lw=3;
%if %sysevalf(%superq(wgt)=,boolean) = 0 %then %let w=1;
%if %sysevalf(%superq(wgt)=,boolean) = 1 %then %let w=2;


/** Check which kind of replication weights are given **/
%if %sysevalf(%superq(repwp)=,boolean) = 0 %then %let k=1;
%else %if %sysevalf(%superq(repwp)=,boolean) = 1 and %sysevalf(%superq(jkzone)=,boolean) = 0 and %sysevalf(%superq(jkrep)=,boolean) = 0 %then %let k=2;
%else %let k=3;
%if %sysevalf(%superq(nrwgt)=,boolean) = 1 %then %do;
	%put %str(WA)RNING: Argument NRWGT is missing. NRWGT is set to the default value of 1.;
	%let nrwgt = 1;
%end;



/*** Define the kind of analysis that is performed ***/
%if &level=1 %then %do;
	%let l1wgt=;
	%let l2wgt=;
	%if &k=1 %then %do;
		%let rwnames=%namep(&repwp,&nrwgt);
		%let jkzone=;
		%let jkrep=;
		%if &w = 1 %then %do;
			%put A weighted GLMM analysis with REPWP is performed.;
			%let dataset=3;
		%end;
		%else %do;
			%put %str(WA)RNING: Argument WGT is missing. An unweighted analysis with REPWP is performed.; 
			%let dataset=4;
		%end;
	%end;
	%else %if &k=2 %then %do;
		%if &w = 1 %then %do;
			%put A weighted GLMM analysis with JKREP is performed.;
			%let dataset=7;
		%end; 
		%else %do;
			%put %str(WA)RNING: Argument WGT is missing. An unweighted analysis with JKREP is performed.; 
			%let dataset=8;
		%end;
	%end;
	%else %if &k=3 %then %do;
		%let jkzone=;
		%let jkrep=;
		%put %str(WA)RNING: Replication weights are missing. Estimates of variances and standard errors are based on model based sampling procedures.;
		%if &w = 1 %then %do;
			%put A weighted GLMM analysis is performed.;
			%let dataset=11;	
		%end; 
		%else %do;
			%put %str(WA)RNING: Argument WGT is missing. An unweighted analysis is performed.; 
			%let jkzone=;
			%let jkrep=;
			%let dataset=12;	
		%end;
	%end;
%end;
%else %if &level=2 %then %do;
	%if &k=1 %then %do;
		%let jkzone=;
		%let jkrep=;
			%if &lw=2 and &w=1 %then %do;
				%put A weighted GLMM analysis with L1WGT, L2WGT, WGT and REPWP is performed.;
				%let rwnames=%namep(&repwp,%sysevalf(3*&nrwgt));
				%let dataset=1;
			%end;
			%else %if &lW = 2 and &w = 2 %then %do;
				%put A weighted GLMM analysis with L1WGT, L2WGT and REPWP is performed.;
				%put %str(WA)RNING: Argument WGT is missing. Descriptive statistics are based on unweighted analysis.;
				%let rwnames=%namep(&repwp,%sysevalf(2*&nrwgt));
				%let dataset=2;
			%end;
			%else %if &lw = 3 and &w = 1 %then %do;
				%put A weighted GLMM analysis with WGT and REPWP is performed.;
				%put %str(WA)RNING: Argument L1WGT and/or L2WGT are missing. GLMM analysis are based on WGT and maybe biased.;
				%let rwnames=%namep(&repwp,&nrwgt);
				%let l1wgt=;
				%let l2wgt=;
				%let dataset=3;
			%end;
			%else %if &lw = 3 and &w = 2 %then %do;
				%put %str(WA)RNING: Argument WGT, L1WGT and/or L2WGT are missing. An unweighted analysis with REPWP is performed.; 
				%let rwnames=%namep(&repwp,&nrwgt);
				%let l1wgt=;
				%let l2wgt=;
				%let dataset=4;
			%end;
	%end;
	%else %if &k=2 %then %do;
		%if &lw=2 and &w=1 %then %do;
			%put A weighted GLMM analysis with L1WGT, L2WGT, WGT and JKREP is performed.;
			%let dataset=5;
		%end;
		%else %if &lw = 2 and &w = 2 %then %do;
			%put A weighted GLMM analysis with L1WGT, L2WGT and JKREP is performed.;
			%put %str(WA)RNING: Argument WGT is missing. Descriptive statistics are based on unweighted analysis.;
			%let dataset=6;
		%end;
		%else %if &lw = 3 and &w = 1 %then %do;
			%put A weighted GLMM analysis with WGT and JKREP is performed.;
			%put %str(WA)RNING: Argument L1WGT and/or L2WGT is missing. GLMM analysis are based on WGT and maybe biased.;
			%let l1wgt=;
			%let l2wgt=;
			%let dataset=7;
		%end;
		%else %if &lw = 3 and &w = 2 %then %do;
			%put %str(WA)RNING: Argument WGT, L1WGT and/or L2WGT are missing. An unweighted analysis with JKREP is performed.;
			%let l1wgt=;
			%let l2wgt=;
			%let dataset=8;
		%end;
	%end;
	%else %if &k=3 %then %do;
		%let jkzone=;
		%let jkrep=;
		%put %str(WA)RNING: Replication weights are missing. Estimates of variances and standard errors are based on model based sampling procedures.;
		%if &lw=2 and &w=1 %then %do;
			%put A weighted GLMM analysis with L1WGT, L2WGT and WGT is performed.;
			%let dataset=9;
		%end;
		%else %if &lw = 2 and &w = 2 %then %do;
			%put A weighted GLMM analysis with L1WGT and L2WGT is performed.;
			%put %str(WA)RNING: Argument WGT is missing. Descriptive statistics are based on unweighted analysis.;
			%let dataset=10;
		%end;
		%else %if &lw = 3 and &w = 1 %then %do;
			%put A weighted GLMM analysis with WGT is performed.;
			%put %str(WA)RNING: Argument L1WGT and/or L2WGT is missing. GLMM analysis are based on WGT and maybe biased.;
			%let l1wgt=;
			%let l2wgt=;
			%let dataset=11;
		%end;
		%else %if &lw = 3 and &w = 2 %then %do;
			%put %str(WA)RNING: WGT, L1WGT and/or L2WGT are missing. An unweighted analysis is performed.; 
			%let l1wgt=;
			%let l2wgt=;
			%let dataset=12;
		%end;
	%end;
%end;
%else %if &level=3 %then %do;
	%if &k=1 %then %do;
		%let jkzone=;
		%let jkrep=;
		%if &lw=1 and &w=1 %then %do;
				%put A weighted GLMM analysis with L1WGT, L2WGT, L3WGT, WGT and REPWP is performed.;
				%let rwnames=%namep(&repwp,%sysevalf(4*&nrwgt));
				%let dataset=13;
		%end;
		%else %if &lW = 1 and &w = 2 %then %do;
				%put A weighted GLMM analysis with L1WGT, L2WGT, L3WGT and REPWP is performed.;
				%put %str(WA)RNING: Argument WGT is missing. Descriptive statistics are based on unweighted analysis.;
				%let rwnames=%namep(&repwp,%sysevalf(3*&nrwgt));
				%let dataset=14;
		%end;
		%else %if &lw = 3 and &w = 1 %then %do;
				%put A weighted GLMM analysis with WGT and REPWP is performed.;
				%put %str(WA)RNING: Argument L1WGT, L2WGT and/or L3WGT are missing. GLMM analysis are based on WGT and maybe biased.;
				%let rwnames=%namep(&repwp,&nrwgt);
				%let l1wgt=;
				%let l2wgt=;
				%let l3wgt=;
				%let dataset=3;
		%end;
		%else %if &lw = 3 and &w = 2 %then %do;
				%put %str(WA)RNING: Argument WGT, L1WGT, L2WGT and/or L3WGT are missing. An unweighted analysis with REPWP is performed.; 
				%let rwnames=%namep(&repwp,&nrwgt);
				%let l1wgt=;
				%let l2wgt=;
				%let l3wgt=;
				%let dataset=4;
		%end;
	%end;
	%else %if &k=2 %then %do;
		%if &lw=1 and &w=1 %then %do;
			%put A weighted GLMM analysis with L1WGT, L2WGT, L3WGT, WGT and JKREP is performed.;
			%let dataset=15;
		%end;
		%else %if &lw = 1 and &w = 2 %then %do;
			%put A weighted GLMM analysis with L1WGT, L2WGT, L3WGT and JKREP is performed.;
			%put %str(WA)RNING: Argument WGT is missing. Descriptive statistics are based on unweighted analysis.;
			%let dataset=16;
		%end;
		%else %if &lw = 3 and &w = 1 %then %do;
			%put A weighted GLMM analysis with WGT and JKREP is performed.;
			%put %str(WA)RNING: Argument L1WGT, L2WGT and/or L3WGT is missing. GLMM analysis are based on WGT and maybe biased.;
			%let l1wgt=;
			%let l2wgt=;
			%let l3wgt=;
			%let dataset=7;
		%end;
		%else %if &lw = 3 and &w = 2 %then %do;
			%put %str(WA)RNING: Argument WGT, L1WGT, L2WGT and/or L3WGT are missing. An unweighted analysis with JKREP is performed.;
			%let l1wgt=;
			%let l2wgt=;
			%let l3wgt=;
			%let dataset=8;
		%end;
	%end;
	%else %if &k=3 %then %do;
		%let jkzone=;
		%let jkrep=;
		%put %str(WA)RNING: Replication weights are missing. Estimates of variances and standard errors are based on model based sampling procedures.;
		%if &lw=1 and &w=1 %then %do;
			%put A weighted GLMM analysis with L1WGT, L2WGT, L3WGT and WGT is performed.;
			%let dataset=17;
		%end;
		%else %if &lw = 1 and &w = 2 %then %do;
			%put A weighted GLMM analysis with L1WGT, L2WGT and L3WGT is performed.;
			%put %str(WA)RNING: Argument WGT is missing. Descriptive statistics are based on unweighted analysis.;
			%let dataset=18;
		%end;
		%else %if &lw = 3 and &w = 1 %then %do;
			%put A weighted GLMM analysis with WGT is performed.;
			%put %str(WA)RNING: Argument L1WGT, L2WGT and/or L3WGT is missing. GLMM analysis are based on WGT and maybe biased.;
			%let l1wgt=;
			%let l2wgt=;
			%let l3wgt=;
			%let dataset=11;
		%end;
		%else %if &lw = 3 and &w = 2 %then %do;
			%put %str(WA)RNING: WGT, L1WGT, L2WGT and/or L3WGT are missing. An unweighted analysis is performed.; 
			%let l1wgt=;
			%let l2wgt=;
			%let l3wgt=;
			%let dataset=12;
		%end;
	%end;
%end;



/*** distributional properties of the dv ***/
%let distdv = &dist;
%let distdvs = %substr(&dist,1,2);
%if %lowcase(&distdv) in beta|exponential|expo|gamma|gam|gaussian|g|normal|n|invgauss|igaussian|ig|lognormal|logn|tcentral|tdist|t %then %let continuous=1;
%else %if %lowcase(&distdvs) = by %then %let continuous=2;	
%else %let continuous=0;



/*** Construct the data set ***/
%let datnr = %NewDatasetName(temp);
data &datnr (keep=&nest1 &nest2 &nest3 &dvnames &rwnames &jkrep &jkzone &cvar1 &cvar2 &cvar3 &xvar1 &xvar2 &xvar3 &gcent &l1wgt &l2wgt &l3wgt &wgt);
set d.&datn;
run;
%if &syserr. > 3 %then %put %str(ER)ROR: Input reading of the data fails. Analysis were interrupted.;
%if &syserr. > 3 %then %return;



/*** Rename REPWP variables if they exists ***/
%if &dataset. in 1|2|13|14 %then %do;
	%let l1rwgtb = %sysevalf(&nrwgt+1);
	%let l1rwgte = %sysevalf(2*&nrwgt);
	%let l2rwgtb = %sysevalf(&l1rwgte+1);
	%let l2rwgte = %sysevalf(3*&nrwgt);
 	%let renamrwgt = &repwp.1-&repwp&nrwgt=rwgt1-rwgt&nrwgt;
	%let renaml1gt = &repwp&l1rwgtb-&repwp&l1rwgte=l1rwgt1-l1rwgt&nrwgt;
	%let renaml2gt = &repwp&l2rwgtb-&repwp&l2rwgte=l2rwgt1-l2rwgt&nrwgt;
%end;
%if &dataset. in 13|14 %then %do;
	%let l3rwgtb = %sysevalf(&l2rwgte+1);
	%let l3rwgte = %sysevalf(4*&nrwgt);
	%let renaml3gt = &repwp&l3rwgtb-&repwp&l3rwgte=l3rwgt1-l3rwgt&nrwgt;
%end;
%if &dataset. in 1|2|3|4|13|14 %then %do;
	%let renamrwgt = &repwp.1-&repwp&nrwgt=rwgt1-rwgt&nrwgt;
	data &datnr (rename=(&renamrwgt &renaml1gt &renaml2gt &renaml3gt));
	set &datnr;
	run;
%end;



/*** If necessary produce replication weights now (Part I) ***/
%if &k=2 %then %do;
	%if %sysevalf(%superq(jktyp)=,boolean) = 1 %then %do;
		%put %str(WA)RNING: Argument JKTYP is missing. JKTYP is set to the default value FULL.;
		%let jktyp=full;	
	%end;
	%let checkjktyp = %lowcase(&jktyp);
	%if &checkjktyp ~= full and &checkjktyp ~= half %then %do;
		%put %str(WA)RNING: Argument JKTYP is not correctly specified (either typ "full" or "half"). JKTYP is set to the default value FULL.;
		%let jktyp=full;	
	%end;
%end; 
%if &dataset. in 5|6|7|8|15|16 %then %do;
	proc sql;
	select count(distinct(&jkzone)) into :nrwgtsql from &datnr;
	quit;
	%if &dataset. in 5|7|15 %then %do;
		%let wgtnrep = weight &wgt;
	%end;		
	%if %lowcase(&jktyp)=half %then %do;
		%let nrwgt2 = %sysevalf(2*&nrwgtsql);
		%let nrwgt = &nrwgtsql;
		proc surveymeans data=&datnr varmethod=jk(outweights=&datnr outjkcoefs=e.test_coef) nobs;
		strata &jkzone;
		cluster &jkrep;
		&wgtnrep; 
		run;		
		%let i = 0;
		%do rwn1 = 1 %to &nrwgt2;
			%let j = %sysfunc(inputn(&rwn1,8.));
			%if %sysfunc(mod(&j,2))=0 %then %do;
				%let dropr = RepWt_&rwn1.;
				%let renamehalfr = ; 
			%end;
			%else %do;
				%let i = %sysevalf(&i+1);
				%let dropr = ;
				%let renamehalfr = RepWt_&rwn1=rwgt&i; 
			%end;
			%let dropra = &dropra &dropr;
			%let renamehalf = &renamehalf &renamehalfr;
		%end;
		data &datnr(rename=(&renamehalf));
		set &datnr (drop=&dropra);		
		run;
	%end;
	%else %if %lowcase(&jktyp)=full %then %do;
		%let nrwgt = %sysevalf(2*&nrwgtsql);
		proc surveymeans data=&datnr varmethod=jk(outweights=&datnr) nobs;
		strata &jkzone;
		cluster &jkrep;
		&wgtnrep; 
		run;
		%let renamrwgtsvy = RepWt_1-RepWt_&nrwgt=rwgt1-rwgt&nrwgt;
		data &datnr (rename=(&renamrwgtsvy));
		set &datnr;
		run;
	%end;	
%end;



/*** Create descriptive statistics ***/
/** means **/
%let tdvnames = ;
%let rdvnames = ;
%let cntind = ;
%let ddvnames = ;
%if &continuous=1 %then %do;
	%if &npv=1 %then %do;
		%let tdvnames = &rootpv;
		%let rdvnames = &rootpv;
		%let ddvnames = &rootpv;
		%let cntind = 1;
	%end;
	%else %if &npv>1 %then %do;
		%let ddvnames = &rootpv.1;
		%if %lowcase(&shrtcut)=y %then %do;
			%let tdvnames = &rootpv.1;
			%let rdvnames = &rootpv.1;
			%let cntind = 1;
		%end;
		%else %if %lowcase(&shrtcut)=n %then %do;
			%let tdvnames = &dvnames;
			%let cntind = 2;
		%end;
	%end;
%end;
%if %sysevalf(%superq(xvar1)=,boolean) = 0 or %sysevalf(%superq(xvar2)=,boolean) = 0 or %sysevalf(%superq(xvar3)=,boolean) = 0 or %sysevalf(%superq(tdvnames)=,boolean) = 0 %then %do;
	%if &dataset. in 1|2|3|4|5|6|7|8|13|14|15|16 %then %do;
		%let njk1 = %sysevalf(&nrwgt+1);
		%if &dataset. in 1|3|5|7|13|15 %then %do;
			%let vardef = vardef=wdf;
			%let wgtn = weight &wgt;
		%end;		
		%let datndesc0 = %NewDatasetName(temp);		
		proc means data=&datnr &vardef;
		var &dvnames &xvar1 &xvar2 &xvar3;
		&wgtn ; 
		output out=&datndesc0.0 mean=&dvnames &xvar1 &xvar2 &xvar3;
		run;
		%do i=1 %to &nrwgt;
			proc means data=&datnr vardef=wdf;
			var &dvnames &xvar1 &xvar2 &xvar3;
			weight rwgt&i.;
			output out=&datndesc0.&i. mean=&dvnames &xvar1 &xvar2 &xvar3;
			run;
		%end;
		data &datndesc0;
		set &datndesc0.0 %namep(&datndesc0,&nrwgt);
		run;
	%end;
	proc iml;
	%if %sysevalf(%superq(xvar1)=,boolean) = 0 or %sysevalf(%superq(xvar2)=,boolean) = 0 or %sysevalf(%superq(xvar3)=,boolean) = 0 or &cntind = 1 %then %do;	
		use &datnr;
		read all var {&rdvnames &xvar1 &xvar2 &xvar3} into y1;
		n = countn(y1,"col");
		mis = countmiss(y1,"col");
		totn = n+mis;
		totnt = t(totn);
		nt = t(n);
		mist = t(mis);
		min = y1[><,];
		mint = t(min);
		max = y1[<>,];
		maxt = t(max);
		Out=totnt||mist||nt||mint||maxt;
	%end;
	%if &cntind=2 %then %do;
		use &datnr;
		read all var {&tdvnames} into y2;
		np = countn(y2,"col");
		misp = countmiss(y2,"col");
		totnp = np+misp;
		minp = y2[><,];
		maxp = y2[<>,];
		totnpt = totnp[,:];
		mispt = misp[,:];
		npt = np[,:];
		minpt = minp[,:];
		maxpt = maxp[,:];
		Out1 = totnpt||mispt||npt||minpt||maxpt;
		Out=Out1//Out;
	%end;
	use &datnr;
	read all var {&dvnames &xvar1 &xvar2 &xvar3} into y3;			
	%if &dataset. in 2|4|6|8|10|12|14|16|18 %then %do;	
		wgt = j(nrow(y3), 1, 1);
	%end;
	%else %if &dataset. in 1|3|5|7|9|11|13|15|17 %then %do;
		use &datnr;
		read all var {&wgt} into y0;
		wgt = y0;
	%end;
	dvw = y3#wgt;
	wcs = dvw[+,];
	wgtmis = wgt#(y3 ^= .);
	sumwgt = wgtmis[+,];
	m = wcs/sumwgt;
	mt = t(m);
	dif = y3-m;
	dif2dif = dif#dif;
	difw = dif2dif#wgt;
	difs = difw[+,];
	var = difs/(sumwgt-1);
	vart = t(var);
	std = sqrt(var);
	stdt = t(std);
	%if &dataset. in 9|10|11|12|17|18 %then %do;
		n = countn(y3,"col");
		stder = std/sqrt(n);
	%end;
	%else %if &dataset. in 1|2|3|4|5|6|7|8|13|14|15|16 %then %do;
		use &datndesc0;
		read all var{&dvnames &xvar1 &xvar2 &xvar3} into p1;
		d=p1;
		d1=p1[1,];
		d2=p1[2:&njk1,];
		do i=1 to &nrwgt;
			pdif=(d1-d2[i,])##2;
			pdifall=pdifall//pdif;
		end;
		jkvar=pdifall[+,];
		totvar=&jkfac*jkvar;
		stder=totvar##.5;
	%end;
	stdet = t(stder);
	Out2=mt||stdet||vart||stdt;
	%if &npv>1 and &continuous=1 %then %do;
		outs = out2[1:&npv,1];
		gm = outs[:,];
		impv = var(outs);
		impvar=%sysevalf(%sysevalf(&npv+1)/&npv)*impv;
		vars = var[,1:&npv];
		mvar = (vars[,+])/(&npv);
		mvars = mvar + impvar;
		std = mvars##.5;
		%if &dataset. in 9|10|11|12|17|18 %then %do;
			stder = std/sqrt(n[1,1]);
		%end;
		%else %if &dataset. in 1|2|3|4|5|6|7|8|13|14|15|16 %then %do;
			totvars = totvar[,1:&npv];
			%if %lowcase(&shrtcut)=y %then %do;
				totvar=totvars[,1]+impvar;
				stder=totvar##.5;
			%end;
			%else %if %lowcase(&shrtcut)=n %then %do;
				totvar=(totvars[,+])/(&npv);
				totvar1 = totvar+impvar;
				stder=totvar1##.5;
			%end;
		%end;
		Out3=gm||stder||mvars||std;
		%if %sysevalf(%superq(xvar1)=,boolean) = 0 or %sysevalf(%superq(xvar2)=,boolean) = 0 or %sysevalf(%superq(xvar3)=,boolean) = 0 %then %do;
			Out4=Out3//Out2[%sysevalf(&npv+1):nrow(out2),];
		%end;
		%else %do;
			Out4=Out3;
		%end;
	%end;
	%else %do;
		Out4=Out2;
	%end;	
	Outf=Out||Out4;
	%let tempdesc1 = %NewDatasetName(temp);
	create &tempdesc1 from Outf[colname={"Total" "Missing" "N" "Minimum" "Maximum" "Mean" "StdErr" "Variance" "StdDev"}];
	append from Outf;
	close &tempdesc1;
	quit;	
	%if &continuous=1 %then %do;
		%let datndesc1 = %NewDatasetName(temp);
		data &datndesc1 (keep=&ddvnames &xvar1 &xvar2 &xvar3);
		retain &ddvnames &xvar1 &xvar2 &xvar3;
		set &datnr;
		run;
		%if &npv>1 %then %do;
			%let root = %substr(&rootpv,1,6);
			data &datndesc1 (rename=(&rootpv.1=&root));
			set &datndesc1;
			run;
			data &datndesc1;
			retain &root &xvar1 &xvar2 &xvar3;
			set &datndesc1;
			run;
		%end;
		%let datndesc2 = %NewDatasetName(temp);
		proc contents data = &datndesc1 noprint out = &datndesc2 (keep = name varnum);
		run;
		proc sort data=&datndesc2 out=&datndesc2(keep = name);
		by varnum;
		run;
	%end;
	%else %do;
		%let datndesc1 = %NewDatasetName(temp);
		data &datndesc1 (keep=&xvar1 &xvar2 &xvar3);
		retain &xvar1 &xvar2 &xvar3;
		set &datnr;
		run;
		data &datndesc1;
		retain &xvar1 &xvar2 &xvar3;
		set &datndesc1;
		run;
		%let datndesc2 = %NewDatasetName(temp);
		proc contents data = &datndesc1 noprint out = &datndesc2 (keep = name varnum);
		run;
		proc sort data=&datndesc2 out=&datndesc2(keep = name);
		by varnum;
		run;
	%end;
	data svyhlm_means_print (rename=(name=Variable));
	merge &datndesc2 &tempdesc1;
	run;
%end;


/** frequencys **/
%let tdvcnames = ;
%if &continuous=0 %then %do;
	%if &npv=1 %then %do;
		%let tdvcnames = &rootpv;
		%let ntvar = %sysfunc(countw(&tdvcnames));
	%end;
	%else %if &npv>1 %then %do;
		%let tdvcnames = &dvnames;
		%let ntvar = %sysfunc(countw(&tdvcnames));
	%end;
%end;
%if %sysevalf(%superq(cvar1)=,boolean) = 0 or %sysevalf(%superq(cvar2)=,boolean) = 0 or %sysevalf(%superq(cvar3)=,boolean) = 0 or %sysevalf(%superq(tdvcnames)=,boolean) = 0 %then %do;
	%let cvars = &cvar1 &cvar2 &cvar3;
	%let ncvars = %sysfunc(countw(&cvars));
	%let ncvarst = %sysevalf(&ncvars+1);
	%let cvar = &cvars &tdvcnames;
	%let ncvar = %sysfunc(countw(&cvar));
	%let ncall = &ncvar;
	%let ncvart = %sysevalf(&ncvar+1);
	%let cvarp = &cvar;
	%let dsnid = %sysfunc(open(&datnr));
	%let nobs=%sysfunc(attrn(&dsnid,nlobs));
	%if &dsnid > 0 %then %let rc=%sysfunc(close(&dsnid));
	%if &dataset. in 1|3|5|7|9|11|13|15|17 %then %let wgtn = weight &wgt;
	%let datndesc3 = %NewDatasetName(temp);
	%do t=1 %to &ncvar;
			%let cvarf = %scan(&cvar,&t);
			proc freq data=&datnr;
			tables &cvarf / outcum out=&datndesc3._&t.;
			&wgtn ;
			run;
	%end;
	%if &ntvar > 1 %then %do;
		%let ncall = &ncvart;
		%let cvarp = &cvarp &rootpv;
		%let pvs = 1;
		%do pv=&ncvarst %to &ncvar;
			data pvfreq_&pvs. (rename=(&rootpv&pvs.=&rootpv));
			set &datndesc3._&pv.;
			_Imputation_=&pvs.;
			stderr = round(sqrt(percent*(100-percent)/&nobs),0.001);
			run;
			%let pvs = %sysevalf(&pvs+1);
		%end;
		data pvfreq_c;
		set %namep(pvfreq_,&npv);;
		run;
		proc sort data=pvfreq_c;
		by &rootpv;
		run;	
		%let pvfreq1 = %NewDatasetName(temp);
		proc means data=pvfreq_c;
		var count percent cum_freq cum_Pct;
		by &rootpv;
		output out=&pvfreq1 mean=count percent cum_freq cum_Pct;
		run;
		%let pvfreq2 = %NewDatasetName(temp);
		data &pvfreq2 (drop=_type_ _freq_);
		set &pvfreq1;
		run;
		%let pvfreq3 = %NewDatasetName(temp);
		proc mianalyze data=pvfreq_c;
		modeleffects percent;
		stderr stderr;
		by &rootpv;	
		ods output parameterestimates=&pvfreq3 varianceinfo=svyhlm_pcfreq_vi_m;
		run;
		%let pvfreq4 = %NewDatasetName(temp);
		data &pvfreq4;
		set &pvfreq3 (keep=&rootpv stderr);
		run;
		data &datndesc3._&ncvart.;
		merge &pvfreq2 &pvfreq4;
		by &rootpv;
		run;
	%end;
	%do u=1 %to &ncall;	
		%let cvarf = %scan(&cvarp,&u);
		data svyhlm_&cvarf._print (keep=&cvarf Frequency Percent CumFrequency CumPercent StdErr);
		set &datndesc3._&u.;
		if percent=. then delete;
		countn = round(count,1);
		percentn = round(percent,0.01);
		cum_freqn = round(cum_freq,1);
		cum_pctn = round(cum_pct,0.01);
		if stderr=. then do;
			stderr1 = round(sqrt(percent*(100-percent)/&nobs),0.001);
			stderr = round(stderr1,0.01);
			drop stderr1;
		end;	
		drop count percent cum_freq cum_Pct;
		rename countn=Frequency percentn=Percent cum_freqn=CumFrequency cum_pctn=CumPercent;
		run;
		data svyhlm_&cvarf._print;
		retain &cvarf Frequency CumFrequency Percent StdErr CumPercent;
		set svyhlm_&cvarf._print;
		run;
	%end;
	%if &dataset. in 1|2|3|4|5|6|7|8|13|14|15|16 %then %do;
		%let pnames=%namep(percent,&nrwgt);
		%let jkfreq = %NewDatasetName(temp);
		%do t=1 %to &ncvar;
			%let cvarf = %scan(&cvar,&t);
			%do k=1 %to &nrwgt;
				proc freq data=&datnr;
				tables &cvarf / outcum out=&jkfreq.&k.;
				weight rwgt&k.;
				run;
				data &jkfreq.&k. (keep=&cvarf Percent&k.);
				set &jkfreq.&k.;
				if percent=. then delete;
				rename percent=percent&k.;				
				run;
				data &jkfreq.&k.;
				retain &cvarf percent&k.;
				set &jkfreq.&k.;
				run;
			%end;
			%let jkfreqd = %NewDatasetName(temp);
			data &jkfreqd;
			merge %namep(&jkfreq,&nrwgt);
			run;
			%let jkname = "jkvar&t.";
			proc iml;
			use svyhlm_&cvarf._print;
			read all var {Percent} into y1;
			use &jkfreqd;
			read all var {&pnames} into y2;
			%if &ntvar > 1 and &t > &ncvars %then %do;
				use svyhlm_pcfreq_vi_m;
				read all var {BetVar} into y3;
				impvar=%sysevalf(%sysevalf(&npv+1)/&npv)*y3;
			%end;
			do i=1 to &nrwgt;
				pdif=(y1-y2[,i])##2;
				pdifall=pdifall||pdif;
			end;
			jkvar=pdifall[,+];
			jkvarf=&jkfac*jkvar;
			%if &ntvar > 1 and &t > &ncvars %then %do;
				totvar=jkvarf+impvar;
			%end;
			%else %do;
				totvar=jkvarf;
			%end;
			stderf=totvar##.5;
			stder1=round(stderf, 0.001);
			stder=round(stder1,0.01);
			create _tempdesc&cvarf from stder[colname={"StdErr"}];
			append from stder;
			close _tempdesc&cvarf;
			create _tempjkv&t. from jkvarf[colname={&jkname}];
			append from jkvarf;
			close _tempjkv&t.;
			quit;
			data svyhlm_&cvarf._print  (drop=stderr);
			set svyhlm_&cvarf._print;
			run;
			data svyhlm_&cvarf._print;
			retain &cvarf Frequency CumFrequency Percent StdErr CumPercent;
			merge svyhlm_&cvarf._print _tempdesc&cvarf;
			run;
		%end;
		%if &ntvar > 1 %then %do;
			%if %lowcase(&shrtcut)=y %then %do;
				%let stderrot = %NewDatasetName(temp);
				data &stderrot;
				set svyhlm_&rootpv.1_print (keep=stderr);
				run;
				data svyhlm_&rootpv._print  (drop=stderr);
				set svyhlm_&rootpv._print;
				run;
				data svyhlm_&rootpv._print;
				retain &rootpv Frequency CumFrequency Percent StdErr CumPercent;
				merge svyhlm_&rootpv._print &stderrot;
				run;				
			%end;
			%else %if %lowcase(&shrtcut)=n %then %do;
				%let pvfreq4 = %NewDatasetName(temp);
				data &pvfreq4;
				merge _tempjkv&ncvarst.-_tempjkv&ncvar.;
				run;
				proc iml;
				use &pvfreq4;
				read all var _ALL_ into y1;
				use svyhlm_pcfreq_vi_m;
				read all var {BetVar} into y2;
				impvar=%sysevalf(%sysevalf(&npv+1)/&npv)*y2;
				jkvarm=y1[,:];
				totvar=jkvarm+impvar;
				stderf=totvar##.5;
				stder1=round(stderf,0.001);
				stder=round(stder1,0.01);
				create _temppvfreq5 from stder[colname={"StdErr"}];
				append from stder;
				close _temppvfreq5;
				quit;
				data svyhlm_&rootpv._print  (drop=stderr);
				set svyhlm_&rootpv._print;
				run;
				data svyhlm_&rootpv._print;
				retain &rootpv Frequency CumFrequency Percent StdErr CumPercent;
				merge svyhlm_&rootpv._print _temppvfreq5;
				run;
			%end;
		%end;
	%end;
%end;



/*** Save and print results ***/
%if %sysevalf(%superq(xvar1)=,boolean) = 0 or %sysevalf(%superq(xvar2)=,boolean) = 0 or %sysevalf(%superq(xvar3)=,boolean) = 0 or %sysevalf(%superq(rootpvn)=,boolean) = 0 or %sysevalf(%superq(tdvnames)=,boolean) = 0 %then %do;
	data e.svyhlm_&label._m;
	set svyhlm_means_print;
	run;
%end;
%if %sysevalf(%superq(cvar1)=,boolean) = 0 or %sysevalf(%superq(cvar2)=,boolean) = 0 or %sysevalf(%superq(cvar3)=,boolean) = 0 or %sysevalf(%superq(tdvcnames)=,boolean) = 0 %then %do;
	%do l=1 %to &ncall;
		%let cvarf = %scan(&cvarp,&l);
		data e.svyhlm_&label._f_&cvarf;
		set svyhlm_&cvarf._print;
		run;
	%end;
%end;
ods listing;
ods select all;
filename results "&libe.svyhlm_&label._des.xml";
ods TAGSETS.EXCELXP
file=results
style=htmlblue
options (FitToPage='yes' absolute_column_width='12' Pages_FitWidth='1' sheet_interval='none' sheet_name='Descriptive statistics'   embedded_titles='yes' embedded_footnotes='yes' embed_titles_once = 'yes');
%if %sysevalf(%superq(xvar1)=,boolean) = 0 or %sysevalf(%superq(xvar2)=,boolean) = 0 or %sysevalf(%superq(xvar3)=,boolean) = 0 or %sysevalf(%superq(rootpvn)=,boolean) = 0 or %sysevalf(%superq(tdvnames)=,boolean) = 0 %then %do;
	proc print data=svyhlm_means_print noobs;
	title1 "Means and standard errors";
	run;
%end;
%if %sysevalf(%superq(cvar1)=,boolean) = 0 or %sysevalf(%superq(cvar2)=,boolean) = 0 or %sysevalf(%superq(cvar3)=,boolean) = 0 or %sysevalf(%superq(tdvcnames)=,boolean) = 0 %then %do;
	%do l=1 %to &ncall;
		%let cvarf = %scan(&cvarp,&l);
		proc print data=svyhlm_&cvarf._print noobs;
		title1 "Frequencies, percentages and standard errors for variable &cvarf";
		run;
	%end;
%end;
%if %lowcase(%substr(&odesc, 1, 1)) = y %then %do;
	ods tagsets.excelxp close;
	%goto exit;
%end;
ods tagsets.excelxp close;
ods select none;
ods listing close;	



/*** perform multilevel analysis ***/
/** use only available cases (listwise deletion) **/
%let datnrn = %NewDatasetName(temp);
proc sort data=&datnr out=&datnrn;
by &nest1;
run;
%let datav = %NewDatasetName(temp);
data &datav (keep=&nest1 &nest2 &nest3 &dvnames &cvar1 &cvar2 &cvar3 &xvar1 &xvar2 &xvar3 &gcent &l1wgt &l2wgt &l3wgt &wgt);
set &datnrn;
if cmiss(of _all_)=0;
run;
data &datav (keep=&nest1 &nest2 &nest3 &ccent2 &ccent3 &gcent &l1wgt &l2wgt &l3wgt &wgt);
set &datav;
run;


/** if necessary center the variables now **/
%if %sysevalf(%superq(ccent2)=,boolean) = 0 or %sysevalf(%superq(ccent3)=,boolean) = 0 or %sysevalf(%superq(gcent)=,boolean) = 0 %then %do;
	proc sort data=&datav out=&datav;
	by &nest1 &nest2 &nest3;
	run;
	%if &dataset. in 1|3|5|7|9|11|13|15|17 %then %let wgtnc = weight &wgt;
	%if %sysevalf(%superq(ccent2)=,boolean) = 0 %then %do;
		%let cmean = %NewDatasetName(temp);
		proc means data=&datav noprint;
		by &nest2;
		var &ccent2;
		&wgtnc ;
		output out=&cmean (drop=_TYPE_ _FREQ_) mean=/autoname;
		run; 
		%let idtemp = %NewDatasetName(temp);
		data &idtemp (keep=&nest2);
		set &datav;
		run;
		%let cmtemp = %NewDatasetName(temp);
		data &cmtemp (drop=&nest2);
		merge &idtemp &cmean;
		by &nest2;
		run;
	%end;
	%if %sysevalf(%superq(ccent3)=,boolean) = 0 %then %do;
		%let c3mean = %NewDatasetName(temp);
		proc means data=&datav noprint;
		by &nest3;
		var &ccent3;
		&wgtnc ;
		output out=&c3mean (drop=_TYPE_ _FREQ_) mean=/autoname;
		run; 
		%let id3temp = %NewDatasetName(temp);
		data &id3temp (keep=&nest3);
		set &datav;
		run;
		%let cm3temp = %NewDatasetName(temp);
		data &cm3temp (drop=&nest3);
		merge &id3temp &c3mean;
		by &nest3;
		run;
	%end;
	%if %sysevalf(%superq(gcent)=,boolean) = 0 %then %do;
		%let gmean = %NewDatasetName(temp);
		proc means data=&datav noprint;
		var &gcent;
		&wgtnc ;
		output out=&gmean (drop=_TYPE_ _FREQ_) mean=/autoname;
		run;
	%end;	
	proc iml;
	use &datav;
	read all var {&nest1} into y01;
	read all var {&nest2} into y02;
	%if %sysevalf(%superq(nest3)=,boolean) = 0 %then %do;
		read all var {&nest3} into y03;
	%end;
	close &datav;
	%if %sysevalf(%superq(nest3)=,boolean) = 0 %then %do;
		id=y01||y02||y03;
	%end;
	%else %do;
		id=y01||y02;
	%end;
	%if %sysevalf(%superq(ccent2)=,boolean) = 0 %then %do;
		use &datav;
		read all var {&ccent2} into y1;
		close &datav;
		use &cmtemp;
		read all into y3;
		close &cmtemp;
		cvar=y1;
		cmean=y3;
		ccenter=cvar-cmean;
		Out1=id||ccenter;
		%let iml1 = %NewDatasetName(temp);
		create &iml1 from Out1[colname={&nest1 &nest2 &nest3 &ccent2}];
		append from Out1;
		close &iml1;
	%end;
	%if %sysevalf(%superq(ccent3)=,boolean) = 0 %then %do;
		use &datav;
		read all var {&ccent3} into y13;
		close &datav;
		use &cm3temp;
		read all into y33;
		close &cm3temp;
		svar=y13;
		smean=y33;
		ssenter=svar-smean;
		Out3=id||ssenter;
		%let iml8 = %NewDatasetName(temp);
		create &iml8 from Out3[colname={&nest1 &nest2 &nest3 &ccent3}];
		append from Out3;
		close &iml8;
	%end;
	%if %sysevalf(%superq(gcent)=,boolean) = 0 %then %do;	
		use &datav;
		read all var {&gcent} into y2;	
		close &datav;
		use &gmean;
		read all into y4;
		close &gmean;
		gvar=y2;
		gmean=y4;
		gcenter=gvar-gmean;
		Out2=id||gcenter;
		%let iml2 = %NewDatasetName(temp);
		create &iml2 from Out2[colname={&nest1 &nest2 &nest3 &gcent}];
		append from Out2;
		close &iml2;
	%end;
	quit;
	%if %sysevalf(%superq(ccent2)=,boolean) = 0 %then %do;
		%let iml3 = %NewDatasetName(temp);
		data &iml3;
		set &iml1;
		%RenameVariable(&ccent2, cc);
		run;
	%end;
	%if %sysevalf(%superq(ccent3)=,boolean) = 0 %then %do;
		%let iml9 = %NewDatasetName(temp);
		data &iml9;
		set &iml8;
		%RenameVariable(&ccent3, sc);
		run;
	%end;
	%if %sysevalf(%superq(gcent)=,boolean) = 0 %then %do;	
		%let iml5 = %NewDatasetName(temp);
		data &iml5;
		set &iml2;
		%RenameVariable(&gcent, gc);
		run;
	%end;
	data &datav;
	merge &datav &iml3 &iml9 &iml5;
	by &nest1 &nest2 &nest3;
	run;	
%end;


/** scale the weights **/
%if &dataset. in 1|2|5|6|9|10|13|14|15|16|17|18 %then %do;
	%if &dataset. in 13|14|15|16|17|18 %then %do;
		%let l1l3c0=l1l3c0;
		%let l1l3c1=l1l3c1;
		%let l1l3c2=l1l3c2;
		%let l1l3c0s=l1l3c0s;
		%let l1l3c1s=l1l3c1s;
		%let l1l3c2s=l1l3c2s;
		%let n6=n_6;
		%let n7=n_7;
		%let n8=n_8;
	%end;
	data &datav;
	set &datav;
	l1weight2 = &l1wgt*&l1wgt;
	run;
	%let datavm = %NewDatasetName(temp);
	proc means data=&datav noprint;
	by &nest2;
	var &l1wgt l1weight2;
	output out=&datavm sum=l1weight_s l1weight2_s n=n_1 n_2;
	run;
	data &datav (drop=_Type_ _Freq_ n_2);
	merge &datav &datavm;
	by &nest2;
	sfw1=n_1/l1weight_s;
	sfw2=l1weight_s/l1weight2_s;
	l1weightc1=&l1wgt*sfw1;
	l1weightc2=&l1wgt*sfw2;
	l1l2c0=&l1wgt*&l2wgt;
	l1l2c1=l1weightc1*&l2wgt;
	l1l2c2=l1weightc2*&l2wgt;
	%if &dataset. in 13|14|15|16|17|18 %then %do;
		l1l3c0=&l1wgt*&l3wgt;
		l1l3c1=l1weightc1*&l3wgt;
		l1l3c2=l1weightc2*&l3wgt;
	%end;
	run;
	%let datavm2 = %NewDatasetName(temp);
	proc means data=&datav noprint;
	var l1l2c0 l1l2c1 l1l2c2 &l1l3c0 &l1l3c1 &l1l3c2;
	output out=&datavm2 sum=l1l2c0s l1l2c1s l1l2c2s &l1l3c0s &l1l3c1s &l1l3c2s n=n_3 n_4 n_5 &n6 &n7 &n8;
	run;
	data &datav (drop=_Type_ _Freq_ n_4 n_5 &n6 &n7 &n8);
	set &datav;
	if _n_ eq 1 then do;
		set &datavm2;
	end;
	sfb0=n_3/l1l2c0s;
	sfb1=n_3/l1l2c1s;
	sfb2=n_3/l1l2c2s;
	l2weightc0=&l2wgt*sfb0;
	l2weightc1=&l2wgt*sfb1;
	l2weightc2=&l2wgt*sfb2;
	%if &dataset. in 13|14|15|16|17|18 %then %do;
		sfb30=n_3/l1l3c0s;
		sfb31=n_3/l1l3c1s;
		sfb32=n_3/l1l3c2s;
		l3weightc0=&l3wgt*sfb30;
		l3weightc1=&l3wgt*sfb31;
		l3weightc2=&l3wgt*sfb32;
	%end;
	run;
%end;


/** merge available cases with all other cases **/
proc sort data=&datav out=&datav;
by &nest1;
run;
%let datnrlm = %NewDatasetName(temp);
data &datnrlm;
merge &datav &datnrn;
by &nest1;
run;


/** get predictor variables names **/
%if %sysevalf(%superq(xvar1)=,boolean) = 0 %then %pvarname(old=&xvar1, ccent2=&ccent2, ccent3=&ccent3, gcent=&gcent, new=xvar1n);
%if %sysevalf(%superq(xvar2)=,boolean) = 0 %then %pvarname(old=&xvar2, ccent3=&ccent3, gcent=&gcent, new=xvar2n); 
%if %sysevalf(%superq(xvar3)=,boolean) = 0 %then %pvarname(old=&xvar3, gcent=&gcent, new=xvar3n);
%if %sysevalf(%superq(cvar1)=,boolean) = 0 %then %pvarname(old=&cvar1, ccent2=&ccent2, ccent3=&ccent3, gcent=&gcent, new=cvar1n);
%if %sysevalf(%superq(cvar2)=,boolean) = 0 %then %pvarname(old=&cvar2, ccent3=&ccent3, gcent=&gcent, new=cvar2n); 
%if %sysevalf(%superq(cvar3)=,boolean) = 0 %then %pvarname(old=&cvar3, gcent=&gcent, new=cvar3n);
%if %sysevalf(%superq(rslope2)=,boolean) = 0 %then %pvarname(old=&rslope2, ccent2=&ccent2, ccent3=&ccent3, gcent=&gcent, new=rslopen);
%if %sysevalf(%superq(rslope3)=,boolean) = 0 %then %pvarname(old=&rslope3, ccent2=&ccent2, ccent3=&ccent3, gcent=&gcent, new=rslopen3);


/** define the weigths **/
%if &dataset. in 1|2|5|6|9|10|13|14|15|16|17|18 %then %do;
	%if &sfw = 0 %then %do;
		%let l1wgts = &l1wgt; 
	%end;
	%else %if &sfw = 1 %then %do;
		%let l1wgts = l1weightc1;
	%end;
	%else %if &sfw = 2 %then %do;
		%let l1wgts = l1weightc2;
	%end;
	%else %do;
		%put %str(WA)RNING: An unknown value for SFW is observed. SFW is set to the default value of 2.;
		%let l1wgts = l1weightc2;
	%end;
	%if &sfb2 = 0 %then %do;
		%let l2wgts = &l2wgt;
	%end;
	%else %if &sfb2 = 1 %then %do;
		%let l2wgts = l2weightc0;
	%end;
	%else %if &sfb2 = 2 %then %do;
		%let l2wgts = l2weightc1;
	%end;
	%else %if &sfb2 = 3 %then %do;
		%let l2wgts = l2weightc2;
	%end;
	%else %do;
		%put %str(WA)RNING: An unknown value for SFB2 is observed. SFB2 is set to the default value of 3.;
		%let l2wgts = l2weightc2;
	%end;
	%if &dataset. in 13|14|15|16|17|18 %then %do;
		%if &sfb3 = 0 %then %do;
			%let l3wgts = &l3wgt;
		%end;
		%else %if &sfb3 = 1 %then %do;
			%let l3wgts = l3weightc0;
		%end;
		%else %if &sfb3 = 2 %then %do;
			%let l3wgts = l3weightc1;
		%end;
		%else %if &sfb3 = 3 %then %do;
			%let l3wgts = l3weightc2;
		%end;
		%else %do;
			%put %str(WA)RNING: An unknown value for SFB3 is observed. SFB3 is set to the default value of 3.;
			%let l3wgts = l3weightc2;
		%end;
	%end;
%end;


/** If necessary produce replication weights now (Part II) **/
%if &dataset. in 5|6|15|16 %then %do;
	%if %lowcase(&jktyp)=half %then %do;
		data &datnrlm;
		set &datnrlm;
		array l1rwgt l1rwgt1 - l1rwgt&nrwgt; 
		array l2rwgt l2rwgt1 - l2rwgt&nrwgt;
		%if &dataset. in 15|16 %then array l3rwgt l3rwgt1 - l3rwgt&nrwgt;; 
		do j=1 to &nrwgt;
			if       &jkzone ~= j     then do;
				l1rwgt(j) = &l1wgts * 1;
				l2rwgt(j) = &l2wgts * 1;
				%if &dataset. in 15|16 %then l3rwgt(j) = &l3wgts * 1;;
			end;
			else if (&jkzone  = j & &jkrep = 1) then do;
				l1rwgt(j) = &l1wgts * 2;
				l2rwgt(j) = &l2wgts * 1;
				%if &dataset. in 15|16 %then l3rwgt(j) = &l3wgts * 1;;
			end;
			else if (&jkzone  = j & &jkrep = 0) then do;
				l1rwgt(j) = &l1wgts * 0;
				l2rwgt(j) = &l2wgts * 0;
				%if &dataset. in 15|16 %then l3rwgt(j) = &l3wgts * 0;;
			end;	
		end;
		drop j;	
		run;
	%end;
	%else %if %lowcase(&jktyp)=full %then %do;
		data &datnrlm;
		set &datnrlm;
		array l1rwgt l1rwgt1 - l1rwgt&nrwgt; 
		array l2rwgt l2rwgt1 - l2rwgt&nrwgt; 
		%if &dataset. in 15|16 %then array l3rwgt l3rwgt1 - l3rwgt&nrwgt;; 
		do j=1 to &nrwgt;
			if       &jkzone ~= int((j+1)/2) then do;
				l1rwgt(j) = &l1wgts * 1;
				l2rwgt(j) = &l2wgts * 1;
				%if &dataset. in 15|16 %then l3rwgt(j) = &l3wgts * 1;;
			end;
			else if  &jkzone  = int((j+1)/2) then do;
				l1rwgt(j) = &l1wgts * (2 * mod(&jkrep + j,2));
				l2rwgt(j) = &l2wgts * (1 * mod(&jkrep + j,2));
				%if &dataset. in 15|16 %then l3rwgt(j) = &l3wgts * (1 * mod(&jkrep + j,2));; 
			end;
		end;
		drop j;	
		run;
	%end;
%end;


/** specify model components **/
%if %sysevalf(%superq(maxfunc)=,boolean) = 0 %then %let maxf = maxfunc=&maxfunc;
%if %sysevalf(%superq(maxiter)=,boolean) = 0 %then %let maxi = maxiter=&maxiter;
%if %lowcase(&noint) = y %then %let nointr = noint;
%if &mod. in 1|2|4|5|6|7|8|9|10|11|12|13 %then %do;
	%let sub2 = &nest2;
	%let lmmmethod = n;
	%if %lowcase(&dist) = gaussian or %lowcase(&dist) = g or %lowcase(&dist) = normal or %lowcase(&dist) = n %then %do;
		%if &dataset. in 3|4|7|8|11|12 %then %do;
			%let method = method=mspl;
			%let lmmmethod = y;
		%end;
		%else %do;
			%if %sysevalf(%superq(qpoints)=,boolean) = 1 %then %do;
				%let method = method=quad;
			%end;
			%else %do;
				%let method = method=quad(qpoints=&qpoints);
			%end;
		%end;
	%end;
	%else %do;
		%if %sysevalf(%superq(qpoints)=,boolean) = 1 %then %do;
			%let method = method=quad;
		%end;
		%else %do;
			%let method = method=quad(qpoints=&qpoints);
		%end;
	%end;
%end; 
%else %do;
	%let method = method=mspl;
	%let sub2 = &nest1;
	%let res = _residual_;
	%let type2=vc;
%end;
%if &mod. in 5|6|7|8|9|10|11|12|13 %then %do;
	%let sub3 = &nest3;
%end;
%if &mod. in 1|2|5|6|7|11|12|13 %then %do;
	%let inter2 = intercept;
%end;
%if &mod. in 5|7|8|10|11|13 %then %do;
	%let inter3 = intercept;
%end;
%if &dataset. in 1|2|5|6|9|10|13|14|15|16|17|18 %then %do;
	%let obsweight = obsweight=&L1WGTs;
	%let rweight2 = weight=&L2WGTs;
%end;
%if &dataset. in 13|14|15|16|17|18 %then %do;
	%let rweight3 = weight=&L3WGTs;
%end;
%if &dataset. in 3|7|11 %then %do;
	%let wgtlm = weight &wgt;
%end;


/** check TYP argument **/
%if %sysevalf(%superq(type2)=,boolean) = 1 %then %let type2 = vc;
%let checktyp = %substr(&type2,1,2);
%if %lowcase(&checktyp) = li %then %do;
	%if %sysevalf(%superq(ldata2)=,boolean) = 1 %then %do;
		%put %str(WA)RNING: Necessary argument for LDATA2 is missing. The default TYPE=VC is assumed.;
		%let type2 = vc;	
	%end; 
	%else %do;
		%let ldatan2 = ldata=&ldata2;
	%end;
%end;
%else %if %lowcase(&checktyp) = sp %then %do;
	%let type2 = &type2.(&clist2);
%end;
%if &mod. in 1|2|3|5|6|7|11|12|13 %then %do;
	%if %lowcase(&checktyp) = ps %then %do;
		%put %str(WA)RNING: TYPE=PSPLINE is supported for only a single continuous random effect. The default TYPE=VC is assumed.;
		%let type2 = vc;	
	%end;
	%else %if %lowcase(&checktyp) = rs %then %do;
		%put %str(WA)RNING: TYPE=RSMOOTH is supported for only continuous random effects. The default TYPE=VC is assumed.;
		%let type2 = vc;	
	%end;
%end;
%else %if &mod. in 4|8|9|10 %then %do;
	%if &nrslope > 1 %then %do;
		%if %lowcase(&checktyp) = ps %then %do;
			%put %str(WA)RNING: TYPE=PSPLINE is supported for only a single continuous random effect. The default TYPE=VC is assumed.;
			%let type2 = vc;	
		%end;
		%else %if %lowcase(&checktyp) = rs %then %do;
			%put %str(WA)RNING: TYPE=RSMOOTH is supported for only continuous random effects. The default TYPE=VC is assumed.;
			%let type2 = vc;	
		%end;
	%end;
%end;
%if &level = 3 %then %do;
	%if %sysevalf(%superq(type3)=,boolean) = 1 %then %let type3 = vc;
	%let checktyp2 = %substr(&type3,1,2);
	%if %lowcase(&checktyp2) = li %then %do;
		%if %sysevalf(%superq(ldata3)=,boolean) = 1 %then %do;
			%put %str(WA)RNING: Necessary argument for LDATA3 is missing. The default TYPE=VC is assumed.;
			%let type3 = vc;	
		%end; 
		%else %do;
			%let ldatan3 = ldata=&ldata3;
		%end;
	%end;
	%else %if %lowcase(&checktyp2) = sp %then %do;
		%let type3 = &type3.(&clist3);
	%end;
	%if &mod. in 5|6|7|11|12|13 %then %do;
		%if %lowcase(&checktyp2) = ps %then %do;
			%put %str(WA)RNING: TYPE=PSPLINE is supported for only a single continuous random effect. The default TYPE=VC is assumed.;
			%let type3 = vc;	
		%end;
		%else %if %lowcase(&checktyp2) = rs %then %do;
			%put %str(WA)RNING: TYPE=RSMOOTH is supported for only continuous random effects. The default TYPE=VC is assumed.;
			%let type3 = vc;	
		%end;
	%end;
	%else %if &mod. in 8|9|10 %then %do;
		%if &nrslope3 > 1 %then %do;
			%if %lowcase(&checktyp2) = ps %then %do;
				%put %str(WA)RNING: TYPE=PSPLINE is supported for only a single continuous random effect. The default TYPE=VC is assumed.;
				%let type3 = vc;	
			%end;
			%else %if %lowcase(&checktyp2) = rs %then %do;
				%put %str(WA)RNING: TYPE=RSMOOTH is supported for only continuous random effects. The default TYPE=VC is assumed.;
				%let type3 = vc;	
			%end;
		%end;
	%end;
%end;


/** perform GLMM analysis **/
%if %lowcase(&srvysam) = y and &dataset. in 9|10|11|12|17|18 %then %do;
	%put %str(WA)RNING: Replication weights are missing. Survey sample variances can not be calculated.;
	%let &srvysam = n;
%end;
%if %lowcase(&srvysam) = y %then %do;
	%let nrwgtc = &nrwgt;
%end;
%else %if %lowcase(&srvysam) = n %then %do;
	%let nrwgtc = 0;
	%put %str(WA)RNING: No survey sample variances are requested for GLMM coefficients (SRVYSAM~=Y). The presented standard errors are based on ordinary GLMM analysis.;
%end;

%do pv = 1 %to &npv;
	%let nrwgtpv = nrwgt&pv.;
	%let nrwgt&pv. = 0;	
	%if &npv = 1 %then %do;
		%let y = &dvnames;
	%end;
	%else %do;
		%let y = &rootpv&pv;		
	%end;
	%if %lowcase(&shrtcut) = y and &pv > 1 %then %do;
		%let nrwgtc = 0;
	%end;	
	%do i = 0 %to &nrwgtc;
		%if &i > 0 %then %do;
			%let fitstat = ;
			%let nobsglm = ; 
			%let gmat = ;
			%let constat = ;
			%let gs = ;
			%let gv = ;
			%if &dataset. in 1|2|5|6|13|14|15|16 %then %do; 
				%let obsweightc = obsweight=l1rwgt&i.;
				%let rweight2c = weight=l2rwgt&i.;
			%end;
			%else %do;
				%let obsweightc = ;
				%let rweight2c = ;
			%end;		
			%if &dataset. in 13|14|15|16 %then %do;
				%let rweight3c = weight=l3rwgt&i.;
			%end;
			%else %do;
				%let rweight3c = ;
			%end;
			%if &dataset. in 3|7 %then %do;
				%let wgtlmc = weight rwgt&i.;
			%end;	
			%else %do;
				%let wgtlmc = ;
			%end;
			%if %lowcase(&graph)=y %then %do;
				%let plotsm = ;
			%end;
		%end;
		%else %do;
			%let obsweightc = &obsweight;
			%let rweight2c = &rweight2;
			%let rweight3c = &rweight3;
			%let wgtlmc = &wgtlm;
			%let fitstat = FitStatistics=svyhlm_f_&pv.;
			%let nobsglm = NObs=svyhlm_n_&pv.; 
			%if &mod. ~= 3 %then %do;
				%let gmat = G=svyhlm_g_&pv.;
				%if &lmmmethod = n %then %do;
					%let constat = CondFitStatistics=svyhlm_cf_&pv.;
				%end;
			%end;
			%if %lowcase(&reffect)=y %then %do;
				%let gs = s;
				%let gv = SolutionR=svyhlm_gv_&pv.;
			%end;
			%if %lowcase(&graph)=y %then %do;
				%let plotsm = plots=residualpanel;
			%end;
		%end;
		%if &mod. ~= 3 %then %do;
			%let convst = ConvergenceStatus=svyhlm_c_&pv._jk_&i.; 
		%end;	
		
		%if %lowcase(&graph)=y and &i=0 %then %do;
			ods graphics on / imagename="svyhlm_&label._plot&pv._" ;
			ods listing gpath="&libe.";
			ods select residualpanel;						
		%end;
						
		proc glimmix data=&datnrlm &method &plotsm empirical=classical;
		nloptions gconv=&gconv &maxf &maxi technique=&tec;
		class &sub3 &sub2 &cvar1n &cvar2n &cvar3n;
		model &y = &xvar1n &xvar2n &xvar3n &cvar1n &cvar2n &cvar3n / &nointr solution dist=&dist &obsweightc;
		%if &mod. in 1|2|3|4 %then %do;
			random &inter2 &res &rslopen / sub=&sub2 g &gs &rweight2c type=&type2 &ldatan2;
		%end;
		%else %if &mod. in 5|6|7|8|9|10|11|12|13 %then %do;
			random &inter2 &rslopen / sub=&sub2(&sub3) g &gs &rweight2c type=&type2 &ldatan2;
			random &inter3 &rslopen3 / sub=&sub3 g &gs &rweight3c type=&type3 &ldatan3;
		%end;
		;
		%if &i=0 and &pv > 1 and %lowcase(&start)=y %then %do;
			parms / pdata=svyhlm_vp_1;
		%end;
		;
		&wgtlmc ;
		ods output CovParms=svyhlm_vp_&pv._jk_&i. ParameterEstimates=svyhlm_p_&pv._jk_&i. &convst
		&gmat &constat &gv &fitstat &nobsglm ;
		run; 
		
		%if %lowcase(&graph)=y and &i=0 %then %do;
			ods select none;
			ods listing close;
			ods graphics off;						
		%end;
		
		%if &i > 0 %then %do;
			%if &mod~=3 %then %do;
				%if %sysfunc(exist(svyhlm_c_&pv._jk_&i.)) = 0 or %sysfunc(exist(svyhlm_p_&pv._jk_&i.)) = 0 or %sysfunc(exist(svyhlm_vp_&pv._jk_&i.)) = 0 %then %do;
					data svyhlm_c_&pv._jk_&i. (drop=Status);
					set svyhlm_c_&pv.;
					Statusr = 2;
					run;
					data svyhlm_c_&pv._jk_&i. (rename=(Statusr=Status));
					set svyhlm_c_&pv._jk_&i;
					run;
				%end;
				data svyhlm_c_&pv._jk_&i. (keep=Status);
				set svyhlm_c_&pv._jk_&i.;
				call symput('constat',Status);	
				run;
				%if &constat = 0 %then %do;
					%let nrwgt&pv. = %sysevalf(&&nrwgt&pv. + 1);
					data svyhlm_vp_&pv._jk_&&nrwgt&pv. (keep=Estimate rename=(Estimate=Estimate&&nrwgt&pv.));
					set svyhlm_vp_&pv._jk_&i.;
					run;
					data svyhlm_p_&pv._jk_&&nrwgt&pv. (keep=Estimate rename=(Estimate=Estimate&&nrwgt&pv.));
					set svyhlm_p_&pv._jk_&i.;
					run;
				%end;
				%else %do;
					data svyhlm_nc_&pv._jk_&i.;
					set svyhlm_c_&pv._jk_&i.;
					rwgt = &i;
					run;				
				%end;
			%end;
			%else %do;
				%if %sysfunc(exist(svyhlm_p_&pv._jk_&i.)) = 1 or %sysfunc(exist(svyhlm_vp_&pv._jk_&i.)) = 1 %then %do;
					%let nrwgt&pv. = %sysevalf(&&nrwgt&pv. + 1);
					data svyhlm_vp_&pv._jk_&&nrwgt&pv. (keep=Estimate rename=(Estimate=Estimate&&nrwgt&pv.));
					set svyhlm_vp_&pv._jk_&i.;
					run;
					data svyhlm_p_&pv._jk_&&nrwgt&pv. (keep=Estimate rename=(Estimate=Estimate&&nrwgt&pv.));
					set svyhlm_p_&pv._jk_&i.;
					run;			
				%end;
			%end;
		%end;
		%else %do;	
			/* rename data sets */
			data svyhlm_vp_&pv.;
			set svyhlm_vp_&pv._jk_&i.;
			run;			
			data svyhlm_p_&pv.;
			set svyhlm_p_&pv._jk_&i.;
			run;
			%if &mod. ~= 3 %then %do;
				data svyhlm_c_&pv.;
				set svyhlm_c_&pv._jk_&i.;
				run;
			%end;
				
			/* check for convergent */
			%if &mod. ~= 3 %then %do;
				data svyhlm_c_&pv. (keep=reason status);
				set svyhlm_c_&pv.;
				call symput('constat', status);
				call symput('reason', reason);
				run;
				proc transpose data=svyhlm_c_&pv. out=svyhlm_c_t_&pv.;
				var reason status;
				run;
				proc sort data=svyhlm_c_t_&pv.;
				by descending _name_;
				run;
				data svyhlm_c_print_&pv. (rename=(_name_=convergent col1=%upcase(&y)));
				set svyhlm_c_t_&pv.;
				label _name_ = ' ';
				run;
				%if &constat ~= 0 %then %do;
					proc print data=svyhlm_c_print_&pv. noobs;
					title "Convergent status";
					run;
					%put %str(ER)ROR: Model is not converged. Analysis were interrupted.;
				%end;
				%if &constat ~= 0 %then %return;
			%end;
			
			/* check for post-processing information */
			%if %sysfunc(exist(svyhlm_n_&pv.)) = 0 or %sysfunc(exist(svyhlm_f_&pv.)) = 0 or %sysfunc(exist(svyhlm_vp_&pv.)) = 0 or %sysfunc(exist(svyhlm_p_&pv.)) = 0 %then %do;
				%if &mod. ~=3 %then %do;
					%put %str(WA)RNING: &reason;
				%end;
				%put %str(ER)ROR: Post-processing information is missing. Analysis were interrupted.;
			%end;
			%if %sysfunc(exist(svyhlm_n_&pv.)) = 0 or %sysfunc(exist(svyhlm_f_&pv.)) = 0 or %sysfunc(exist(svyhlm_vp_&pv.)) = 0 or %sysfunc(exist(svyhlm_p_&pv.)) = 0 %then %return;
			%if &mod. ~=3 %then %do;
				%if %sysfunc(exist(svyhlm_c_&pv.)) = 0 or %sysfunc(exist(svyhlm_g_&pv.)) = 0 %then %do;
					%put %str(ER)ROR: Post-processing information is missing. Analysis were interrupted.;
				%end; 
				%if %sysfunc(exist(svyhlm_c_&pv.)) = 0 or %sysfunc(exist(svyhlm_g_&pv.)) = 0 %then %return;
				%if &lmmmethod = n %then %do;
					%if %sysfunc(exist(svyhlm_cf_&pv.)) = 0 %then %do;
						%put %str(ER)ROR: Post-processing information is missing. Analysis were interrupted.;
					%end;
					%if %sysfunc(exist(svyhlm_cf_&pv.)) = 0 %then %return;
				%end;
			%end;	
			
			/* restructure data sets */
			data svyhlm_n_r_&pv. (keep=Label N rename=(N=%upcase(&y)));
			set svyhlm_n_&pv.;
			run;
			data svyhlm_f_r_&pv. (rename=(Value=Value&PV.));
			set svyhlm_f_&pv.;
			run;
			data svyhlm_vp_r_&pv.;
			set svyhlm_vp_&pv.;
			%if &npv > 1 %then %do;
				_Imputation_=&pv.;
			%end;
			run;
			data svyhlm_p_r_&pv.;
			set svyhlm_p_&pv.;
			%if &npv > 1 %then %do;
				_Imputation_=&pv.;
			%end;
			run;
			%if &mod. ~=3 and &lmmmethod = n %then %do;
				data svyhlm_cf_r_&pv. (rename=(Value=Value&PV.));
				set svyhlm_cf_&pv.;
				run;
			%end;
		%end;
	%end;
%end;


/** get results **/
/* number of observations */
data svyhlm_n_print;
merge %namep(svyhlm_n_r_,&npv);
run;

/* fit statistics */
data svyhlm_f_r_c (rename=(Descr=Description));
merge %namep(svyhlm_f_r_,&npv);
run;
data svyhlm_f_print (keep=Description Value);
set svyhlm_f_r_c;
Value = mean(of Value1--Value&npv);
run;

/* random components */
proc sql noprint; 
select count(*) into :NVPEF from svyhlm_vp_1; 
quit;
%let Gname1 = %NewDatasetName(temp);
data &gname1 (keep=Effect);
do i=1 to %sysevalf(&nvpef-1);
	Row = "Row";
	Effect = cats(Row,I);
	output;
end;
run;
%let Gname2 = %NewDatasetName(temp);
data &gname2;
Effect = "Residual";
run;
%let Gname = %NewDatasetName(temp);
data &gname;
set &gname1 &gname2;
run;
%do i=1 %to &npv;
	data svyhlm_vp_r_r_&i. (drop=CovParm);
	set svyhlm_vp_r_&i.;
	if StdErr = . then StdErr = 1;
	run;
	data svyhlm_vp_r_c_&i. (keep=StdErr&i.);
	set svyhlm_vp_r_&i.;
	if StdErr = . then StdErr&i. = .;
	else StdErr&i. = StdErr;
	run;
	data svyhlm_vp_r_nc_&i.;
	merge &gname svyhlm_vp_r_r_&i.; 
	run;
%end;  
data svyhlm_vp_c;
set %namep(svyhlm_vp_r_nc_,&npv);
run;
proc sql noprint; 
select Effect into :VPEF1-:VPEF%left(&NVPEF) from svyhlm_vp_r_nc_1; 
quit;
%do i = 1 %to &nvpef;
	%let VPEFn = &VPEFn &&VPEF&I;
%end;
%if &npv =1 %then %do;
	data svyhlm_vp_p_print;
	set  svyhlm_vp_1;
	run;
%end;
%else %do;
	proc mianalyze parms=svyhlm_vp_c;
	modeleffects &VPEFn;
	ods output ParameterEstimates=svyhlm_vp_p_print VarianceInfo=svyhlm_vp_vi_m;
	run;
%end;
data svyhlm_vp_p_print (keep=Estimate StdErr);
set svyhlm_vp_p_print;
run; 
%if &mod~=3 %then %do;
	%let Subject = Subject;
%end;
data svyhlm_vp_r_sc_1 (keep=CovParm &subject);  
set svyhlm_vp_r_1;
run;
data svyhlm_vp_r_c;
merge %namep(svyhlm_vp_r_c_,&npv);
run;
data svyhlm_vp_mis (keep=DM);
set svyhlm_vp_r_c;
y = nmiss(of StdErr1-StdErr&npv.);
if y ~= 0 then DM = 1;
else DM = 0;
run;
data svyhlm_vp_print_1;
merge svyhlm_vp_mis svyhlm_vp_r_sc_1 svyhlm_vp_p_print;
if DM = 1 then StdErr = .;
run;
data svyhlm_vp_print (drop=DM);
set svyhlm_vp_print_1;
run;

/* parameter estimates */
data svyhlm_p_c;
set %namep(svyhlm_p_r_,&npv);
run;
%if %lowcase(&noint) ~= y %then %let intercept = Intercept;
%if %lowcase(&distdv) = multinomial or %lowcase(&distdv) = multi or %lowcase(&distdv) = mult %then %do;
	%if &npv=1 %then %do;
		%let listdv = &rootpv;
	%end;
	%else %do;
		%let listdv = &rootpv.1-&rootpv&npv.;
	%end;	
	data svyhlm_p_c;
	set svyhlm_p_c;
	Intercept = sum(of &listdv.);
	run;
	%let interceptn = intercept;
%end;
%if &npv = 1 %then %do;
	data svyhlm_p_print;
	set svyhlm_p_1;
	run;
%end;
%else %do;
	proc mianalyze parms=svyhlm_p_c;
	class &interceptn &cvar1n &cvar2n &cvar3n;
	modeleffects &intercept &xvar1n &xvar2n &xvar3n &cvar1n &cvar2n &cvar3n;
	ods output ParameterEstimates=svyhlm_p_m VarianceInfo=svyhlm_p_vi_m;
	run;
	data svyhlm_p_print (drop=LCLMean UCLMean Min Max Theta0);
	set svyhlm_p_m;
	run;
%end;

%if &mod. ~=3 %then %do;
	/* convergent status */
	data svyhlm_c (keep=reason status);
	length Reason $98;
	set %namep(svyhlm_c_,&npv);
	run;
	proc transpose data=svyhlm_c out=svyhlm_c_t;
	var reason status;
	run;
	proc sort data=svyhlm_c_t;
	by descending _name_;
	run;
	%if &npv = 1 %then %do;
		%let conv1 = col1=&dvnames;
	%end;
	%else %do;
		%let conv1 = col1-col&npv.=%upcase(&rootpv.1-&rootpv&npv.);
	%end;
	data svyhlm_c_print (rename=(_name_=Convergent &conv1));
	set svyhlm_c_t;
	label _name_ = ' ';
	run;

	/* conditional fit statistics */
	%if &lmmmethod = n %then %do;
		data svyhlm_cf_r_c (rename=(Descr=Description));
		merge %namep(svyhlm_cf_r_,&npv);
		run;
		data svyhlm_cf_print (keep=Description Value);
		set svyhlm_cf_r_c;
		Value = mean(of Value1--Value&npv);
		run;
	%end;

	/* G matrix */
	data svyhlm_g_c;
	set %namep(svyhlm_g_,&npv);
	run;
	%if &level = 3 %then %do;
		proc sort data=svyhlm_g_c;
		by Stmt;
		run;
	%end;
	%let gred = %NewDatasetName(temp);
	data &gred (keep=col:);
	set svyhlm_g_c;
	run;
	%let dsnid2 = %sysfunc(open(&gred));
	%let neffect=%sysfunc(attrn(&dsnid2,nvars));
	%if &dsnid2 > 0 %then %let rc=%sysfunc(close(&dsnid2));
	proc means data=svyhlm_g_c;
	class row;
	var COL1-COL%left(&NEffect);
	%if &level = 3 %then %do;
	 by Stmt;
	%end;
	output out=svyhlm_g_mean mean=;
	run;	
	data svyhlm_g_labels (drop=col:);
	set svyhlm_g_1;
	run;
	data svyhlm_g_print_1 (drop=Row _TYPE_ _FREQ_);
	set svyhlm_g_mean;
	if _type_ = 0 then delete;
	run;
	data svyhlm_g_print;
	merge svyhlm_g_labels svyhlm_g_print_1;
	run;

	/* correlation matrix of random components */
	%if &level ~= 3 %then %do;
		proc sql noprint; 
		select count(*) into :NEffect from svyhlm_g_print; 
		quit;
		%do i = 1 %to &NEffect;
			%let Coln = &Coln "Col&I";
		%end;
		proc iml;
		use svyhlm_g_print;
		read all var {&Coln} into G;
		close svyhlm_g_print;
		Corr = Cov2Corr(G);
		%let iml7 = %NewDatasetName(temp);
		create &iml7 from Corr;
		append from Corr;
		close &iml7;
		quit;
		data svyhlm_r_print;
		merge svyhlm_g_labels &iml7;
		run;
	%end;
	%else %do;
		data svyhlm_g_1_print;
		set svyhlm_g_print;
		if stmt ~=1 then delete;
		run;
		proc sql noprint; 
		select count(*) into :NEffect1 from svyhlm_g_1_print; 
		quit; 
		%do i = 1 %to &NEffect1;
			%let Coln1 = &Coln1 "Col&I";
		%end;
		data svyhlm_g_2_print;
		set svyhlm_g_print;
		if stmt ~=2 then delete;
		run;
		proc sql noprint; 
		select count(*) into :NEffect2 from svyhlm_g_2_print; 
		quit; 
		%do i = 1 %to &NEffect2;
			%let Coln2 = &Coln2 "Col&I";
		%end;
		proc iml;
		use svyhlm_g_1_print;
		read all var {&Coln1} into G1;
		close svyhlm_g_1_print;
		use svyhlm_g_2_print;
		read all var {&Coln2} into G2;
		close svyhlm_g_2_print;
		Corr1 = Cov2Corr(G1);
		Corr2 = Cov2Corr(G2);
		%let iml7 = %NewDatasetName(temp);
		create &iml7 from Corr1;
		append from Corr1;
		close &iml7;
		%let iml10 = %NewDatasetName(temp);
		create &iml10 from Corr2;
		append from Corr2;
		close &iml10;
		quit;
		%let iml11 = %NewDatasetName(temp);
		data &iml11;
		set &iml7 &iml10;
		run;
		data svyhlm_r_print;
		merge svyhlm_g_labels &iml11;
		run;
	%end;
%end;


/** calculate survey variance **/
%if %lowcase(&srvysam) = y %then %do;
	%if %lowcase(&shrtcut) = y %then %do;
		%let npvr = 1;
	%end;
	%else %do;
		%let npvr = &npv;
	%end;
	
	/* convergence status */
	%if &mod~=3 %then %do;
		%do pv = 1 %to &npvr;
			%let tempcc = %NewDatasetName(temp);
			data &tempcc;
			set %namep(svyhlm_c_&pv._jk_,&nrwgt);
			run;
			%let tempcf = %NewDatasetName(temp);
			proc freq data=&tempcc;
			tables Status /out=&tempcf;
			run;
			%let tempcfr = %NewDatasetName(temp);
			data &tempcfr (keep=countr);
			set &tempcf;
			if status ~= 0 then countr = count;
			else countr = 0; 
			run;
			%let tempcm = %NewDatasetName(temp);
			proc means data=&tempcfr;
			var countr;
			output out=&tempcm sum=countrs;
			run;
			%let tempcmr = %NewDatasetName(temp);
			data &tempcmr (keep=countrs);
			set &tempcm;
			call symputx('countc',countrs);
			run;
			%if &countc ~= 0 %then %do;
				%let sucrep = %sysevalf(&nrwgt-&countc);
				%put %str(WA)RNING: &countc replications from &nrwgt replications for NPV=&PV. are not converged or failed to obtain post-processing information. Survey sample standard errors for the GLMM coefficients are based on &sucrep successful replications.;
			%end;
		%end;
	%end;
		
	/* calculate variance */
	%do pv = 1 %to &npvr;
		%let tempjkp = %NewDatasetName(temp);
	 	data &tempjkp;
		merge %namep(svyhlm_p_&pv._jk_,&&nrwgt&pv.);
		run;
		%let tempjkv = %NewDatasetName(temp);
	 	data &tempjkv;
		merge %namep(svyhlm_vp_&pv._jk_,&&nrwgt&pv.);
		run;
		proc iml;
		start e(p,p1,p2) global(E);
		pp=p[,1];
		do i=1 to &&nrwgt&pv.;
			pdif=(pp-p1[,i])##2;
			pdifall=pdifall||pdif;
		end;
		jkvar1=pdifall[,+];
		jkvar=&jkfac*jkvar1;
		%if &npv>1 %then %do;
			impv = p2[,1];
			impvar=%sysevalf(%sysevalf(&npv+1)/&npv)*impv;
			totvar=jkvar+impvar;
		%end;
		%else %do;
			totvar=jkvar;
		%end;
		E=totvar;
		finish e;
		use svyhlm_p_&pv.;
		read all var{Estimate} into p;
		use &tempjkp;
		read all var _num_ into p1;
		%if &npv>1 %then %do;
			use svyhlm_p_vi_m;
			read all var{BetVar} into p2;
		%end;
		%else %do;
			use svyhlm_p_&pv.;
			read all var{Estimate} into p2;
		%end;
		run e(p,p1,p2);
		create svyhlm_p_iml1_&pv. from E[colname={"Tot&pv."}];
		append from E;
		close svyhlm_p_iml1_&pv.;
		use svyhlm_vp_&pv.;
		read all var{Estimate} into p;
		use &tempjkv;
		read all var _num_ into p1;
		%if &npv>1 %then %do;
			use svyhlm_vp_vi_m;
			read all var{BetVar} into p2;
		%end;
		%else %do;
			use svyhlm_vp_&pv.;
			read all var{Estimate} into p2;
		%end;
		run e(p,p1,p2);
		create svyhlm_vp_iml1_&pv. from E[colname={"Tot&pv."}];
		append from E;
		close svyhlm_vp_iml1_&pv.;
		quit;			
	%end;
	%let tempjkpc = %NewDatasetName(temp);
	data &tempjkpc;
	merge %namep(svyhlm_p_iml1_,&npvr);
	run;
	proc iml;
	start f(p,p1) global(F);
	pf=p[,1];
	df=p[,2];
	totvar=p1;
	totvar1=(totvar[,+])/(&npvr);
	stder=totvar1##.5;
	tvalue=divide(pf,stder);
	do i = 1 to nrow(df);
		if df[i] > 0 then prtc = 1-probt(abs(tvalue[i]),df[i]);
		else prtc = .;
		prt=prt//prtc;
	end;
	F=stder||df||tvalue||prt;
    finish f;
	use svyhlm_p_print;
	read all var{Estimate DF} into p;
	use &tempjkpc;
	read all var _num_ into p1;
	run f(p,p1);
	%let tempjkpr = %NewDatasetName(temp);
	create &tempjkpr from F[colname={"StdErr" "DF" "tvalue" "Probt"}];
	append from F;
	close &tempjkpr;
	quit;		
	%let tempjkvc = %NewDatasetName(temp);
	data &tempjkvc;
	merge %namep(svyhlm_vp_iml1_,&npvr);
	run;
	proc iml;
	start g(p) global(G);
	totvar=p;
	totvar1=(totvar[,+])/(&npvr);
	stder=totvar1##.5;
	G=stder;
    finish g;
	use &tempjkvc;
	read all var _num_ into p;
	run g(p);
	%let tempjkvr = %NewDatasetName(temp);
	create &tempjkvr from G[colname={"StdErr"}];
	append from G;
	close &tempjkvr;
	quit;		
	
	/* merge survey variance with parameter estimates */
	data svyhlm_vp_so (rename=(StdErr=StdErro));
	set svyhlm_vp_print;
	run;
	data svyhlm_vp_print (drop=StdErro);
	merge svyhlm_vp_so &tempjkvr;
	run;
	data svyhlm_p_rs (drop=StdErr DF tValue Probt);
	set svyhlm_p_print;
	run;	
	data svyhlm_p_print;
	merge svyhlm_p_rs &tempjkpr;
	format probt pvalue6.4;
	run;
%end;



/*** Save and print results ***/
%if &mod. ~=3 %then %do;
	data e.svyhlm_&label._c;
	set svyhlm_c_print;
	run;
	%if &lmmmethod = n %then %do;
		data e.svyhlm_&label._cf;
		set svyhlm_cf_print;
		run;
	%end;
	data e.svyhlm_&label._g;
	set svyhlm_g_print;
	run;
	data e.svyhlm_&label._r;
	set svyhlm_r_print;
	run;
%end;
data e.svyhlm_&label._n;
set svyhlm_n_print;
run;
data e.svyhlm_&label._f;
set svyhlm_f_print;
run;
data e.svyhlm_&label._vp;
set svyhlm_vp_print;
run;
data e.svyhlm_&label._p;
set svyhlm_p_print;
run;
%do pv=1 %to &npv;
	data e.svyhlm_&label._p&pv.;
	set svyhlm_p_&pv.;
	run;
	data e.svyhlm_&label._vp&pv.;
	set svyhlm_vp_&pv.;
	run;
%end;
%if %upcase(&reffect)=Y %then %do;
	%do pv=1 %to &npv;
		data e.svyhlm_&label._gv&pv.;
		set svyhlm_gv_&pv.;
		run;
	%end;
%end;

ods listing;
ods select all;
filename results "&libe.svyhlm_&label._hlm.xml";
ODS TAGSETS.EXCELXP
file=results
STYLE=htmlblue
OPTIONS (FitToPage='yes' absolute_column_width='20' autofit_height='yes' Pages_FitWidth='1' sheet_interval='none' sheet_name='GLMM results'   embedded_titles='yes' embedded_footnotes='yes' embed_titles_once = 'yes');
%if &mod. ~=3 %then %do;
	proc print data=svyhlm_c_print noobs;
	title1 "Convergent status";
	run;
%end;
proc print data=svyhlm_n_print noobs;
title1 "Number of observations";
run;
proc print data=svyhlm_f_print noobs;
title1 "Fit statistics";
run;
%if &mod. ~=3 %then %do;
	%if &lmmmethod = n %then %do;
		proc print data=svyhlm_cf_print noobs;
		title1 "Conditional fit statistics";
		run;
	%end;
	proc print data=svyhlm_g_print noobs;
	title1 "Random components (covariances)";
	run;
	proc print data=svyhlm_r_print noobs;
	title1 "Random components (correlations)";
	run;
%end;
proc print data=svyhlm_vp_print noobs;
title1 "Random components (standard errors)";
run;
proc print data=svyhlm_p_print noobs;
title1 "Fixed effects";
run;	
ods tagsets.excelxp close;



/*** Tidy up ***/
%exit:
proc datasets nolist;   
    delete _temp _temp: svy: survey: pvfreq:;   
 run; quit;
%mend surveyhlm;