# The versions of the packages are available in the Project.toml and Manifest.toml files, which are provided as part of the online material.
# This file should be read in conjunction with the paper, as most of the comments are done in the paper's main body.

using Pkg
Pkg.activate(pwd())
Pkg.instantiate()
Pkg.add("StatisticalProcessMonitoring")

using StatisticalProcessMonitoring

#——————————————————————————————————————————————————————————
@info "Section 3.2.1: Deterministic time-varying control limits"
using LaTeXStrings, Plots
plt = plot(dpi = 300, legend = :outertop, xlab = L"t", ylab = "value")
colours = palette(:tab10)
λ_vec = [0.05, 0.1, 0.2]
for i in 1:3
    λ = λ_vec[i]
    f(t) = (1 - 0.5^((7 + 3 * t) / 10)) * sqrt(λ/(2 - λ) * (1 - (1 - λ)^(2 * t)))
    LIM = TwoSidedCurvedLimit(1.0, f)

    ll = hcat([get_value(LIM, t) for t in 1:100]...)'
    plot!(plt, ll[:,1], linestyle = :dash, colour = colours[i], label = "", dpi = 300)
    plot!(plt, ll[:,2], linestyle = :dash, colour = colours[i], label = L"\lambda = "*"$(λ)")
end
plt
savefig(plt, "Figures/fir-limits-ewma.png")
println("Saved image Figures/fir-limits-ewma.png")

#——————————————————————————————————————————————————————————
# Section 3.2.1: Time-varying control limits with constant false-alarm rate
STAT = EWMA(0.2, 0.0)
LIM = TwoSidedBootstrapLimit(STAT, 10000)

#——————————————————————————————————————————————————————————
# Section 3.2.2: Control limit design
# Type the character `?` in the julia REPL. The prompt should turn to `help?>`. Type the following input and press `<Enter>`:
# help?> saCL!
# Function documentation output should be displayed in the julia REPL.

#——————————————————————————————————————————————————————————
# Section 3.3: Nominal properties
NM = ARL(200)
NM = QRL(200, 0.3)

#——————————————————————————————————————————————————————————
# Section 3.4: Simulating new observations
using Distributions
PH2 = Phase2Distribution(Normal(0,1))
PH2 = Phase2(Bootstrap(), randn(500))

#——————————————————————————————————————————————————————————
# Section 4: Examples
using StatisticalProcessMonitoring, CategoricalArrays, CSV, DataFrames, Distributions, LaTeXStrings, LinearAlgebra, NLopt, Parameters, Plots, Random 

#——————————————————————————————————————————————————————————
@info "Section 4.1: Jointly monitoring the mean and covariance using a multi-chart scheme"
seed = 54397858713
Random.seed!(seed)
# Normal distribution given by example 7.7 from Qiu (2013)
μ = [0, 0, 0]
Σ = [1.0 0.8 0.5; 0.8 1.0 0.8; 0.5 0.8 1.0]
DIST = MultivariateNormal(μ, Σ)

p = length(μ)
STATS = (MCUSUM(k = 0.25, p = p), MCUSUM(k = 0.5, p = p), MCUSUM(k = 1.0, p = p), MEWMC(λ = 0.2, p = p))
# Statistics with estimated parameters
Σ_sqm1 = inv(sqrt(Σ))
EST_STATS = Tuple(LocationScaleStatistic(s, μ, Σ_sqm1) for s in STATS)

LIMS = Tuple(OneSidedFixedLimit(1.0, true) for _ in 1:4)

NM = QRL(200, 0.5)

PH2 = Phase2Distribution(DIST)

CH = ControlChart(EST_STATS, LIMS, NM, PH2)
h = bootstrapCL!(CH)
println(h)
# (h = [14.568353665174286, 8.627690357206408, 4.677555568948659, 2.360499495019121], iter = 8, status = "Convergence")

data = CSV.read("Data/example77.csv", DataFrame)
xnew = Matrix(data)[:, 1:3]

proc = apply_chart(CH, xnew)
plt = plot_series(proc, dpi = 300, label = "", xlab = L"t", ylab = L"C_t", subtitles = ["MCUSUM k = 0.25", "MCUSUM k = 0.5", "MCUSUM k = 1", "MEWMC λ = 0.2"])
plt
savefig(plt, "Figures/example-multiple-mean-covariance.png")
println("Saved image Figures/example-multiple-mean-covariance.png")

#——————————————————————————————————————————————————————————
@info "Section 4.2: Residual-based monitoring of autocorrelated data"
import StatisticalProcessMonitoring.residual!, StatisticalProcessMonitoring.new_data!
mutable struct AR1Statistic{S} <: ResidualStatistic
    stat::S
    phi::Float64
    ym1::Float64
end

function residual!(x, S::AR1Statistic)
    yhat = x - S.phi * S.ym1   
    S.ym1 = x
    return yhat
end

@with_kw mutable struct Phase2AR1<:StatisticalProcessMonitoring.AbstractPhase2
    phi::Float64
    y::Float64 = 0.0
    init::Bool = false
end

function new_data!(PH2::Phase2AR1)
    if !PH2.init
        PH2.y = randn() / sqrt(1 - PH2.phi^2)
        PH2.init = true
    end
    yhat = PH2.phi * PH2.y + randn()
    PH2.y = yhat
    return yhat
end

seed = 4398354798
Random.seed!(seed)
phi = 0.5
PH2 = Phase2AR1(phi = phi)

STAT = AR1Statistic(EWMA(λ = 0.1), phi, 0.0)

LIM = TwoSidedFixedLimit(1.0)

NOM = ARL(500)
CH = ControlChart(STAT, LIM, NOM, PH2)

delta = 2.0
# run_sim_oc generates a run length for data which undergo a mean shift of `shift`
rlsim_oc = x -> run_sim_oc(x, shift = delta)
settings = OptSettings(verbose = false, minpar = [0.001], maxpar = [0.99])
opt = optimize_design!(CH, rlsim_oc, settings, optimizer = :LN_BOBYQA)
println(opt)
# 1-element Vector{Float64}:
#  0.13830568163003895

n = 100
tau = 50
DGP = Phase2AR1(phi = phi)
y = [new_data!(DGP) for _ in 1:n]
y[(tau+1):n] .+=  delta

proc = apply_chart(CH, y)
plt = plot_series(proc, dpi = 300, label = "", xlab = L"t", ylab = L"C_t")
vline!(plt, [tau], label = L"\tau", linestyle = :dot, colour = "black")
plt
savefig(plt, "Figures/example-autocorrelated.png")
println("Saved image Figures/example-autocorrelated.png")

#——————————————————————————————————————————————————————————
@info "Section 4.3: Monitoring surgical outcomes using risk-adjusted control charts"
dat = CSV.read("Data/cardiacsurgery.csv", DataFrame)
dat.surgeon = categorical(dat.surgeon)

dat_ic = dat[dat.date .<=  730, :]

using MixedModels
mod = fit(MixedModel, @formula(status ~ Parsonnet + (1|surgeon)), dat_ic, Bernoulli())
println(mod)
# Generalized Linear Mixed Model fit by maximum likelihood (nAGQ = 1)
#   status ~ 1 + Parsonnet + (1 | surgeon)
#   Distribution: Bernoulli{Float64}
#   Link: LogitLink()

#    logLik    deviance     AIC       AICc        BIC    
#   -388.8235   777.6471   783.6471   783.6607   800.0816

# Variance components:
#            Column   Variance Std.Dev. 
# surgeon (Intercept)  0.037837 0.194518

#  Number of obs: 1769; levels of grouping factors: 6

# Fixed-effects parameters:
# ─────────────────────────────────────────────────────
#                   Coef.  Std. Error       z  Pr(>|z|)
# ─────────────────────────────────────────────────────
# (Intercept)  -3.65655    0.17509     -20.88    <1e-96
# Parsonnet     0.0818093  0.00723527   11.31    <1e-28
# ─────────────────────────────────────────────────────

invlogit(x) = exp(x)/(1+exp(x))

eta = predict(mod, dat_ic, type = :linpred)
invlogit(mean(eta))
# 0.05176958228186166
invlogit(mean(eta) + 0.75)
# 0.10360507481694466

dat_oc = dat[730 .< dat.date .<=  1095, :]

Random.seed!(239184367)
STAT = RiskAdjustedCUSUM(Δ = 0.75, model = mod, response = :status)

NOM = ARL(1000)
LIM = OneSidedFixedLimit(1.0, true)

PH2 = Phase2(Bootstrap(), dat_ic)

CH = ControlChart(STAT, LIM, NOM, PH2)
h = saCL!(CH, verbose = true, gamma = 0.05)
# Running SA ...
# Running adaptive gain ...
# Estimated gain D = 0.4117860274331409
# Running optimization ...
# i: 0/50000	h: 2.77856	hm: 0.0	stop: 0
# i: 1000/50000	h: 2.86375	hm: 2.9541	stop: 3457
# i: 2000/50000	h: 2.94813	hm: 2.95045	stop: 3435
# i: 3000/50000	h: 2.92253	hm: 2.95877	stop: 3420
# i: 3359/50000	Convergence!
println(h)
# (h = 2.9568877844997226, iter = 3359, status = "Convergence")


proc = apply_chart(CH, dat_oc)
plt = plot_series(proc, dpi = 300, label = "", xlab = L"t", ylab = L"C_t")
plt
savefig(plt, "Figures/example-risk-adjusted.png")
println("Saved image Figures/example-risk-adjusted.png")

#——————————————————————————————————————————————————————————
@info "Section 4.4: Detecting changes in profiles"

n = 500
nj = 20
x_grid = collect(0.5:0.5:10)
xs = Matrix{Float64}(undef, n, nj)
for i in 1:n
    xs[i, :] = x_grid
end

Random.seed!(41289355)
ys = sin.(xs) .+ randn(n, nj)

dat = FunctionalData(xs, ys)
PH2 = Phase2(Bootstrap(), dat)

using Loess
g = loess(vec(xs), vec(ys), span = 0.3)
plt = plot(minimum(xs):0.01:maximum(xs), (x) -> predict(g, x), linewidth = 1.75, label = L"\hat{f}(x)", xlab = L"x", ylab = L"y", dpi = 300)
scatter!(plt, vec(xs), vec(ys), markersize = 1, label = "", colour = "black")
plt
savefig(plt, "Figures/example-loess-ic.png")
println("Saved image Figures/example-loess-ic.png")

STAT = NEWMA(0.2, g, dat)

LIM = OneSidedBootstrapLimit(STAT, true, 1000)

NM = ARL(500)

CH = ControlChart(STAT, LIM, NM, PH2)


n2 = 150
xs_oc = xs[1:n2, :]
ys_ic = sin.(xs_oc) .+ randn(n2, nj)                  # In-control
ys_oc = sin.(xs_oc) + 2*cos.(xs_oc) .+ randn(n2, nj)  # Profile shift
ys_oc2 = sin.(xs_oc) .+ 2*randn(n2, nj)               # Variance shift

dat_ic = FunctionalData(xs_oc, ys_ic)
dat_oc = FunctionalData(xs_oc, ys_oc)
dat_oc2 = FunctionalData(xs_oc, ys_oc2)

proc_ic = apply_chart(CH, dat_ic)
proc_oc = apply_chart(CH, dat_oc)
proc_oc2 = apply_chart(CH, dat_oc2)

markersize = 1.5
plt1 = plot_series(proc_ic, marker = :d, title = "In-control", l_linestyle = :dot, markersize = markersize, dpi = 300, label = "", xlab = L"t", ylab = L"C_t")
plt2 = plot_series(proc_oc, marker = :o, title = "Profile shift",l_linestyle = :dot,  markersize = markersize, dpi = 300, colour = palette(:tab10)[2], label = "", xlab = L"t", ylab = L"C_t")
plt3 = plot_series(proc_oc2, marker = :square, title = "Variance shift", l_linestyle = :dot, markersize = markersize, dpi = 300, colour = palette(:tab10)[3], label = "", xlab = L"t", ylab = L"C_t")
l = @layout [grid(1,2); _{0.2w} c{0.55w} _{0.25w}]
plt = plot(plt1,plt2,plot(frame = :none), plt3; layout = l)
plt
savefig(plt, "Figures/example-loess-chart.png")
println("Saved image Figures/example-loess-chart.png")
