## Timing comparison between tramME and glmmTMB

library("glmmTMB")
library("tramME")
library("microbenchmark")
library("parallel")
seed <- 432
options(ncores = 8)

TMB::openmp(n = 1, DLL = "tramME")

## Simulate data with a similar structure as lme4::sleepstudy
## N: # of observations within groups
## K: # of groups
sim_lmer <- function(N = 10, K = 20, b = c(251, 10), sd = c(26, 25, 6)) {
  x <- runif(N*K, 0, 9)
  y <- drop(cbind(1, x) %*% b) + rep(rnorm(K, sd = sd[2]), each = N) +
    rep(rnorm(K, sd = sd[3]), each = N) * x + rnorm(N*K, sd = sd[1])
  data.frame(x = x, y = y, g = factor(rep(seq(K), each = N)))
}


Ns <- c(10, 20, 50, 100, 200, 500, 1000)

RNGkind("L'Ecuyer-CMRG")
set.seed(seed)
rt1 <- mclapply(Ns, function(n) {
  df <- sim_lmer(N = n, K = 20)
  microbenchmark(
    glmmTMB = {
      m1 <- glmmTMB(y ~ x + (x | g), data = df, REML = FALSE)
    },
    LmME = {
      m2 <- LmME(y ~ x + (x | g), data = df)
    },
    BoxCoxME = {
      ## BoxCoxME with default settings: the model will figure out the support
      ## and other inputs
      m3 <- BoxCoxME(y ~ x + (x | g), data = df)
    },
    times = 40)
}, mc.cores = getOption("ncores", round(detectCores() / 2 - 1L)))

Ks <- c(20, 100, 200, 500, 1000, 2000, 5000)

set.seed(seed)
rt2 <- mclapply(Ks, function(k) {
  df <- sim_lmer(N = 10, K = k)
  microbenchmark(
    glmmTMB = {
      m1 <- glmmTMB(y ~ x + (x | g), data = df, REML = FALSE)
    },
    LmME = {
      m2 <- LmME(y ~ x + (x | g), data = df)
    },
    BoxCoxME = {
      ## BoxCoxME with default settings: the model will figure out the support
      ## and other inputs
      m3 <- BoxCoxME(y ~ x + (x | g), data = df)
    },
    times = 40)
}, mc.cores = getOption("ncores", round(detectCores() / 2 - 1L)))

save(rt1, rt2, Ns, Ks, seed, file = "data/timings.rda")
