/*    Local Influence Diagnostics for the songbird dataset                                              */
/*    Copyright (C) 2025 Jhessica Leticia Kirch                                                         */
/*                                                                                                      */
/*    This program is free software: you can redistribute it and/or modify                              */ 
/*    it under the terms of the GNU General Public License as published by                              */ 
/*    the Free Software Foundation, either version 3 of the License, or                                 */ 
/*    (at your option) any later version.                                                               */ 
/*                                                                                                      */
/*    This program is distributed in the hope that it will be useful,                                   */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of                                    */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                                     */
/*    GNU General Public License for more details.                                                      */
/*                                                                                                      */
/*    You should have received a copy of the GNU General Public License                                 */
/*    along with this program.  If not, see <https://www.gnu.org/licenses/>.                            */
/*                                                                                                      */                                            

/*   Dataset available at https://ibiostat.be/online-resources as supplementary material for the book   */
/*   Molenberghs, G. and Verbeke, G. (2005) Models for Discrete Longitudinal Data. New York: Springer   */

data songbirddata;
	set MyLibRef.songbirddata;
run;

proc format;
	value Period
	1 = 'First'
	2 = 'Second';
run;

proc format;
	value Subject
	 1 = "Bird 1"
	 2 = "Bird 2"
	 3 = "Bird 3"
	 4 = "Bird 4"
	 5 = "Bird 5"
	 6 = "Bird 6"
	 7 = "Bird 7"
	 8 = "Bird 8"
	 9 = "Bird 9"
	10 = "Bird 10";
run;

data songbirddata;
	set songbirddata;
	Period = put(periode, Period.);
	Subject = put(vogel, Subject.);
	if time <= 0 then delete;
run;

*First period;
proc nlmixed data = songbirddata(where = (periode = 1)) HESS SUBGRADIENT = gradi;
	parms phi0 = .48 eta0 = 2.56 tau0 = 3.1 d11 = 0.02 d12 = 0 d22 = 0.24 d33 = 0.14 sigma2 = 0.01;
	num = (phi0 + f) * (time ** (eta0 + n));
	den = (tau0 + t) ** (eta0 + n) + time ** (eta0 + n);
	model SI_RA ~ normal(num/den, sigma2);
	random f t n ~ normal([0, 0, 0],[d11, d12, d22, 0, 0, d33]) subject = Subject;
	ods output Hessian = h;
	predict num/den out = pred1;
 run;

* Local influence measures;
proc iml;
	f = 3; *number of fixed parameters;
	r = 5; *number of random parameters;
	p = f + r; *number total of parameters;
	use h;
	read all var _NUM_ into L;
	L = -L[1:nrow(L), 2:ncol(L)];
	close h;
	use gradi;
	read all var _NUM_ into Delta;
	Delta = -Delta`;
	use gradi;
	read all var _CHAR_ into S[colname = Names];
	close gradi;
	Ci = vecdiag(2 * abs(Delta` * inv(L) * Delta));
	Cut_off_Ci = repeat(2 * (sum(Ci) / nrow(Ci)), nrow(Ci), 1);
	Lfixed = repeat(0, p, p);
	Lfixed[(f + 1):p, (f + 1):p] = inv(L[(f + 1):p, (f + 1):p]);
	Ci_fixed = vecdiag(2 * abs(Delta` * (inv(L) - Lfixed) * Delta));
	Cut_off_Ci_fixed = repeat(2 * (sum(Ci_fixed) / nrow(Ci_fixed)), nrow(Ci_fixed), 1);
	Lrandom = repeat(0, p, p);
	Lrandom[1:f, 1:f] = inv(L[1:f, 1:f]);
	Ci_random = vecdiag(2 * abs(Delta` * (inv(L) - Lrandom) * Delta));
	Cut_off_Ci_random = repeat(2 * (sum(Ci_random) / nrow(Ci_random)), nrow(Ci_random), 1);
	create Local var {"S" "Ci" "Cut_off_Ci" "Ci_fixed" "Cut_off_Ci_fixed" "Ci_random" "Cut_off_Ci_random"}; 
	append;      
	close Local; 
run;

data Local; 
   set Local;
   if Ci > Cut_off_Ci then Label = catx('', '#', S);
   else Label = ' ';
   if Ci_fixed > Cut_off_Ci_fixed or Ci_random > Cut_off_Ci_random then Label2 = catx('', '#', S);
   else Label2 = ' ';
run;

*Figure 4a;
ods html style = journal;
ods escapechar = "^";
proc sgplot data = Local  noautolegend ;
	scatter x = S y = Ci / datalabel = Label markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14); 
	refline 7.77270 / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
	yaxis label = "C^{unicode '1D62'x}" values = (0 to 20 by 2) labelattrs = (size = 14) valueattrs = (size = 14);
	xaxis label = "Birds" labelattrs = (size = 14) valueattrs = (size = 14);
run;
  
*Figura 4b;
proc sgplot data = Local noautolegend;
	scatter x = Ci_fixed y = Ci_random / datalabel = Label2 markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14);
	refline 1.15731 / axis = x lineattrs = (thickness = 3 color = black pattern = dash);
	refline 6.60388 / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
	xaxis label = "C^{unicode '1D62'x} on fixed effects parameters" labelattrs = (size = 14) valueattrs = (size = 14);
	yaxis values = (0 to 20 by 2) label = "C^{unicode '1D62'x} on variance components" labelattrs = (size = 14) valueattrs = (size = 14);
run;

*Second period;
proc nlmixed data = songbirddata(where = (periode = 2)) HESS SUBGRADIENT = gradi;
	parms phi0 = .64 eta0 = 1.88 eta1 = 0 tau0 = 3.68 d11 = .01 d22 = .01 sigma2 = 0.01;
	num = (phi0 + f) * (time ** (eta0 + eta1 * groep));
	den = (tau0 + t) ** (eta0 + eta1 * groep) + time ** (eta0 + eta1 * groep);
	model SI_RA ~ normal(num/den, sigma2);
	random f t ~ normal([0, 0],[d11, 0, d22]) subject = Subject;
	ods output Hessian = h;
	predict num/den out = pred2;
run;

* Local influence measures;
proc iml;
	f = 4; *number of fixed parameters;
	r = 3; *number of random parameters;
	p = f + r; *number total of parameters;
	use h;
	read all var _NUM_ into L;
	L = -L[1:nrow(L), 2:ncol(L)];
	close h;
	use gradi;
	read all var _NUM_ into Delta;
	Delta = -Delta`;
	use gradi;
	read all var _CHAR_ into S[colname = Names];
	close gradi;
	Ci = vecdiag(2 * abs(Delta` * inv(L) * Delta));
	Cut_off_Ci = repeat(2 * (sum(Ci) / nrow(Ci)), nrow(Ci), 1);
	Lfixed = repeat(0, p, p);
	Lfixed[(f + 1):p, (f + 1):p] = inv(L[(f + 1):p, (f + 1):p]);
	Ci_fixed = vecdiag(2 * abs(Delta` * (inv(L) - Lfixed) * Delta));
	Cut_off_Ci_fixed = repeat(2 * (sum(Ci_fixed) / nrow(Ci_fixed)), nrow(Ci_fixed), 1);
	Lrandom = repeat(0, p, p);
	Lrandom[1:f, 1:f] = inv(L[1:f, 1:f]);
	Ci_random = vecdiag(2 * abs(Delta` * (inv(L) - Lrandom) * Delta));
	Cut_off_Ci_random = repeat(2 * (sum(Ci_random) / nrow(Ci_random)), nrow(Ci_random), 1);
	create Local var {"S" "Ci" "Cut_off_Ci" "Ci_fixed" "Cut_off_Ci_fixed" "Ci_random" "Cut_off_Ci_random"}; 
	append;      
	close Local; 
run;

data Local; 
   set Local;
   if Ci > Cut_off_Ci then Label = catx('', '#', S);
   else Label = ' ';
   if Ci_fixed > Cut_off_Ci_fixed or Ci_random > Cut_off_Ci_random then Label2 = catx('', '#', S);
   else Label2 = ' ';
run;

*Figure 5a;
proc sgplot data = Local  noautolegend ;
  scatter x = S y = Ci / datalabel = Label markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14); 
  refline 6.75280 / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
  yaxis label = "C^{unicode '1D62'x}" values = (0 to 14 by 2) labelattrs = (size = 14) valueattrs = (size = 14);
  xaxis label = "Birds" labelattrs = (size = 14) valueattrs = (size = 14);
run;

*Figura 5b;
proc sgplot data = Local noautolegend;
	scatter x = Ci_fixed y = Ci_random / datalabel = Label2 markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14);
	refline 1.84959  / axis = x lineattrs = (thickness = 3 color = black pattern = dash);
	refline 4.88393  / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
	xaxis label = "C^{unicode '1D62'x} on fixed effects parameters" labelattrs = (size = 14) valueattrs = (size = 14);
	yaxis value = (0 to 14 by 2) label = "C^{unicode '1D62'x} on variance components" labelattrs = (size = 14) valueattrs = (size = 14);
run;

data pred;
	set pred1
	pred2;
run;

*Figure 3;
proc sgpanel data = pred;
	panelby Subject / onepanel novarname columns = 5 sort = data;
	rowaxis  label = "SI(RA)";
	colaxis label = "Time";
	styleattrs datasymbols = (circlefilled square);
	scatter x = time y = SI_RA/group = Period markerattrs = (size = 10 color = black); 
	series x = time y = pred/group = Period lineattrs = (color = black thickness = 2);
run;


* Two compartment model;

data songbirddata;
	set songbirddata;
	SI_RA2 = SI_RA * 100;
run;

* First period;
proc nlmixed data = songbirddata(where = (periode = 1)) HESS SUBGRADIENT = gradi;
	parms beta0 = 3.17 tau0 = 1.5 gamma0 = -1.1 d11 = 0.001  d33 = 0 d44 = 0 sigma2 = 1;
	k1 = beta0 + b;
	k3 = tau0 + t;
	k4 = gamma0 +  g;
	predmean = exp(k1) - exp(k3) * exp(-exp(k4) * time);
	model SI_RA2 ~ normal(predmean, sigma2);
	random b t g ~ normal([0, 0, 0],[d11, 0, d33, 0, 0, d44]) subject = Subject;
	ods output Hessian = h;
	predict predmean out = pred1;
run;

* Local influence measures;
proc iml;
	f = 3; *number of fixed parameters;
	r = 4; *number of random parameters;
	p = f + r; *number total of parameters;
	use h;
	read all var _NUM_ into L;
	L = -L[1:nrow(L), 2:ncol(L)];
	close h;
	use gradi;
	read all var _NUM_ into Delta;
	Delta = -Delta`;
	use gradi;
	read all var _CHAR_ into S[colname = Names];
	close gradi;
	Ci = vecdiag(2 * abs(Delta` * inv(L) * Delta));
	Cut_off_Ci = repeat(2 * (sum(Ci) / nrow(Ci)), nrow(Ci), 1);
	Lfixed = repeat(0, p, p);
	Lfixed[(f + 1):p, (f + 1):p] = inv(L[(f + 1):p, (f + 1):p]);
	Ci_fixed = vecdiag(2 * abs(Delta` * (inv(L) - Lfixed) * Delta));
	Cut_off_Ci_fixed = repeat(2 * (sum(Ci_fixed) / nrow(Ci_fixed)), nrow(Ci_fixed), 1);
	Lrandom = repeat(0, p, p);
	Lrandom[1:f, 1:f] = inv(L[1:f, 1:f]);
	Ci_random = vecdiag(2 * abs(Delta` * (inv(L) - Lrandom) * Delta));
	Cut_off_Ci_random = repeat(2 * (sum(Ci_random) / nrow(Ci_random)), nrow(Ci_random), 1);
	create Local var {"S" "Ci" "Cut_off_Ci" "Ci_fixed" "Cut_off_Ci_fixed" "Ci_random" "Cut_off_Ci_random"}; 
	append;      
	close Local; 
run;

data Local; 
   set Local;
   if Ci > Cut_off_Ci then Label = catx('', '#', S);
   else Label = ' ';
   if Ci_fixed > Cut_off_Ci_fixed or Ci_random > Cut_off_Ci_random then Label2 = catx('', '#', S);
   else Label2 = ' ';
run;

*Figure 7a;
proc sgplot data = Local  noautolegend ;
  scatter x = S y = Ci / datalabel = Label markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14); 
  refline 5.85424 / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
  yaxis label = "C^{unicode '1D62'x}" values = (0 to 20 by 2) labelattrs = (size = 14) valueattrs = (size = 14);
  xaxis label = "Birds" labelattrs = (size = 14) valueattrs = (size = 14);
run;

*Figura 7b;
proc sgplot data = Local noautolegend;
	scatter x = Ci_fixed y = Ci_random / datalabel = Label2 markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14);
	refline 1.40355  / axis = x lineattrs = (thickness = 3 color = black pattern = dash);
	refline 4.39994  / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
	xaxis label = "C^{unicode '1D62'x} on fixed effects parameters" labelattrs = (size = 14) valueattrs = (size = 14);
	yaxis value = (0 to 15 by 3) label = "C^{unicode '1D62'x} on variance components" labelattrs = (size = 14) valueattrs = (size = 14);
run;

*Second period;
proc nlmixed data=songbirddata(where = (periode = 2)) HESS SUBGRADIENT = gradi;
	parms beta0 = 4.2 phi0 = -0.0001 tau0 = 4.4 gamma0 = .25 d11 = 0.001 d33 = 0 d44 = 0 sigma2 = 2;
	k1 = beta0 + b;
	k2 = phi0;
	k3 = tau0 +  t;
	k4 = gamma0 + g;
	predmean = exp(k1) * exp(-exp(k2) * time) - exp(k3) * exp(-exp(k4) * time);
	model SI_RA2 ~ normal(predmean, sigma2);
	random b t g ~ normal([0, 0, 0],[d11, 0, d33, 0, 0, d44]) subject = Subject;
	ods output Hessian = h;
	predict predmean out = pred2;
run;

* Local influence measures;
proc iml;
	f = 4; *number of fixed parameters;
	r = 4; *number of random parameters;
	p = f + r; *number total of parameters;
	use h;
	read all var _NUM_ into L;
	L = -L[1:nrow(L), 2:ncol(L)];
	close h;
	use gradi;
	read all var _NUM_ into Delta;
	Delta = -Delta`;
	use gradi;
	read all var _CHAR_ into S[colname = Names];
	close gradi;
	Ci = vecdiag(2 * abs(Delta` * inv(L) * Delta));
	Cut_off_Ci = repeat(2 * (sum(Ci) / nrow(Ci)), nrow(Ci), 1);
	Lfixed = repeat(0, p, p);
	Lfixed[(f + 1):p, (f + 1):p] = inv(L[(f + 1):p, (f + 1):p]);
	Ci_fixed = vecdiag(2 * abs(Delta` * (inv(L) - Lfixed) * Delta));
	Cut_off_Ci_fixed = repeat(2 * (sum(Ci_fixed) / nrow(Ci_fixed)), nrow(Ci_fixed), 1);
	Lrandom = repeat(0, p, p);
	Lrandom[1:f, 1:f] = inv(L[1:f, 1:f]);
	Ci_random = vecdiag(2 * abs(Delta` * (inv(L) - Lrandom) * Delta));
	Cut_off_Ci_random = repeat(2 * (sum(Ci_random) / nrow(Ci_random)), nrow(Ci_random), 1);
	create Local var {"S" "Ci" "Cut_off_Ci" "Ci_fixed" "Cut_off_Ci_fixed" "Ci_random" "Cut_off_Ci_random"}; 
	append;      
	close Local; 
run;

data Local; 
   set Local;
   if Ci > Cut_off_Ci then Label = catx('', '#', S);
   else Label = ' ';
   if Ci_fixed > Cut_off_Ci_fixed or Ci_random > Cut_off_Ci_random then Label2 = catx('', '#', S);
   else Label2 = ' ';
run;

*Figure 7c;
proc sgplot data = Local  noautolegend ;
  scatter x = S y = Ci / datalabel = Label markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14); 
  refline 4.95411 / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
  yaxis label = "C^{unicode '1D62'x}" values = (0 to 7 by 1) labelattrs = (size = 14) valueattrs = (size = 14);
  xaxis label = "Birds" labelattrs = (size = 14) valueattrs = (size = 14);
run;

*Figura 7d;
proc sgplot data = Local noautolegend;
	scatter x = Ci_fixed y = Ci_random / datalabel = Label2 markerattrs = (size = 10 color = black symbol = CircleFilled) datalabelattrs = (size = 14);
	refline 1.98955  / axis = x lineattrs = (thickness = 3 color = black pattern = dash);
	refline 3.16068  / axis = y lineattrs = (thickness = 3 color = black pattern = dash);
	xaxis label = "C^{unicode '1D62'x} on fixed effects parameters" labelattrs = (size = 14) valueattrs = (size = 14);
	yaxis value = (0 to 5 by 1) label = "C^{unicode '1D62'x} on variance components" labelattrs = (size = 14) valueattrs = (size = 14);
run;

data pred;
	set pred1
	pred2;
run;

*Figure 6;
proc sgpanel data = pred;
	panelby Subject / onepanel novarname columns = 5 sort = data;
	rowaxis  label = "SI(RA)";
	colaxis label = "Time";
	styleattrs datasymbols = (circlefilled square);
	scatter x = time y = SI_RA2/group = Period markerattrs = (size = 10 color = black); 
	series x = time y = pred/group = Period lineattrs = (color = black thickness = 2);
run;
